<?php
// Register Group Post Type
function rohi_register_group_post_type() {
    $labels = array(
        'name'                  => 'Groups',
        'singular_name'         => 'Group',
        'menu_name'             => 'Groups',
        'name_admin_bar'        => 'Group',
        'add_new'               => 'Add New',
        'add_new_item'          => 'Add New Group',
        'new_item'             => 'New Group',
        'edit_item'             => 'Edit Group',
        'view_item'             => 'View Group',
        'all_items'             => 'All Groups',
        'search_items'          => 'Search Groups',
        'parent_item_colon'     => 'Parent Groups:',
        'not_found'             => 'No groups found.',
        'not_found_in_trash'    => 'No groups found in Trash.'
    );

    $args = array(
        'labels'                => $labels,
        'public'              => true,
        'publicly_queryable'  => true,
        'show_ui'             => true,
        'show_in_menu'        => true,
        'query_var'           => true,
        'rewrite'             => array('slug' => 'groups'),
        'capability_type'     => 'post',
        'has_archive'        => true,
        'hierarchical'       => false,
        'menu_position'     => 22,
        'menu_icon'         => 'dashicons-groups',
        'supports'          => array('title', 'editor', 'thumbnail', 'excerpt'),
        'show_in_rest'       => true
    );

    register_post_type('group', $args);
}
add_action('init', 'rohi_register_group_post_type');

// Add custom fields to group post type
function rohi_add_group_meta_boxes() {
    add_meta_box(
        'rohi_group_details',
        'Group Details',
        'rohi_group_details_callback',
        'group',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes', 'rohi_add_group_meta_boxes');

function rohi_group_details_callback($post) {
    wp_nonce_field('rohi_group_details_nonce', 'rohi_group_details_nonce');
    
    $location = get_post_meta($post->ID, '_rohi_group_location', true);
    $contact = get_post_meta($post->ID, '_rohi_group_contact', true);
    $meeting_times = get_post_meta($post->ID, '_rohi_group_meeting_times', true);
    
    ?>
    <div class="rohi-meta-field">
        <label for="rohi_group_location">Location:</label>
        <input type="text" id="rohi_group_location" name="rohi_group_location" value="<?php echo esc_attr($location); ?>" style="width: 100%;">
    </div>
    
    <div class="rohi-meta-field">
        <label for="rohi_group_contact">Contact Information:</label>
        <input type="text" id="rohi_group_contact" name="rohi_group_contact" value="<?php echo esc_attr($contact); ?>" style="width: 100%;">
    </div>
    
    <div class="rohi-meta-field">
        <label for="rohi_group_meeting_times">Meeting Times:</label>
        <input type="text" id="rohi_group_meeting_times" name="rohi_group_meeting_times" value="<?php echo esc_attr($meeting_times); ?>" style="width: 100%;" placeholder="e.g., Every Tuesday at 7 PM">
    </div>
    
    <style>
        .rohi-meta-field {
            margin-bottom: 15px;
        }
        .rohi-meta-field label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
    </style>
    <?php
}

function rohi_save_group_meta($post_id) {
    if (!isset($_POST['rohi_group_details_nonce']) || 
        !wp_verify_nonce($_POST['rohi_group_details_nonce'], 'rohi_group_details_nonce')) {
        return;
    }
    
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    
    if (!current_user_can('edit_post', $post_id)) return;
    
    $fields = array(
        '_rohi_group_location',
        '_rohi_group_contact',
        '_rohi_group_meeting_times'
    );
    
    foreach ($fields as $field) {
        if (isset($_POST[str_replace('_rohi_', 'rohi_', $field)])) {
            update_post_meta(
                $post_id,
                $field,
                sanitize_text_field($_POST[str_replace('_rohi_', 'rohi_', $field)])
            );
        }
    }
}
add_action('save_post', 'rohi_save_group_meta');

// Group listing shortcode
function rohi_group_listing_shortcode($atts) {
    $atts = shortcode_atts(array(
        'posts_per_page' => -1,
        'orderby' => 'date',
        'order' => 'DESC'
    ), $atts);
    
    $args = array(
        'post_type' => 'group',
        'posts_per_page' => $atts['posts_per_page'],
        'orderby' => $atts['orderby'],
        'order' => $atts['order'],
        'post_status' => 'publish'
    );
    
    $groups = new WP_Query($args);
    
    ob_start();
    
    if ($groups->have_posts()) :
        ?>
        <div class="rohi-group-container">
            <div class="rohi-group-grid">
                <?php while ($groups->have_posts()) : $groups->the_post(); 
                    $location = get_post_meta(get_the_ID(), '_rohi_group_location', true);
                    $contact = get_post_meta(get_the_ID(), '_rohi_group_contact', true);
                ?>
                    <article class="rohi-group-card">
                        <a href="<?php the_permalink(); ?>" class="rohi-group-card-link">
                            <?php if (has_post_thumbnail()) : ?>
                                <div class="rohi-group-thumbnail">
                                    <?php the_post_thumbnail('large'); ?>
                                    <div class="rohi-thumbnail-overlay"></div>
                                </div>
                            <?php endif; ?>
                            <div class="rohi-card-info">
                                <h3><?php the_title(); ?></h3>
                                <div class="rohi-group-meta">
                                    <?php if ($location) : ?>
                                        <span class="rohi-meta-item"><i class="fas fa-map-marker-alt"></i> <?php echo esc_html($location); ?></span>
                                    <?php endif; ?>
                                    <?php if ($contact) : ?>
                                        <span class="rohi-meta-item"><i class="fas fa-phone"></i> <?php echo esc_html($contact); ?></span>
                                    <?php endif; ?>
                                </div>
                                <?php if (has_excerpt()) : ?>
                                    <div class="rohi-group-excerpt"><?php the_excerpt(); ?></div>
                                <?php endif; ?>
                                <span class="rohi-view-details">View Details &rarr;</span>
                            </div>
                        </a>
                        
                         
                           
                        </div>
                    </article>
                    
                    <!-- How to Join Section -->
                            <div class="rohi-how-to-join">
                                <h2>How to Join</h2>
                                <div class="rohi-join-steps">
                                    <div class="rohi-step">
                                        <div class="rohi-step-number">1</div>
                                        <h3>Find a Group</h3>
                                        <p>Browse our groups above and find one that matches your interests and location.</p>
                                    </div>
                                    <div class="rohi-step">
                                        <div class="rohi-step-number">2</div>
                                        <h3>Get in Touch</h3>
                                        <p>Use the Provided Info above or fill the form</p>
                                    </div>
                                    <div class="rohi-step">
                                        <div class="rohi-step-number">3</div>
                                        <h3>Attend a Meeting</h3>
                                        <p>Join the group for a meeting or event to see if it's the right fit for you.</p>
                                    </div>
                                    <div class="rohi-step">
                                        <div class="rohi-step-number">4</div>
                                        <h3>Become a Member</h3>
                                        <p>Once you've found the right group, complete the membership process.</p>
                                    </div>
                                </div>
                <?php endwhile; ?>
            </div>
        </div>

        <?php
    else :
        echo '<p class="rohi-no-groups">No groups are currently available. Please check back later.</p>';
    endif;
    
    wp_reset_postdata();
    
    return ob_get_clean();
}
add_shortcode('groups_listings', 'rohi_group_listing_shortcode');

// Single group template handler
function rohi_single_group_template($template) {
    global $post;
    
    if ($post->post_type == 'group' && is_single()) {
        // Check if theme has a template
        $theme_template = locate_template(array('single-group.php'));
        if (!empty($theme_template)) {
            return $theme_template;
        }
        
        // Fallback to default template if no custom template exists
        $plugin_template = plugin_dir_path(__FILE__) . '/single-group.php';
        if (file_exists($plugin_template)) {
            return $plugin_template;
        }
    }
    
    return $template;
}
add_filter('single_template', 'rohi_single_group_template');

// Create template file programmatically if it doesn't exist
function rohi_create_single_group_template() {
    $template_dir = get_stylesheet_directory() . '/';
    $template_file = $template_dir . 'single-group.php';
    
    // Don't overwrite if template already exists in theme
    if (file_exists($template_file)) {
        return;
    }
    
    // Copy template to theme directory
    $plugin_template = plugin_dir_path(__FILE__) . 'templates/single-group.php';
    if (file_exists($plugin_template)) {
        copy($plugin_template, $template_file);
    }
}
add_action('after_switch_theme', 'rohi_create_single_group_template');



// Handle group join request submission
function rohi_handle_group_join_request() {
    // Verify nonce for security
    if (!isset($_POST['rohi_group_join_nonce']) || 
        !wp_verify_nonce($_POST['rohi_group_join_nonce'], 'rohi_group_details_nonce')) {
        wp_send_json_error('Security check failed.');
        return;
    }

    // Initialize response array
    $response = array(
        'success' => false,
        'message' => '',
        'data' => array()
    );

    try {
        // Check if all required fields are present
        $required = array('group_id', 'name', 'location', 'phone');
        foreach ($required as $field) {
            if (empty($_POST[$field])) {
                throw new Exception("Please fill in all required fields.");
            }
        }

        // Sanitize input
        $group_id = intval($_POST['group_id']);
        $name = sanitize_text_field($_POST['name']);
        $location = sanitize_text_field($_POST['location']);
        $phone = sanitize_text_field($_POST['phone']);
        $group_title = sanitize_text_field($_POST['group_title']);

        // Save to database
        global $wpdb;
        $table_name = $wpdb->prefix . 'rohi_group_requests';
        
        $result = $wpdb->insert(
            $table_name,
            array(
                'group_id' => $group_id,
                'group_title' => $group_title,
                'name' => $name,
                'location' => $location,
                'phone' => $phone,
                'status' => 'pending'
            ),
            array(
                '%d', '%s', '%s', '%s', '%s', '%s'
            )
        );

        if (!$result) {
            throw new Exception('Failed to save request to database.');
        }

        // Send email notification
        $to = get_option('admin_email');
        $subject = 'New Group Join Request: ' . $name . ' for ' . $group_title;
        $email_message = "A new group join request has been submitted:\n\n";
        $email_message .= "Name: $name\n";
        $email_message .= "Location: $location\n";
        $email_message .= "Phone: $phone\n";
        $email_message .= "Group: $group_title\n";
        
        if (!empty($message)) {
            $email_message .= "Message: $message\n";
        }
        
        $email_message .= "\nView all requests in WordPress admin: " . admin_url('admin.php?page=rohi_group_requests');
        
        wp_mail($to, $subject, $email_message);

        // Success response
        $response['success'] = true;
        $response['message'] = 'Thank you! Your request has been submitted successfully. We will contact you shortly.';
        
    } catch (Exception $e) {
        $response['message'] = $e->getMessage();
    }

    wp_send_json($response);
}
add_action('wp_ajax_rohi_submit_group_join_request', 'rohi_handle_group_join_request');
add_action('wp_ajax_nopriv_rohi_submit_group_join_request', 'rohi_handle_group_join_request');

// Add admin menu for group requests
function rohi_group_requests_admin_menu() {
    add_submenu_page(
        'edit.php?post_type=group',
        'Group Join Requests',
        'Join Requests',
        'manage_options',
        'rohi_group_requests',
        'rohi_display_group_requests'
    );
}
add_action('admin_menu', 'rohi_group_requests_admin_menu');

// Display group requests in admin
function rohi_display_group_requests() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'rohi_group_requests';
    
    // Handle status updates
    if (isset($_GET['action']) && $_GET['action'] == 'update_status' && isset($_GET['id']) && isset($_GET['status'])) {
        $id = intval($_GET['id']);
        $status = sanitize_text_field($_GET['status']);
        
        $wpdb->update(
            $table_name,
            array('status' => $status),
            array('id' => $id),
            array('%s'),
            array('%d')
        );
        
        echo '<div class="notice notice-success"><p>Request status updated.</p></div>';
    }
    
    // Handle request deletion
    if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
        $id = intval($_GET['id']);
        
        // Delete from database
        $wpdb->delete(
            $table_name,
            array('id' => $id),
            array('%d')
        );
        
        echo '<div class="notice notice-success"><p>Request deleted.</p></div>';
    }
    
    // Get all requests
    $requests = $wpdb->get_results("SELECT * FROM $table_name ORDER BY request_date DESC");
    
    ?>
    <div class="wrap">
        <h1 class="wp-heading-inline">Group Join Requests</h1>
        <hr class="wp-header-end">
        
        <div class="rohi-requests-admin">
            <div class="rohi-requests-table-container">
                <table class="wp-list-table widefat fixed striped table-view-list">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Location</th>
                            <th>Group</th>
                            <th>Date</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($requests)) : ?>
                            <tr>
                                <td colspan="8" class="rohi-no-requests">No requests found.</td>
                            </tr>
                        <?php else : ?>
                            <?php foreach ($requests as $req) : ?>
                                <tr data-request-id="<?php echo $req->id; ?>">
                                    <td><?php echo $req->id; ?></td>
                                    <td>
                                        <strong><?php echo esc_html($req->name); ?></strong>
                                        <div class="row-actions">
                                            <span class="phone"><?php echo esc_html($req->phone); ?></span>
                                        </div>
                                    </td>
                                    <td><?php echo esc_html($req->location); ?></td>
                                    <td>
                                        <?php if ($req->group_id) : ?>
                                            <a href="<?php echo get_edit_post_link($req->group_id); ?>">
                                                <?php echo esc_html($req->group_title); ?>
                                            </a>
                                        <?php else : ?>
                                            <?php echo esc_html($req->group_title); ?>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('M j, Y', strtotime($req->request_date)); ?></td>
                                    <td>
                                        <span class="rohi-status-badge status-<?php echo esc_attr($req->status); ?>">
                                            <?php echo ucfirst($req->status); ?>
                                        </span>
                                    </td>
                                    <td class="rohi-actions">
                                        <!-- Mark Contacted -->
                                        <a href="<?php echo admin_url('edit.php?post_type=group&page=rohi_group_requests&action=update_status&id=' . $req->id . '&status=contacted'); ?>" 
                                           class="button" 
                                           title="Mark Contacted">
                                           <span class="dashicons dashicons-phone"></span>
                                        </a>
                                    
                                        <!-- Approve -->
                                        <a href="<?php echo admin_url('edit.php?post_type=group&page=rohi_group_requests&action=update_status&id=' . $req->id . '&status=approved'); ?>" 
                                           class="button button-primary" 
                                           title="Approve">
                                           <span class="dashicons dashicons-yes-alt"></span>
                                        </a>
                                    
                                        <!-- Delete -->
                                        <a href="<?php echo admin_url('edit.php?post_type=group&page=rohi_group_requests&action=delete&id=' . $req->id); ?>" 
                                           class="button button-danger" 
                                           onclick="return confirm('Are you sure you want to delete this request?')" 
                                           title="Delete">
                                           <span class="dashicons dashicons-trash"></span>
                                        </a>
                                    </td>
                                    </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php
}