<?php
function activity_cards_shortcode($atts) {
    ob_start();

    // Default attributes
    $atts = shortcode_atts(array(
        'layout' => 'grid', // 'grid' or 'list'
        'columns' => 3,
        'show_excerpt' => 'false',
        'show_button' => 'false'
    ), $atts);

    // Query: only 3 posts, random order
    $query = new WP_Query(array(
        'post_type'      => 'page',
        'posts_per_page' => 3,
        'orderby'        => 'rand',
        'post__not_in'   => array(get_the_ID()),
        'tax_query'      => array(
            array(
                'taxonomy' => 'page_category',
                'field'    => 'slug',
                'terms'    => 'activities',
            ),
        ),
    ));

    // Enqueue the CSS separately
    wp_enqueue_style('activity-cards-style', get_stylesheet_directory_uri() . '/activity-cards.css', array(), '1.0.0');

    if ($query->have_posts()) {
        echo '<div class="activity-cards-container ' . ($atts['layout'] === 'list' ? 'list-layout' : '') . '">';
        while ($query->have_posts()) {
            $query->the_post();
            $title = get_the_title();
            $permalink = get_permalink();
            $thumbnail = get_the_post_thumbnail_url(get_the_ID(), 'large');

            echo '<div class="activity-card" data-aos="fade-up">';
            echo '<div class="activity-card-image">';
            if ($thumbnail) {
                echo '<img src="' . esc_url($thumbnail) . '" alt="' . esc_attr($title) . '">';
            } else {
                echo '<div style="background: var(--kpy-light); height: 100%; display: flex; align-items: center; justify-content: center; color: var(--kpy-primary);"><span>No Image</span></div>';
            }
            echo '<div class="activity-card-badge">Program</div>';
            echo '</div>';
            echo '<div class="activity-card-content">';
            echo '<h3 class="activity-card-title"><a href="' . esc_url($permalink) . '">' . esc_html($title) . '</a></h3>';
            if ($atts['show_excerpt'] === 'true' && !empty($excerpt)) {
                echo '<div class="activity-card-excerpt">' . esc_html($excerpt) . '</div>';
            }
            if ($atts['show_button'] === 'true') {
                echo '<a href="' . esc_url($permalink) . '" class="activity-card-button">Learn More</a>';
            }
            echo '</div>';
            echo '</div>';
        }
        echo '</div>';
    } else {
        echo '<p style="text-align: center; padding: 40px; color: var(--kpy-primary);">No activities found.</p>';
    }

    wp_reset_postdata();
    return ob_get_clean();
}
add_shortcode('activity_cards', 'activity_cards_shortcode');


function Program_cards_shortcode($atts) {
    ob_start();

    // Default attributes
    $atts = shortcode_atts(array(
        'layout' => 'grid', // 'grid' or 'list'
        'columns' => 3,
        'show_excerpt' => 'false',
        'show_button' => 'false'
    ), $atts);

    // Query: only 3 posts, random order
    $query = new WP_Query(array(
        'post_type'      => 'page',
        'posts_per_page' => 6,
        'orderby'        => '',
        'post__not_in'   => array(get_the_ID()),
        'tax_query'      => array(
            array(
                'taxonomy' => 'page_category',
                'field'    => 'slug',
                'terms'    => 'activities',
            ),
        ),
    ));

    // Enqueue the CSS separately
    wp_enqueue_style('program-cards-style', get_stylesheet_directory_uri() . '/program-cards.css', array(), '1.0.0');

    if ($query->have_posts()) {
        echo '<div class="program-cards-container layout-' . esc_attr($atts['layout']) . ' columns-' . esc_attr($atts['columns']) . '">';
        while ($query->have_posts()) {
            $query->the_post();
            $title = get_the_title();
            $permalink = get_permalink();
            $thumbnail = get_the_post_thumbnail_url(get_the_ID(), 'large');
            $excerpt = wp_trim_words(get_the_excerpt(), 16, '...');

            echo '<article class="program-card">';
            echo '<div class="program-card__media">';
            if ($thumbnail) {
                echo '<img src="' . esc_url($thumbnail) . '" alt="' . esc_attr($title) . '" class="program-card__image">';
            } else {
                echo '<div class="program-card__placeholder"><span>No Image</span></div>';
            }
            echo '<div class="activity-card-badge">Program</div>';
            echo '</div>';
            echo '<div class="program-card__content">';
            echo '<h3 class="program-card__title"><a href="' . esc_url($permalink) . '">' . esc_html($title) . '</a></h3>';
            if ($atts['show_excerpt'] === 'true' && !empty($excerpt)) {
                echo '<div class="program-card__excerpt">' . esc_html($excerpt) . '</div>';
            }
            if ($atts['show_button'] === 'true') {
                echo '<a href="' . esc_url($permalink) . '" class="program-card__button">Learn More</a>';
            }
            echo '</div>';
            echo '</article>';
        }
        echo '</div>';
    } else {
        echo '<p class="program-cards__not-found">No activities found.</p>';
    }

    wp_reset_postdata();
    return ob_get_clean();
}
add_shortcode('Programs', 'Program_cards_shortcode');




// Register Custom Post Type for Coaching Programs
function kavod_register_coaching_programs_cpt() {
    $labels = array(
        'name'                  => _x('Coaching', 'Post Type General Name', 'kavod'),
        'singular_name'         => _x('Coaching', 'Post Type Singular Name', 'kavod'),
        'menu_name'             => __('Coaching Programs', 'kavod'),
        'name_admin_bar'        => __('Coaching Program', 'kavod'),
        'archives'              => __('Program Archives', 'kavod'),
        'attributes'            => __('Program Attributes', 'kavod'),
        'parent_item_colon'     => __('Parent Program:', 'kavod'),
        'all_items'             => __('All Programs', 'kavod'),
        'add_new_item'          => __('Add New Program', 'kavod'),
        'add_new'               => __('Add New', 'kavod'),
        'new_item'              => __('New Program', 'kavod'),
        'edit_item'             => __('Edit Program', 'kavod'),
        'update_item'           => __('Update Program', 'kavod'),
        'view_item'             => __('View Program', 'kavod'),
        'view_items'            => __('View Programs', 'kavod'),
        'search_items'          => __('Search Program', 'kavod'),
        'not_found'             => __('Not found', 'kavod'),
        'not_found_in_trash'    => __('Not found in Trash', 'kavod'),
        'featured_image'        => __('Program Image', 'kavod'),
        'set_featured_image'    => __('Set program image', 'kavod'),
        'remove_featured_image' => __('Remove program image', 'kavod'),
        'use_featured_image'    => __('Use as program image', 'kavod'),
        'insert_into_item'      => __('Insert into program', 'kavod'),
        'uploaded_to_this_item' => __('Uploaded to this program', 'kavod'),
        'items_list'            => __('Programs list', 'kavod'),
        'items_list_navigation' => __('Programs list navigation', 'kavod'),
        'filter_items_list'     => __('Filter programs list', 'kavod'),
    );
    $args = array(
        'label'                 => __('Coaching', 'kavod'),
        'description'           => __('Identity Coaching Programs', 'kavod'),
        'labels'                => $labels,
        'supports'              => array('title', 'editor', 'thumbnail', 'excerpt', 'custom-fields'),
        'taxonomies'            => array('program_category'),
        'hierarchical'          => false,
        'public'                => true,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'menu_position'         => 5,
        'menu_icon'             => 'dashicons-businessperson',
        'show_in_admin_bar'     => true,
        'show_in_nav_menus'     => true,
        'can_export'            => true,
        'has_archive'           => true,
        'exclude_from_search'   => false,
        'publicly_queryable'    => true,
        'capability_type'       => 'page',
    );
    register_post_type('coaching_program', $args);
}
add_action('init', 'kavod_register_coaching_programs_cpt', 0);

// Add meta boxes for program details
function kavod_add_program_meta_boxes() {
    add_meta_box(
        'kavod_program_details',
        'Program Details',
        'kavod_program_details_callback',
        'coaching_program',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes', 'kavod_add_program_meta_boxes');

function kavod_program_details_callback($post) {
    wp_nonce_field('kavod_program_details_nonce', 'kavod_program_details_nonce');
    
    $duration = get_post_meta($post->ID, '_program_duration', true);
    $features = get_post_meta($post->ID, '_program_features', true);
    $button_text = get_post_meta($post->ID, '_program_button_text', true);
    $button_url = get_post_meta($post->ID, '_program_button_url', true);
    $icon_class = get_post_meta($post->ID, '_program_icon_class', true);
    
    echo '<p><label for="program_duration">Duration: </label>';
    echo '<input type="text" id="program_duration" name="program_duration" value="' . esc_attr($duration) . '" size="25" /></p>';
    
    echo '<p><label for="program_features">Features (one per line): </label>';
    echo '<textarea id="program_features" name="program_features" rows="5" cols="50">' . esc_textarea($features) . '</textarea></p>';
    
    echo '<p><label for="program_icon_class">Font Awesome Icon Class: </label>';
    echo '<input type="text" id="program_icon_class" name="program_icon_class" value="' . esc_attr($icon_class) . '" size="25" placeholder="fas fa-user-check" /></p>';
    
    echo '<p><label for="program_button_text">Button Text: </label>';
    echo '<input type="text" id="program_button_text" name="program_button_text" value="' . esc_attr($button_text) . '" size="25" /></p>';
    
    echo '<p><label for="program_button_url">Button URL: </label>';
    echo '<input type="text" id="program_button_url" name="program_button_url" value="' . esc_attr($button_url) . '" size="25" /></p>';
}

function kavod_save_program_details($post_id) {
    if (!isset($_POST['kavod_program_details_nonce']) || 
        !wp_verify_nonce($_POST['kavod_program_details_nonce'], 'kavod_program_details_nonce')) {
        return;
    }
    
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }
    
    if (isset($_POST['program_duration'])) {
        update_post_meta($post_id, '_program_duration', sanitize_text_field($_POST['program_duration']));
    }
    
    if (isset($_POST['program_features'])) {
        update_post_meta($post_id, '_program_features', sanitize_textarea_field($_POST['program_features']));
    }
    
    if (isset($_POST['program_icon_class'])) {
        update_post_meta($post_id, '_program_icon_class', sanitize_text_field($_POST['program_icon_class']));
    }
    
    if (isset($_POST['program_button_text'])) {
        update_post_meta($post_id, '_program_button_text', sanitize_text_field($_POST['program_button_text']));
    }
    
    if (isset($_POST['program_button_url'])) {
        update_post_meta($post_id, '_program_button_url', sanitize_text_field($_POST['program_button_url']));
    }
}
add_action('save_post', 'kavod_save_program_details');

// Shortcode to display coaching programs
function kavod_coaching_programs_shortcode($atts) {
    $atts = shortcode_atts(array(
        'category' => '',
        'limit' => -1
    ), $atts, 'coaching_programs');
    
    $args = array(
        'post_type' => 'coaching_program',
        'posts_per_page' => $atts['limit'],
        'post_status' => 'publish'
    );
    
    if (!empty($atts['category'])) {
        $args['tax_query'] = array(
            array(
                'taxonomy' => 'program_category',
                'field' => 'slug',
                'terms' => $atts['category']
            )
        );
    }
    
    $programs = new WP_Query($args);
    
    if (!$programs->have_posts()) {
        return '<p>No coaching programs found.</p>';
    }
    
    // Enqueue necessary scripts and styles
    wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css');
    wp_enqueue_style('montserrat-font', 'https://fonts.googleapis.com/css2?family=Montserrat:wght@400;500;600;700&display=swap');
    
    ob_start();
    ?>
    
    
    <div class="kavod-programs-wrapper">
        <div class="kavod-programs-container">
            <?php while ($programs->have_posts()) : $programs->the_post(); 
                $duration = get_post_meta(get_the_ID(), '_program_duration', true);
                $features = get_post_meta(get_the_ID(), '_program_features', true);
                $button_text = get_post_meta(get_the_ID(), '_program_button_text', true);
                $button_url = get_post_meta(get_the_ID(), '_program_button_url', true);
                $icon_class = get_post_meta(get_the_ID(), '_program_icon_class', true);
                
                if (empty($button_text)) {
                    $button_text = 'Learn More';
                }
                
                if (empty($icon_class)) {
                    $icon_class = 'fas fa-user-check';
                }
                
                $feature_list = !empty($features) ? explode("\n", $features) : array();
            ?>
            <div class="kavod-program-card" data-program-id="<?php the_ID(); ?>">
                <div class="kavod-program-icon">
                    <i class="<?php echo esc_attr($icon_class); ?>"></i>
                </div>
                <div class="kavod-program-content">
                    <h3><?php the_title(); ?></h3>
                    <p><?php echo get_the_excerpt(); ?></p>

                    
                    <?php if (!empty($feature_list)) : ?>
                    <ul class="kavod-program-features">
                        <?php foreach (array_slice($feature_list, 0, 4) as $feature) : ?>
                        <li><i class="fas fa-check-circle"></i> <?php echo esc_html($feature); ?></li>
                        <?php endforeach; ?>
                    </ul>
                    <?php endif; ?>
                    
                    <a href="#" class="kavod-btn" data-program-id="<?php the_ID(); ?>"><?php echo esc_html($button_text); ?></a>
                </div>
            </div>
            <?php endwhile; wp_reset_postdata(); ?>
        </div>
        
        <!-- Sidebar Popup -->
        <div class="kavod-program-popup">
            <div class="kavod-popup-content">
                <span class="kavod-popup-close">&times;</span>
                <div class="kavod-popup-header">
                    <h2></h2>
                    <p class="kavod-popup-duration"></p>
                </div>
                <div class="kavod-popup-body">
                    <div class="kavod-popup-description"></div>
                    
                    <div class="kavod-popup-features">
                        <h3>Program Features</h3>
                        <ul></ul>
                    </div>
                </div>
                <div class="kavod-popup-footer">
                    <a href="#" class="kavod-btn kavod-popup-button">Learn More</a>
                </div>
            </div>
        </div>
        
        <!-- Overlay -->
        <div class="kavod-popup-overlay"></div>
    </div>
    
    <script>
    (function($) {
        $(document).ready(function() {
            var programData = {};
            
            // Store program data
            <?php 
            $programs->rewind_posts();
            while ($programs->have_posts()) : $programs->the_post(); 
                $duration = get_post_meta(get_the_ID(), '_program_duration', true);
                $features = get_post_meta(get_the_ID(), '_program_features', true);
                $button_text = get_post_meta(get_the_ID(), '_program_button_text', true);
                $button_url = get_post_meta(get_the_ID(), '_program_button_url', true);
                
                $feature_list = !empty($features) ? explode("\n", $features) : array();
                $features_json = json_encode($feature_list);
            ?>
            programData[<?php the_ID(); ?>] = {
                title: '<?php echo esc_js(get_the_title()); ?>',
                duration: '<?php echo esc_js($duration); ?>',
                content: <?php echo wp_json_encode(wpautop(apply_filters('the_content', get_the_content()))); ?>,
                features: <?php echo $features_json; ?>,
                buttonText: '<?php echo esc_js($button_text); ?>',
                buttonUrl: '<?php echo esc_js($button_url); ?>'
            };
            <?php endwhile; wp_reset_postdata(); ?>
            
            // Open popup when program card is clicked
            $('.kavod-program-card, .kavod-btn').on('click', function(e) {
                e.preventDefault();
                var programId = $(this).data('program-id');
                openPopup(programId);
            });
            
            // Close popup when close button is clicked
            $('.kavod-popup-close, .kavod-popup-overlay').on('click', function(e) {
                e.preventDefault();
                closePopup();
            });
            
            // Close with ESC key
            $(document).on('keyup', function(e) {
                if (e.key === "Escape") {
                    closePopup();
                }
            });
            
            function openPopup(programId) {
                var data = programData[programId];
                
                // Update popup content
                $('.kavod-popup-header h2').text(data.title);
                $('.kavod-popup-duration').text(data.duration);
                $('.kavod-popup-description').html(data.content);
                
                // Update features
                var featuresList = $('.kavod-popup-features ul');
                featuresList.empty();
                
                if (data.features && data.features.length > 0) {
                    data.features.forEach(function(feature) {
                        featuresList.append('<li><i class="fas fa-check"></i> ' + feature + '</li>');
                    });
                    $('.kavod-popup-features').show();
                } else {
                    $('.kavod-popup-features').hide();
                }
                
                // Update button
                var button = $('.kavod-popup-button');
                button.text(data.buttonText || 'Learn More');
                
                if (data.buttonUrl) {
                    button.attr('href', data.buttonUrl);
                    button.show();
                } else {
                    button.hide();
                }
                
                // Show popup and overlay
                $('.kavod-program-popup, .kavod-popup-overlay').fadeIn(300, function() {
                    $('.kavod-program-popup').addClass('active');
                });
                
                // Prevent body scrolling
                $('body').css('overflow', 'hidden');
            }
            
            function closePopup() {
                $('.kavod-program-popup').removeClass('active');
                setTimeout(function() {
                    $('.kavod-program-popup, .kavod-popup-overlay').fadeOut(300);
                }, 300);
                
                // Restore body scrolling
                $('body').css('overflow', 'auto');
            }
        });
    })(jQuery);
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode('coaching_programs', 'kavod_coaching_programs_shortcode');
?>