<?php 

// Register Job Listing Post Type
function register_job_listing_post_type() {
    // Register Custom Post Type
    register_post_type('job_listing', array(
        'labels' => array(
            'name' => 'JC Jobs',
            'singular_name' => 'Job',
            'add_new' => 'Add New Job',
            'add_new_item' => 'Add New Job',
            'edit_item' => 'Edit Job',
            'new_item' => 'New Job',
            'view_item' => 'View Job',
            'search_items' => 'Search Jobs',
            'not_found' => 'No Jobs Found',
            'not_found_in_trash' => 'No Jobs Found in Trash',
            'all_items' => 'All Jobs',
        ),
        'public' => true,
        'menu_icon' => 'dashicons-businessman',
        'has_archive' => true,
        'rewrite' => array('slug' => 'jobs'),
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'show_in_rest' => true,
    ));

    // Register Taxonomies: Job Type and Job Category
    register_taxonomy('job_type', 'job_listing', array(
        'labels' => array(
            'name' => 'Job Types',
            'singular_name' => 'Job Type',
        ),
        'public' => true,
        'hierarchical' => true,
        'show_in_rest' => true,
    ));

    register_taxonomy('job_category', 'job_listing', array(
        'labels' => array(
            'name' => 'Job Categories',
            'singular_name' => 'Job Category',
        ),
        'public' => true,
        'hierarchical' => true,
        'show_in_rest' => true,
    ));
}
add_action('init', 'register_job_listing_post_type');

//Adding Custom Meta Boxes
function job_listing_meta_boxes() {
    add_meta_box('job_details', 'Job Details', 'job_listing_meta_box_callback', 'job_listing', 'normal', 'high');
}
add_action('add_meta_boxes', 'job_listing_meta_boxes');

function job_listing_meta_box_callback($post) {
    // Retrieve existing values
    $location = get_post_meta($post->ID, '_job_location', true);
    $start_date = get_post_meta($post->ID, '_job_start_date', true);
    $expiry_date = get_post_meta($post->ID, '_job_expiry_date', true);
    $company = get_post_meta($post->ID, '_job_company', true);
    $jed_file_id = get_post_meta($post->ID, '_job_jed_file_id', true);
    $apply_url = get_post_meta($post->ID, '_job_apply_url', true);
    $jed_file_url = $jed_file_id ? wp_get_attachment_url($jed_file_id) : '';

    ?>
    <div class="row">
        <div class="col-md-12" style="width: 100%; margin-bottom:10px;float:left;">
                <label for="job_location">Location</label><br>
                <input type="text" id="job_location" name="job_location" value="<?= esc_attr($location); ?>" class="widefat">
        </div>
        <div class="col-md-6" style="width: 49%; margin-bottom:10px; margin-right: 10px;float:left;">
                <label for="job_start_date">Start Date</label><br>
                <input type="date" id="job_start_date" name="job_start_date" value="<?= esc_attr($start_date); ?>" class="widefat">
        </div>
        <div class="col-md-6" style="width: 49%; margin-bottom:10px;float:left;">
                <label for="job_expiry_date">Expiry Date</label><br>
                <input type="date" id="job_expiry_date" name="job_expiry_date" value="<?= esc_attr($expiry_date); ?>" class="widefat">
        </div>
        <div class="col-md-12" style="width: 100%; margin-bottom:10px;float:left;">
            <label for="job_company">Company</label><br>
            <input type="text" id="job_company" name="job_company" value="<?= esc_attr($company); ?>" class="widefat">
        </div>
        <div class="col-md-12" style="width: 100%; margin-bottom:10px;float:left;">
            <label for="job_jed_file">Upload JED File</label><br>
            <?php if ($jed_file_url): ?>
                <a href="<?= esc_url($jed_file_url); ?>" target="_blank">View Uploaded File</a><br>
            <?php endif; ?>
            <input type="file" id="job_jed_file" name="job_jed_file" accept=".pdf,.doc,.docx" class="widefat">
        </div>
        <div class="col-md-12" style="width: 100%; margin-bottom:10px;float:left;">
            <label for="job_apply_url">Application URL</label><br>
            <input type="url" id="job_apply_url" name="job_apply_url" value="<?= esc_attr($apply_url); ?>" class="widefat">
        </div>
    </div>
    <?php
}

// Save the uploaded file and other meta data
function save_job_meta_data($post_id) {
    $fields = ['job_location', 'job_start_date', 'job_expiry_date', 'job_company', 'job_apply_url'];
    foreach ($fields as $field) {
        if (isset($_POST[$field])) {
            update_post_meta($post_id, '_' . $field, sanitize_text_field($_POST[$field]));
        }
    }

    // Handle JED file upload
    if (!empty($_FILES['job_jed_file']['name'])) {
        $uploaded_file = $_FILES['job_jed_file'];
        $upload = wp_handle_upload($uploaded_file, ['test_form' => false]);

        if (isset($upload['file'])) {
            // Upload was successful
            $attachment_id = wp_insert_attachment([
                'post_mime_type' => $upload['type'],
                'post_title' => sanitize_file_name($uploaded_file['name']),
                'post_content' => '',
                'post_status' => 'inherit',
            ], $upload['file'], $post_id);

            // Generate attachment metadata and update the database
            require_once(ABSPATH . 'wp-admin/includes/image.php');
            wp_generate_attachment_metadata($attachment_id, $upload['file']);
            update_post_meta($post_id, '_job_jed_file_id', $attachment_id);
        }
    }
}
add_action('save_post', 'save_job_meta_data');

//Customize Admin Dashboard Columns
function job_listing_admin_columns($columns) {
    unset($columns['date']);
    $columns['company'] = 'Company';
    $columns['job_type'] = 'Job Type';
    $columns['job_category'] = 'Job Category';
    $columns['start_date'] = 'Start Date';
    $columns['expiry_date'] = 'Expiry Date';
    return $columns;
}
add_filter('manage_job_listing_posts_columns', 'job_listing_admin_columns');

function job_listing_admin_columns_content($column, $post_id) {
    switch ($column) {
        case 'company':
            echo esc_html(get_post_meta($post_id, '_job_company', true));
            break;
        case 'job_type':
            echo get_the_term_list($post_id, 'job_type', '', ', ', '');
            break;
        case 'job_category':
            echo get_the_term_list($post_id, 'job_category', '', ', ', '');
            break;
        case 'start_date':
            echo esc_html(get_post_meta($post_id, '_job_start_date', true));
            break;
        case 'expiry_date':
            echo esc_html(get_post_meta($post_id, '_job_expiry_date', true));
            break;
    }
}
add_action('manage_job_listing_posts_custom_column', 'job_listing_admin_columns_content', 10, 2);


//For Single job listing Template
function custom_single_template($single) {
    global $post;

    // Check if the post type is 'job_listing'
    if ($post->post_type === 'job_listing') {
        // Path to your custom single template
        $custom_template = locate_template('single-joblisting.php');
        if ($custom_template) {
            return $custom_template;
        }
    }
    // Default template
    return $single;
}
add_filter('single_template', 'custom_single_template');

//Shortcode to Display Jobs
function display_jobs_shortcode($atts) {
    $atts = shortcode_atts(['category' => ''], $atts, 'job_listings');
    $args = [
        'post_type' => 'job_listing',
        'posts_per_page' => -1,
        'meta_query' => [
            [
                'key' => '_job_expiry_date',
                'value' => date('Y-m-d'),
                'compare' => '>=', // Show jobs that are not expired
                'type' => 'DATE',
            ]
        ],
        'orderby' => 'date',
        'order' => 'DESC',
    ];

    if (!empty($atts['category'])) {
        $args['tax_query'] = [
            [
                'taxonomy' => 'job_category',
                'field' => 'slug',
                'terms' => $atts['category'],
            ],
        ];
    }

    $query = new WP_Query($args);
    ob_start();
    if ($query->have_posts()) {
        echo '<div class="job-listings">';
        while ($query->have_posts()) {
            $query->the_post(); // Sets up global post data for each job
            $location = get_post_meta(get_the_ID(), '_job_location', true);
            $company = get_post_meta(get_the_ID(), '_job_company', true);
            $start_date = get_post_meta(get_the_ID(), '_job_start_date', true);
            $expiry_date = get_post_meta(get_the_ID(), '_job_expiry_date', true);

            $job_type = get_post_meta(get_the_ID(), 'job_type', true);
            // Fetch job categories
            $jobtypes = get_the_terms(get_the_ID(), 'job_type');
            $jobtype_names = !empty($jobtypes) ? implode(', ', wp_list_pluck($jobtypes, 'name')) : 'None';

            // Fetch job categories
            $categories = get_the_terms(get_the_ID(), 'job_category');
            $category_names = !empty($categories) ? implode(', ', wp_list_pluck($categories, 'name')) : 'Uncategorized';

            ?>
            <div class="card card-body border-0 shadow mb-3 kpy-job" data-aos="fade-up">
                <div class="row">
                    <div class="col-md-8" style="border-right: 1px solid #666;">
                        <h3><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
                        <hr><strong>Company:</strong> <?= esc_html($company); ?><hr>
                        <?php the_excerpt(); ?>
                    </div>
                    <div class="col-md-4">
                        <p><strong>Job Type:</strong> <?= esc_html($jobtype_names); ?></p>
                        <p><strong>Category:</strong> <?= esc_html($category_names); ?></p>
                        <p><strong>Expiry:</strong> <?= esc_html(date('jS, F Y', strtotime($expiry_date))); ?></p>
                        <p><strong>Location:</strong> <?= esc_html($location); ?></p>
                        <a href="<?php the_permalink(); ?>">View full details <i class="bi bi-chevron-double-right"></i></a>
                    </div>
                </div>
                
                
                
            </div>
            <?php
        }
        echo '</div>';
    } else {
        echo 'No jobs available.';
    }

    wp_reset_postdata(); // Restore global post data
    return ob_get_clean();
}
add_shortcode('job_listings', 'display_jobs_shortcode');

//Shortcode to Display HOMEPAGE Jobs
function display_home_jobs_shortcode($atts) {
    $atts = shortcode_atts(['category' => ''], $atts, 'job_list');
    $args = [
        'post_type' => 'job_listing',
        'posts_per_page' => 3,
        'meta_query' => [
            [
                'key' => '_job_expiry_date',
                'value' => date('Y-m-d'),
                'compare' => '>=', // Show jobs that are not expired
                'type' => 'DATE',
            ]
        ],
        'orderby' => 'date',
        'order' => 'DESC',
    ];

    if (!empty($atts['category'])) {
        $args['tax_query'] = [
            [
                'taxonomy' => 'job_category',
                'field' => 'slug',
                'terms' => $atts['category'],
            ],
        ];
    }

    $query = new WP_Query($args);
    ob_start();
    if ($query->have_posts()) {
        echo '<div class="job-listings">';
        while ($query->have_posts()) {
            $query->the_post(); // Sets up global post data for each job
            $location = get_post_meta(get_the_ID(), '_job_location', true);
            $company = get_post_meta(get_the_ID(), '_job_company', true);
            $start_date = get_post_meta(get_the_ID(), '_job_start_date', true);
            $expiry_date = get_post_meta(get_the_ID(), '_job_expiry_date', true);

            ?>
            <div class="card card-body border-0 shadow mb-3 kpy-home-job" data-aos="fade-up">
                <div class="row">
                    <div class="col-md-12">
                        <h4><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
                        <span class="date"><strong>Expiry:</strong> <?= esc_html(date('jS, F Y', strtotime($expiry_date))); ?></span>
                    </div>
                    <div class="col-md-7">
                        <strong>Company:</strong> <?= esc_html($company); ?>
                    </div>
                    <div class="col-md-5">
                         <strong>Location:</strong> <?= esc_html($location); ?>
                    </div>
                </div>
            </div>
            <?php
        }
        echo '</div>';
    } else {
        echo 'No jobs available.';
    }

    wp_reset_postdata(); // Restore global post data
    return ob_get_clean();
}
add_shortcode('job_list', 'display_home_jobs_shortcode');


