<?php
/**
 * Template Name: Destination Detail
 */

// 1️⃣ Register the Custom Post Type for Destinations
function register_destinations_cpt() {
    $labels = array(
        'name'               => 'Destinations',
        'singular_name'      => 'Destination',
        'menu_name'          => 'Destinations',
        'name_admin_bar'     => 'Destination',
        'add_new'            => 'Add New',
        'add_new_item'       => 'Add New Destination',
        'new_item'           => 'New Destination',
        'edit_item'          => 'Edit Destination',
        'view_item'          => 'View Destination',
        'all_items'          => 'All Destinations',
    );

    $args = array(
        'labels'             => $labels,
        'public'             => true,
        'has_archive'        => true,
        'menu_icon'          => 'dashicons-location-alt',
        'rewrite'            => array('slug' => 'destinations'),
        'supports'           => array('title', 'editor', 'thumbnail', 'excerpt'),
        'taxonomies'         => array('category', 'post_tag'),
    );

    register_post_type('destination', $args);
}
add_action('init', 'register_destinations_cpt');

// 2️⃣ Add custom meta boxes for destination fields
function destinations_add_meta_boxes() {
    add_meta_box('destination_overview', 'Overview Information', 'destination_overview_callback', 'destination', 'normal', 'high');
    add_meta_box('destination_places', 'Places & Activities', 'destination_places_callback', 'destination', 'normal', 'high');
    add_meta_box('destination_packages', 'Popular Packages Settings', 'destination_packages_callback', 'destination', 'normal', 'high');
    add_meta_box('destination_useful_info', 'Useful Information', 'destination_useful_info_callback', 'destination', 'normal', 'high');
}
add_action('add_meta_boxes', 'destinations_add_meta_boxes');

// Overview meta box
function destination_overview_callback($post) {
    wp_nonce_field('save_destination_details', 'destination_nonce');

    $map_image = get_post_meta($post->ID, '_map_image', true);
    $overview_text = get_post_meta($post->ID, '_overview_text', true);
    
    ?>
    <table style="width: 100%;">
        <tr>
            <td style="width: 100%;">
                <p><label><strong>Country Map Image URL:</strong></label><br>
                    <input type="url" name="map_image" value="<?php echo esc_attr($map_image); ?>" placeholder="https://example.com/map-image.jpg" style="width:100%;">
                    <small>Upload the map image to media library and paste the URL here</small>
                </p>
            </td>
        </tr>
    </table>
    
    <p><label><strong>Overview Description:</strong></label><br>
        <textarea name="overview_text" rows="8" style="width:100%;" placeholder="Enter the overview description of this destination..."><?php echo esc_textarea($overview_text); ?></textarea>
    </p>
    <?php
}

// Places & Activities meta box
function destination_places_callback($post) {
    $places_activities = get_post_meta($post->ID, '_places_activities', true);
    if (!is_array($places_activities)) $places_activities = array();
    
    ?>
    <div id="places-activities-container">
        <p><strong>Places & Activities:</strong></p>
        <div id="places-list">
            <?php 
            if (!empty($places_activities)) {
                foreach ($places_activities as $index => $place) {
                    ?>
                    <div class="place-item" style="border: 1px solid #ddd; padding: 15px; margin: 10px 0; border-radius: 5px;">
                        <table style="width: 100%;">
                            <tr>
                                <td style="width: 70%; padding-right: 15px;">
                                    <p><label><strong>Place Name:</strong></label><br>
                                        <input type="text" name="places_activities[<?php echo $index; ?>][name]" value="<?php echo esc_attr($place['name'] ?? ''); ?>" style="width:100%;">
                                    </p>
                                </td>
                                <td style="width: 30%;">
                                    <p><label><strong>Image URL:</strong></label><br>
                                        <input type="url" name="places_activities[<?php echo $index; ?>][image]" value="<?php echo esc_attr($place['image'] ?? ''); ?>" style="width:100%;">
                                    </p>
                                </td>
                            </tr>
                        </table>
                        <p><label><strong>Description:</strong></label><br>
                            <textarea name="places_activities[<?php echo $index; ?>][description]" rows="4" style="width:100%;"><?php echo esc_textarea($place['description'] ?? ''); ?></textarea>
                        </p>
                        <button type="button" class="remove-place" style="background: #dc3545; color: white; padding: 5px 10px; border: none; border-radius: 3px;">Remove</button>
                    </div>
                    <?php
                }
            }
            ?>
        </div>
        <button type="button" id="add-place" style="background: #28a745; color: white; padding: 8px 15px; border: none; border-radius: 3px; margin-top: 10px;">Add Place/Activity</button>
    </div>
    
    <script>
    jQuery(document).ready(function($) {
        let placeIndex = <?php echo count($places_activities); ?>;
        
        $('#add-place').on('click', function() {
            const html = `
                <div class="place-item" style="border: 1px solid #ddd; padding: 15px; margin: 10px 0; border-radius: 5px;">
                    <table style="width: 100%;">
                        <tr>
                            <td style="width: 70%; padding-right: 15px;">
                                <p><label><strong>Place Name:</strong></label><br>
                                    <input type="text" name="places_activities[${placeIndex}][name]" style="width:100%;">
                                </p>
                            </td>
                            <td style="width: 30%;">
                                <p><label><strong>Image URL:</strong></label><br>
                                    <input type="url" name="places_activities[${placeIndex}][image]" style="width:100%;">
                                </p>
                            </td>
                        </tr>
                    </table>
                    <p><label><strong>Description:</strong></label><br>
                        <textarea name="places_activities[${placeIndex}][description]" rows="4" style="width:100%;"></textarea>
                    </p>
                    <button type="button" class="remove-place" style="background: #dc3545; color: white; padding: 5px 10px; border: none; border-radius: 3px;">Remove</button>
                </div>
            `;
            $('#places-list').append(html);
            placeIndex++;
        });
        
        $(document).on('click', '.remove-place', function() {
            $(this).closest('.place-item').remove();
        });
    });
    </script>
    <?php
}

// Popular Packages meta box
function destination_packages_callback($post) {
    $show_packages = get_post_meta($post->ID, '_show_packages', true);
    $packages_title = get_post_meta($post->ID, '_packages_title', true);
    
    ?>
    <p><label><strong>Display Popular Packages:</strong></label><br>
        <input type="checkbox" name="show_packages" value="1" <?php checked($show_packages, '1'); ?>> 
        Show popular packages for this destination
    </p>
    
    <p><label><strong>Packages Section Title:</strong></label><br>
        <input type="text" name="packages_title" value="<?php echo esc_attr($packages_title ?: 'Popular ' . get_the_title() . ' Packages'); ?>" style="width:100%;">
    </p>
    <?php
}

// Useful Information meta box
function destination_useful_info_callback($post) {
    $useful_info_items = array(
        'before_travel' => 'Before You Travel',
        'visa_requirements' => 'Visa and Entry Requirements', 
        'what_to_pack' => 'What to Pack',
        'weather' => 'Weather',
        'cultural_etiquette' => 'Cultural Etiquette',
        'religion' => 'Religion',
        'currency_payments' => 'Currency and Payments',
        'communication_internet' => 'Communication and Internet',
        'getting_around' => 'Getting Around',
        'health_safety' => 'Health and Safety'
    );
    
    ?>
    <p><strong>Useful Information Sections:</strong></p>
    <?php foreach ($useful_info_items as $key => $label) : 
        $content = get_post_meta($post->ID, '_useful_info_' . $key, true);
    ?>
        <div style="border: 1px solid #ddd; margin: 10px 0; border-radius: 5px;">
            <h4 style="background: #f9f9f9; margin: 0; padding: 15px; border-bottom: 1px solid #ddd;"><?php echo esc_html($label); ?></h4>
            <div style="padding: 15px;">
                <textarea name="useful_info_<?php echo $key; ?>" rows="4" style="width:100%;" placeholder="Enter information for <?php echo esc_attr($label); ?>..."><?php echo esc_textarea($content); ?></textarea>
            </div>
        </div>
    <?php endforeach; ?>
    <?php
}

// Save destination meta data
function save_destination_details($post_id) {
    if (!isset($_POST['destination_nonce']) || !wp_verify_nonce($_POST['destination_nonce'], 'save_destination_details')) {
        return;
    }
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;
    if (get_post_type($post_id) !== 'destination') return;

    // Save overview data
    if (isset($_POST['map_image'])) update_post_meta($post_id, '_map_image', esc_url_raw($_POST['map_image']));
    if (isset($_POST['overview_text'])) update_post_meta($post_id, '_overview_text', sanitize_textarea_field($_POST['overview_text']));
    
    // Save places & activities
    if (isset($_POST['places_activities'])) {
        $places_activities = array();
        foreach ($_POST['places_activities'] as $place) {
            if (!empty($place['name'])) {
                $places_activities[] = array(
                    'name' => sanitize_text_field($place['name']),
                    'image' => esc_url_raw($place['image']),
                    'description' => sanitize_textarea_field($place['description'])
                );
            }
        }
        update_post_meta($post_id, '_places_activities', $places_activities);
    }
    
    // Save packages settings
    update_post_meta($post_id, '_show_packages', isset($_POST['show_packages']) ? '1' : '0');
    if (isset($_POST['packages_title'])) update_post_meta($post_id, '_packages_title', sanitize_text_field($_POST['packages_title']));
    
    // Save useful information
    $useful_info_items = array('before_travel', 'visa_requirements', 'what_to_pack', 'weather', 'cultural_etiquette', 'religion', 'currency_payments', 'communication_internet', 'getting_around', 'health_safety');
    
    foreach ($useful_info_items as $item) {
        if (isset($_POST['useful_info_' . $item])) {
            update_post_meta($post_id, '_useful_info_' . $item, sanitize_textarea_field($_POST['useful_info_' . $item]));
        }
    }
}
add_action('save_post', 'save_destination_details');

// 3️⃣ Shortcode for destination detail tabs
function destination_detail_shortcode($atts) {
    $atts = shortcode_atts(array(
        'id' => get_the_ID()
    ), $atts);
    
    $destination_id = $atts['id'];
    if (!$destination_id) return '<p>No destination specified.</p>';
    
    $destination = get_post($destination_id);
    if (!$destination || $destination->post_type !== 'destination') {
        return '<p>Invalid destination.</p>';
    }

    // Get meta data
    $map_image = get_post_meta($destination_id, '_map_image', true);
    $overview_text = get_post_meta($destination_id, '_overview_text', true);
    $places_activities = get_post_meta($destination_id, '_places_activities', true);
    $show_packages = get_post_meta($destination_id, '_show_packages', true);
    $packages_title = get_post_meta($destination_id, '_packages_title', true);
    
    if (!is_array($places_activities)) $places_activities = array();
    
    ob_start(); ?>
    
    <div class="dest-detail-panel">
        <!-- Tab Navigation -->
        <div class="dest-tabs">
            <div class="dest-tab-item active" data-tab="overview">
                <i class="fas fa-globe-americas"></i>
                <span>Overview</span>
            </div>
            <div class="dest-tab-item" data-tab="places">
                <i class="fas fa-camera"></i>
                <span>Places & Activities</span>
            </div>
            <div class="dest-tab-item" data-tab="packages">
                <i class="fas fa-suitcase"></i>
                <span>Popular <?php echo esc_html($destination->post_title); ?> Packages</span>
            </div>
            <div class="dest-tab-item" data-tab="useful-info">
                <i class="fas fa-info-circle"></i>
                <span>Useful Info</span>
            </div>
        </div>

        <!-- Tab Content -->
        <div class="dest-tab-content">
            <!-- Overview Tab -->
            <div id="overview" class="dest-tab-pane active">
                <div class="overview-content">
                    <?php if ($map_image) : ?>
                        <div class="map-container" style="text-align: center; margin-bottom: 30px;">
                            <img src="<?php echo esc_url($map_image); ?>" alt="<?php echo esc_attr($destination->post_title); ?> Map" style="max-width: 100%; height: auto; border-radius: 8px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($overview_text) : ?>
                        <div class="overview-text" style="font-size: 1.1rem; line-height: 1.7; color: #444;">
                            <?php echo nl2br(esc_html($overview_text)); ?>
                        </div>
                    <?php else : ?>
                        <p style="text-align: center; color: #666; font-style: italic;">No overview information available.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Places & Activities Tab -->
            <div id="places" class="dest-tab-pane">
                <?php if (!empty($places_activities)) : ?>
                    <div class="dest-places-grid">
                        <?php foreach ($places_activities as $place) : ?>
                            <div class="dest-place-card">
                                <?php if (!empty($place['image'])) : ?>
                                    <div class="dest-place-image">
                                        <img src="<?php echo esc_url($place['image']); ?>" alt="<?php echo esc_attr($place['name']); ?>">
                                    </div>
                                <?php endif; ?>
                                <div class="place-content">
                                    <h3><?php echo esc_html($place['name']); ?></h3>
                                    <p><?php echo nl2br(esc_html($place['description'])); ?></p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else : ?>
                    <div class="dest-no-content">
                        <i class="fas fa-camera" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.3;"></i>
                        <p>No places or activities information available.</p>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Popular Packages Tab -->
            <div id="packages" class="dest-tab-pane">
                <?php if ($show_packages === '1') : ?>
                    <div class="packages-section">
                        <h2 style="text-align: center; margin-bottom: 30px; color: #333;">
                            <?php echo esc_html($packages_title ?: 'Popular ' . $destination->post_title . ' Packages'); ?>
                        </h2>
                        
                        <div class="packages-grid">
                            <?php
                            // Get travel packages related to this destination
                            $packages_query = new WP_Query(array(
                                'post_type' => 'travel_package',
                                'posts_per_page' => -1,
                                'meta_query' => array(
                                    array(
                                        'key' => '_destination',
                                        'value' => $destination->post_title,
                                        'compare' => 'LIKE'
                                    )
                                )
                            ));

                            if ($packages_query->have_posts()) :
                                while ($packages_query->have_posts()) : $packages_query->the_post();
                                    $price_usd = get_post_meta(get_the_ID(), '_price_usd', true);
                                    $price_ugx = get_post_meta(get_the_ID(), '_price_ugx', true);
                                    $package_destination = get_post_meta(get_the_ID(), '_destination', true);
                                    $reviews = get_post_meta(get_the_ID(), '_reviews', true);
                                    $badge = get_post_meta(get_the_ID(), '_badge', true);
                                    
                                    $image_id = get_post_thumbnail_id(get_the_ID());
                                    if ($image_id) {
                                        $image = wp_get_attachment_image_url($image_id, 'medium_large');
                                    } else {
                                        $image = 'https://via.placeholder.com/400x250/f0f0f0/999999?text=No+Image';
                                    }
                                    ?>
                                    <div class="dest-package-card-enhanced">
                                        <div class="dest-package-image-enhanced" style="background-image: url('<?php echo esc_url($image); ?>');">
                                            <?php if ($badge) : ?>
                                                <div class="dest-package-badge"><?php echo esc_html($badge); ?></div>
                                            <?php endif; ?>
                                            <div class="dest-package-price">
                                                <?php if ($price_usd) : ?>
                                                    <span class="dest-price-usd"><?php echo esc_html($price_usd); ?></span>
                                                <?php endif; ?>
                                                <?php if ($price_ugx) : ?>
                                                    <span class="dest-price-ugx"><?php echo esc_html($price_ugx); ?></span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        
                                        <div class="dest-package-info-enhanced">
                                            <h3><?php the_title(); ?></h3>
                                            <?php if ($package_destination) : ?>
                                                <div class="destination-info">
                                                    <i class="fas fa-map-marker-alt"></i>
                                                    <?php echo esc_html($package_destination); ?>
                                                </div>
                                            <?php endif; ?>
                                            <div class="dest-package-excerpt">
                                                <?php 
                                                if (has_excerpt()) {
                                                    echo wp_trim_words(get_the_excerpt(), 15, '...');
                                                } else {
                                                    echo wp_trim_words(get_the_content(), 15, '...');
                                                }
                                                ?>
                                            </div>
                                            
                                            <div class="dest-package-footer">
                                                <?php if ($reviews && $reviews > 0) : ?>
                                                    <div class="dest-rating">
                                                        <span class="dest-stars">★★★★★</span>
                                                        <span class="dest-review-count">(<?php echo intval($reviews); ?>)</span>
                                                    </div>
                                                <?php endif; ?>
                                                <a href="<?php the_permalink(); ?>" class="dest-view-package-btn">View Package</a>
                                            </div>
                                        </div>
                                    </div>
                                <?php endwhile;
                                wp_reset_postdata();
                            else : ?>
                                <div class="dest-no-content" style="grid-column: 1/-1;">
                                    <i class="fas fa-suitcase" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.3;"></i>
                                    <p>No packages found for this destination.</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php else : ?>
                    <div class="dest-no-content">
                        <i class="fas fa-suitcase" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.3;"></i>
                        <p>Package display is disabled for this destination.</p>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Useful Info Tab -->
            <div id="useful-info" class="dest-tab-pane">
                <div class="dest-accordion">
                    <?php 
                    $useful_info_items = array(
                        'before_travel' => 'Before You Travel',
                        'visa_requirements' => 'Visa and Entry Requirements', 
                        'what_to_pack' => 'What to Pack',
                        'weather' => 'Weather',
                        'cultural_etiquette' => 'Cultural Etiquette',
                        'religion' => 'Religion',
                        'currency_payments' => 'Currency and Payments',
                        'communication_internet' => 'Communication and Internet',
                        'getting_around' => 'Getting Around',
                        'health_safety' => 'Health and Safety'
                    );
                    
                    foreach ($useful_info_items as $key => $label) :
                        $content = get_post_meta($destination_id, '_useful_info_' . $key, true);
                        if ($content) :
                    ?>
                        <div class="dest-accordion-item">
                            <div class="accordion-header" data-target="<?php echo esc_attr($key); ?>">
                                <h3><?php echo esc_html($label); ?></h3>
                                <i class="fas fa-chevron-down"></i>
                            </div>
                            <div class="dest-accordion-content" id="<?php echo esc_attr($key); ?>">
                                <div class="dest-accordion-body">
                                    <?php echo nl2br(esc_html($content)); ?>
                                </div>
                            </div>
                        </div>
                    <?php 
                        endif;
                    endforeach; 
                    
                    // Check if any content exists
                    $has_content = false;
                    foreach ($useful_info_items as $key => $label) {
                        if (get_post_meta($destination_id, '_useful_info_' . $key, true)) {
                            $has_content = true;
                            break;
                        }
                    }
                    
                    if (!$has_content) :
                    ?>
                        <div class="dest-no-content">
                            <i class="fas fa-info-circle" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.3;"></i>
                            <p>No useful information available.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    
    
    <script>
        jQuery(document).ready(function($) {
            // Tab switching
            $('.dest-tab-item').on('click', function() {
                $('.dest-tab-item').removeClass('active');
                $(this).addClass('active');
                $('.dest-tab-pane').removeClass('active');
                var tabId = $(this).data('tab');
                $('#' + tabId).addClass('active');
            });

            // Accordion functionality
            $('.accordion-header').on('click', function() {
                var $header = $(this);
                var $content = $header.next('.dest-accordion-content');
                var $item = $header.closest('.dest-accordion-item');

                // Toggle active state
                $header.toggleClass('active');
                $content.toggleClass('active');

                // Close other accordion items (optional - remove if you want multiple open)
                $('.accordion-header').not($header).removeClass('active');
                $('.dest-accordion-content').not($content).removeClass('active');
            });
        });
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode('destination_detail', 'destination_detail_shortcode');

// Enqueue assets for destinations
function destinations_enqueue_assets() {
    wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css', array(), '5.15.4');
    
    if (!wp_script_is('jquery', 'enqueued')) {
        wp_enqueue_script('jquery');
    }
}
add_action('wp_enqueue_scripts', 'destinations_enqueue_assets');

// Add featured image support
function destinations_add_thumbnail_support() {
    add_theme_support('post-thumbnails');
    add_image_size('destination-thumb', 800, 500, true);
}
add_action('after_setup_theme', 'destinations_add_thumbnail_support');

function force_jquery_destinations() {
    wp_enqueue_script('jquery');
}
add_action('wp_enqueue_scripts', 'force_jquery_destinations', 1);