<?php 
/* 
Template Name: Contact Page Template 
Description: A modern contact page template with form submission handling and map integration
*/
get_header(); 
include get_template_directory() . '/inc/menus/menu.php'; 

// Register Contact Inquiry CPT
function bss_register_contact_inquiry_cpt() {
    $labels = [
        'name'               => 'Contact Inquiries',
        'singular_name'      => 'Contact Inquiry',
        'menu_name'          => 'Inquiries',
        'name_admin_bar'     => 'Inquiry',
        'add_new'            => 'Add New',
        'add_new_item'       => 'Add New Inquiry',
        'edit_item'          => 'Edit Inquiry',
        'new_item'           => 'New Inquiry',
        'view_item'          => 'View Inquiry',
        'all_items'          => 'All Inquiries',
        'search_items'       => 'Search Inquiries',
        'not_found'          => 'No inquiries found.',
        'not_found_in_trash' => 'No inquiries found in Trash.'
    ];

    $args = [
        'labels'             => $labels,
        'public'             => false,   // keep private
        'show_ui'            => true,    // show in admin
        'show_in_menu'       => true,
        'menu_icon'          => 'dashicons-email',
        'supports'           => ['title'], // no editor
        'capability_type'    => 'post',
    ];

    register_post_type('contact_inquiry', $args);
}
add_action('init', 'bss_register_contact_inquiry_cpt');


// Add custom columns
function bss_set_inquiry_columns($columns) {
    return [
        'cb'           => '<input type="checkbox" />',
        'title'        => 'Name',
        'inquiry_type' => 'Inquiry Type',
        'phone'        => 'Phone',
        'email'        => 'Email',
        'date'         => 'Date',
    ];
}
add_filter('manage_contact_inquiry_posts_columns', 'bss_set_inquiry_columns');

// Fill custom columns
function bss_custom_inquiry_column($column, $post_id) {
    switch ($column) {
        case 'inquiry_type':
            echo esc_html(get_post_meta($post_id, 'inquiry_type', true));
            break;
        case 'phone':
            echo esc_html(get_post_meta($post_id, 'phone', true));
            break;
        case 'email':
            $email = get_post_meta($post_id, 'email', true);
            if ($email) {
                echo '<a href="mailto:' . esc_attr($email) . '">' . esc_html($email) . '</a>';
            }
            break;
    }
}
add_action('manage_contact_inquiry_posts_custom_column', 'bss_custom_inquiry_column', 10, 2);

// Make columns sortable
function bss_inquiry_sortable_columns($columns) {
    $columns['inquiry_type'] = 'inquiry_type';
    $columns['email'] = 'email';
    return $columns;
}
add_filter('manage_edit-contact_inquiry_sortable_columns', 'bss_inquiry_sortable_columns');


// Honeypot Field (hidden from real users)
$honeypot = '<input type="text" name="company_name" style="display:none">';

// Handle Form Submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['contact_inquiry_submit'])) {
    $ip_address  = $_SERVER['REMOTE_ADDR'];
    $form_time   = $_POST['form_time'] ?? time();
    $current_time = time();

    // Simple rate limit: 1 submission per 60 seconds
    if (($current_time - intval($form_time)) < 60) {
        echo '<div class="safari-alert safari-alert-danger">Please wait before submitting again.</div>';
    } elseif (!empty($_POST['company_name'])) {
        echo '<div class="safari-alert safari-alert-danger">Spam detected.</div>';
    } else {
        $full_name    = sanitize_text_field($_POST['full_name']);
        $inquiry_type = sanitize_text_field($_POST['inquiry_type']);
        $phone        = sanitize_text_field($_POST['phone']);
        $email        = sanitize_email($_POST['email']);
        $address      = sanitize_text_field($_POST['address']);
        $user_message = sanitize_textarea_field($_POST['message']); // ✅ renamed

        // Save submission as a custom post type
        $post_id = wp_insert_post([
            'post_title'  => $full_name . ' - ' . $inquiry_type,
            'post_type'   => 'contact_inquiry',
            'post_status' => 'publish',
            'meta_input'  => [
                'full_name'    => $full_name,
                'inquiry_type' => $inquiry_type,
                'phone'        => $phone,
                'email'        => $email,
                'address'      => $address,
                'message'      => $user_message,
                'ip_address'   => $ip_address
            ]
        ]);

        // Send Email Notification
        $to       = 'info@beautesauvagesafaris.com';
        $subject  = 'New Safari Inquiry - Beauté Sauvage Safaris';
        $body     = "New Safari Inquiry received:\n\n";
        $body    .= "Name: $full_name\n";
        $body    .= "Inquiry Type: $inquiry_type\n";
        $body    .= "Phone: $phone\n";
        $body    .= "Email: $email\n";
        $body    .= "Address: $address\n\n";
        $body    .= "Message:\n$user_message\n";
        $headers  = ['Content-Type: text/plain; charset=UTF-8'];

        wp_mail($to, $subject, $body, $headers);

       
    }
}
?>




<main class="contact-template">
        <?php 
    // Store the original post data
    $original_post = $post;
    
    // Get the background image safely
    $backgroundImg = wp_get_attachment_image_src(get_post_thumbnail_id($original_post->ID), 'full'); 
    
    // Get excerpt or trimmed content
    $excerpt = $original_post->post_excerpt;
    if (empty($excerpt)) {
        $excerpt = wp_trim_words($original_post->post_content, 30, '...');
    }
    ?>
    <!-- Hero Section -->
     <section class="parallax-hero">
  <div class="parallax-bg" 
       style="background-image:url('<?php echo $backgroundImg[0] ?: get_template_directory_uri().'/assets/images/default-hero.jpg'; ?>');">
  </div>

  <div class="hero-overlay"></div>

  <div class="container">
    <div class="row">
      <div class="col-lg-10 mx-auto">
        <div class="hero-content">
          <h1 class="hero-title fade-up"><?php echo esc_html($original_post->post_title); ?></h1>


          <div class="hero-breadcrumb fade-up">
            <?php if (function_exists('bcn_display')) : ?>
              <?php bcn_display(); ?> <!-- If Breadcrumb NavXT is installed -->
            <?php else : ?>
              <a href="<?php echo home_url(); ?>">Home</a> » 
              <span><?php echo esc_html($original_post->post_title); ?></span>
            <?php endif; ?>
          </div>
        </div>
      </div>
    </div>
  </div>
</section>

    <!-- Contact Content Section -->
    <section class="contact-content">
        <div class="contact-container">
            <!-- Contact Info -->
            <div class="contact-info">
                <h2 class="contact-info-title">Get In Touch</h2>
                <p class="contact-info-description">
                    Ready to embark on your dream safari adventure? Get in touch with us today to start planning your unforgettable journey through Uganda's magnificent wilderness.
                </p>
                
                <div class="contact-details">
                    <div class="contact-detail-item">
                        <div class="contact-detail-icon">
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                <circle cx="12" cy="10" r="3" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>
                        </div>
                        <div class="contact-detail-content">
                            <h4>Office</h4>
                            <p>Garuga -Entebbe Uganda</p>
                        </div>
                    </div>
                    
                    <div class="contact-detail-item">
                        <div class="contact-detail-icon">
                            <i class="fas fa-phone-alt"></i>
                            </div>


                        <div class="contact-detail-content">
                            <h4>Phone</h4>
                            <p>+4745003286</p>
                        </div>
                    </div>
                    
                    <div class="contact-detail-item">
                        <div class="contact-detail-icon">
                            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                <polyline points="22,6 12,13 2,6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>
                        </div>
                        <div class="contact-detail-content">
                            <h4>Email</h4>
                            <p>vic.ritah@hotmail.com</p>
                        </div>
                    </div>
                </div>
                
                <div class="contact-social">
                    <h4>Follow Us</h4>
                    <div class="contact-social-icons">
                        <a href="#" class="contact-social-icon"><i class="fab fa-twitter"></i></a>
                        <a href="#" class="contact-social-icon"><i class="fab fa-facebook-f"></i></a>
                        <a href="#" class="contact-social-icon"><i class="fab fa-linkedin-in"></i></a>
                        <a href="#" class="contact-social-icon"><i class="fab fa-instagram"></i></a>
                    </div>
                </div>

            </div>
            
            <!-- Contact Form -->
            <div class="contact-form-container">
                <h2 class="contact-form-title">Send Us a Message</h2>
                
                <?php if (isset($_POST['contact_inquiry_submit'])): ?>
                    <?php if (($current_time - intval($form_time)) < 60): ?>
                        <div class="safari-alert safari-alert-danger">Please wait before submitting again.</div>
                    <?php elseif (!empty($_POST['company_name'])): ?>
                        <div class="safari-alert safari-alert-danger">Spam detected.</div>
                    <?php else: ?>
                        <div class="safari-alert safari-alert-success">Thank you for your inquiry! We'll get back to you within 24 hours to plan your perfect safari adventure.</div>
                    <?php endif; ?>
                <?php endif; ?>
                
                <form method="post" class="contact-form">
                    <?php echo $honeypot; ?>
                    <input type="hidden" name="form_time" value="<?php echo time(); ?>">
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="full_name">Full Name *</label>
                            <input type="text" id="full_name" name="full_name" required>
                        </div>
                        <div class="form-group">
                            <label for="email">Email Address *</label>
                            <input type="email" id="email" name="email" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="inquiry_type">Inquiry Type *</label>
                        <select name="inquiry_type" id="inquiry_type" required>
                            <option value="">Select your inquiry type</option>
                            <option value="General Inquiry - New Booking">General Inquiry - New Booking</option>
                            <option value="General Inquiry - Existing Booking">General Inquiry - Existing Booking</option>
                            <option value="Changes to an Existing Booking">Changes to an Existing Booking</option>
                            <option value="Cancel My Booking">Cancel My Booking</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="phone">Phone Number</label>
                            <input type="text" id="phone" name="phone">
                        </div>
                        <div class="form-group">
                            <label for="address">Country/City</label>
                            <input type="text" id="address" name="address">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="message">Message *</label>
                        <textarea id="message" name="message" required></textarea>
                    </div>
                    
                    <div class="form-submit">
                        <button type="submit" name="contact_inquiry_submit" class="submit-btn">
                            Send Message
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </section>

    <!-- Map Section -->
    <section class="map-section">
        <div class="map-container">
            <h2 class="des-title">Find Our Office</h2>
            <div class="map-wrapper">
                <iframe src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d63837.061720043!2d32.48882467050082!3d0.056833393562236!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x177d88c856130287%3A0xc23858e8b9595c1!2sGaruga!5e0!3m2!1sen!2sug!4v1758020876544!5m2!1sen!2sug" 
                        width="100%" 
                        height="450" 
                        style="border:0;" 
                        allowfullscreen="" 
                        loading="lazy" 
                        referrerpolicy="no-referrer-when-downgrade"></iframe>
            </div>
        </div>
    </section>
</main>

<script>
jQuery(document).ready(function($) {
    // Form validation
    $('.contact-form').on('submit', function(e) {
        let valid = true;
        
        $(this).find('input[required], textarea[required], select[required]').each(function() {
            if ($(this).val().trim() === '') {
                valid = false;
                $(this).addClass('error');
            } else {
                $(this).removeClass('error');
            }
        });
        
        if (!valid) {
            e.preventDefault();
            alert('Please fill in all required fields.');
        }
    });
});
</script>

<?php get_footer(); ?>