<?php
/**
 * Template Name: Destination Template
 * Description: Shows destination content with tabs for overview, places & activities, popular packages, and useful info
 */

// ======================
// 1. METABOX REGISTRATION (Fixed)
// ======================
add_action('add_meta_boxes', 'register_destination_metaboxes');
add_action('add_meta_boxes', 'remove_conflicting_metaboxes', 99); // High priority to run last

function register_destination_metaboxes() {
    global $post;
    if (!$post) return;
    
    // Only register for specific post types
    $allowed_post_types = array('page', 'post', 'destination');
    if (!in_array($post->post_type, $allowed_post_types)) return;
    
    // For pages: only register if using our template
    if ($post->post_type === 'page') {
        $template = get_page_template_slug($post->ID);
        if ($template !== 'destination-template.php') return;
    }
    
    // For posts/destinations: check if they're marked as destinations
    if ($post->post_type === 'post' || $post->post_type === 'destination') {
        $is_destination = get_post_meta($post->ID, '_is_destination', true);
        if (!$is_destination && $post->post_type === 'post') return;
    }
    
    // Overview Metabox
    add_meta_box(
        'destination_overview',
        'Destination Overview',
        'render_destination_overview_metabox',
        $post->post_type, // Use current post type instead of array
        'normal',
        'high'
    );

    // Places & Activities Metabox
    add_meta_box(
        'destination_places_activities',
        'Places & Activities',
        'render_destination_places_activities_metabox',
        $post->post_type,
        'normal',
        'high'
    );

    // Popular Packages Metabox
    add_meta_box(
        'destination_popular_packages',
        'Popular Packages',
        'render_destination_popular_packages_metabox',
        $post->post_type,
        'normal',
        'high'
    );

    // Useful Information Metabox
    add_meta_box(
        'destination_useful_info',
        'Useful Information',
        'render_destination_useful_info_metabox',
        $post->post_type,
        'normal',
        'high'
    );
}

function remove_conflicting_metaboxes() {
    global $post;
    if (!$post) return;
    
    // Check if this is a destination post/page
    $is_destination = false;
    
    if ($post->post_type === 'page') {
        $template = get_page_template_slug($post->ID);
        if ($template === 'destination-template.php') {
            $is_destination = true;
        }
    } elseif ($post->post_type === 'post' || $post->post_type === 'destination') {
        $is_destination = get_post_meta($post->ID, '_is_destination', true);
        if ($post->post_type === 'destination') $is_destination = true;
    }
    
    if (!$is_destination) return;
    
    // List of metabox IDs to remove
    $metaboxes_to_remove = array(
        'tripdetailsdiv',       // Example from a travel plugin
        'itinerarydiv',         // Example from a travel plugin
        'postcustom',           // Default custom fields metabox
        'commentstatusdiv',    // Comments status metabox
        'commentsdiv',         // Comments metabox
        // Add any other conflicting metabox IDs here
    );
    
    foreach ($metaboxes_to_remove as $metabox_id) {
        remove_meta_box($metabox_id, $post->post_type, 'normal');
        remove_meta_box($metabox_id, $post->post_type, 'side');
        remove_meta_box($metabox_id, $post->post_type, 'advanced');
    }
}

// Add a checkbox to mark regular posts as destinations
add_action('post_submitbox_misc_actions', 'add_destination_checkbox');
function add_destination_checkbox() {
    global $post;
    if ($post->post_type !== 'post') return;
    
    $is_destination = get_post_meta($post->ID, '_is_destination', true);
    ?>
    <div class="misc-pub-section">
        <label>
            <input type="checkbox" name="is_destination" value="1" <?php checked($is_destination, 1); ?> />
            This is a Destination post
        </label>
    </div>
    <?php
}

// =================
// 2. METABOX RENDER FUNCTIONS
// =================
function render_destination_overview_metabox($post) {
    wp_nonce_field('save_destination_data', 'destination_nonce');
    $map_image_id = get_post_meta($post->ID, '_destination_map_image', true);
    $overview = get_post_meta($post->ID, '_destination_overview', true);
    ?>
    <div class="destination-metabox">
        <div class="map-uploader">
            <label><strong>Map Image:</strong></label>
            <div class="image-preview">
                <?php if ($map_image_id) echo wp_get_attachment_image($map_image_id, 'medium'); ?>
            </div>
            <input type="hidden" name="destination_map_image" class="image-id" value="<?php echo esc_attr($map_image_id); ?>">
            <button type="button" class="button upload-image">Upload Map</button>
            <button type="button" class="button remove-image" style="<?php echo !$map_image_id ? 'display:none' : ''; ?>">Remove</button>
        </div>
        
        <label style="margin-top:20px; display:block;"><strong>Overview Content:</strong></label>
        <?php wp_editor($overview, 'destination_overview', array(
            'textarea_name' => 'destination_overview',
            'media_buttons' => false,
            'textarea_rows' => 10
        )); ?>
    </div>
    <?php
}

function render_destination_places_activities_metabox($post) {
    $items = get_post_meta($post->ID, '_destination_places_activities', true);
    $items = is_array($items) ? $items : array();
    ?>
    <div class="destination-metabox">
        <div id="places-activities-container">
            <?php foreach ($items as $index => $item): ?>
                <div class="item" style="border:1px solid #ddd; padding:15px; margin-bottom:15px;">
                    <p>
                        <label>Title:</label>
                        <input type="text" name="places_activities[<?php echo $index; ?>][title]" 
                               value="<?php echo esc_attr($item['title']); ?>" style="width:100%;">
                    </p>
                    <p>
                        <label>Description:</label>
                        <textarea name="places_activities[<?php echo $index; ?>][description]" 
                                  style="width:100%; height:100px;"><?php echo esc_textarea($item['description']); ?></textarea>
                    </p>
                    <div class="image-upload">
                        <label>Image:</label>
                        <div class="image-preview">
                            <?php if (!empty($item['image_id'])) echo wp_get_attachment_image($item['image_id'], 'medium'); ?>
                        </div>
                        <input type="hidden" name="places_activities[<?php echo $index; ?>][image_id]" 
                               class="image-id" value="<?php echo esc_attr($item['image_id']); ?>">
                        <button type="button" class="button upload-item-image">Select Image</button>
                        <button type="button" class="button remove-item">Remove Item</button>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        <button type="button" id="add-places-activity" class="button">+ Add New</button>
    </div>
    <?php
}

function render_destination_popular_packages_metabox($post) {
    $packages = get_post_meta($post->ID, '_destination_popular_packages', true);
    $packages = is_array($packages) ? $packages : array();
    ?>
    <div class="destination-metabox">
        <div id="popular-packages-container">
            <?php foreach ($packages as $index => $pkg): ?>
                <div class="item" style="border:1px solid #ddd; padding:15px; margin-bottom:15px;">
                    <p>
                        <label>Package Title:</label>
                        <input type="text" name="popular_packages[<?php echo $index; ?>][title]" 
                               value="<?php echo esc_attr($pkg['title']); ?>" style="width:100%;">
                    </p>
                    <p>
                        <label>Description:</label>
                        <textarea name="popular_packages[<?php echo $index; ?>][description]" 
                                  style="width:100%; height:100px;"><?php echo esc_textarea($pkg['description']); ?></textarea>
                    </p>
                    <p>
                        <label>Price:</label>
                        <input type="text" name="popular_packages[<?php echo $index; ?>][price]" 
                               value="<?php echo esc_attr($pkg['price']); ?>" style="width:100%;">
                    </p>
                    <p>
                        <label>Link URL:</label>
                        <input type="url" name="popular_packages[<?php echo $index; ?>][link]" 
                               value="<?php echo esc_url($pkg['link']); ?>" style="width:100%;">
                    </p>
                    <button type="button" class="button remove-package">Remove Package</button>
                </div>
            <?php endforeach; ?>
        </div>
        <button type="button" id="add-popular-package" class="button">+ Add Package</button>
    </div>
    <?php
}

function render_destination_useful_info_metabox($post) {
    $infos = get_post_meta($post->ID, '_destination_useful_info', true);
    $infos = is_array($infos) ? $infos : array();
    ?>
    <div class="destination-metabox">
        <div id="useful-info-container">
            <?php foreach ($infos as $index => $info): ?>
                <div class="item" style="border:1px solid #ddd; padding:15px; margin-bottom:15px;">
                    <p>
                        <label>Title:</label>
                        <input type="text" name="useful_info[<?php echo $index; ?>][title]" 
                               value="<?php echo esc_attr($info['title']); ?>" style="width:100%;">
                    </p>
                    <p>
                        <label>Content:</label>
                        <?php wp_editor(
                            $info['content'] ?? '',
                            'useful_info_'.$index,
                            array(
                                'textarea_name' => "useful_info[{$index}][content]",
                                'media_buttons' => false,
                                'textarea_rows' => 6
                            )
                        ); ?>
                    </p>
                    <button type="button" class="button remove-info">Remove Item</button>
                </div>
            <?php endforeach; ?>
        </div>
        <button type="button" id="add-useful-info" class="button">+ Add Information</button>
    </div>
    <?php
}

// =================
// 3. SAVE METABOXES (Fixed)
// =================
add_action('save_post', 'save_destination_metaboxes');
add_action('save_post', 'save_destination_checkbox');

function save_destination_metaboxes($post_id) {
    // Security checks
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!isset($_POST['destination_nonce']) || !wp_verify_nonce($_POST['destination_nonce'], 'save_destination_data')) return;
    if (!current_user_can('edit_post', $post_id)) return;

    // Save Overview
    if (isset($_POST['destination_overview'])) {
        update_post_meta($post_id, '_destination_overview', wp_kses_post($_POST['destination_overview']));
    }
    
    if (isset($_POST['destination_map_image'])) {
        update_post_meta($post_id, '_destination_map_image', absint($_POST['destination_map_image']));
    }
    
    // Save Places & Activities
    if (isset($_POST['places_activities'])) {
        $clean_items = array();
        foreach ($_POST['places_activities'] as $item) {
            if (!empty($item['title'])) {
                $clean_items[] = array(
                    'title' => sanitize_text_field($item['title']),
                    'description' => wp_kses_post($item['description']),
                    'image_id' => absint($item['image_id'])
                );
            }
        }
        update_post_meta($post_id, '_destination_places_activities', $clean_items);
    } else {
        delete_post_meta($post_id, '_destination_places_activities');
    }
    
    // Save Popular Packages
    if (isset($_POST['popular_packages'])) {
        $clean_packages = array();
        foreach ($_POST['popular_packages'] as $pkg) {
            if (!empty($pkg['title'])) {
                $clean_packages[] = array(
                    'title' => sanitize_text_field($pkg['title']),
                    'description' => wp_kses_post($pkg['description']),
                    'price' => sanitize_text_field($pkg['price']),
                    'link' => esc_url_raw($pkg['link'])
                );
            }
        }
        update_post_meta($post_id, '_destination_popular_packages', $clean_packages);
    } else {
        delete_post_meta($post_id, '_destination_popular_packages');
    }
    
    // Save Useful Information
    if (isset($_POST['useful_info'])) {
        $clean_infos = array();
        foreach ($_POST['useful_info'] as $info) {
            if (!empty($info['title'])) {
                $clean_infos[] = array(
                    'title' => sanitize_text_field($info['title']),
                    'content' => wp_kses_post($info['content'])
                );
            }
        }
        update_post_meta($post_id, '_destination_useful_info', $clean_infos);
    } else {
        delete_post_meta($post_id, '_destination_useful_info');
    }
}

function save_destination_checkbox($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;
    
    if (isset($_POST['is_destination'])) {
        update_post_meta($post_id, '_is_destination', 1);
    } else {
        delete_post_meta($post_id, '_is_destination');
    }
}

// =================
// 4. ADMIN SCRIPTS AND STYLES (Fixed)
// =================
add_action('admin_enqueue_scripts', 'enqueue_destination_admin_assets');

function enqueue_destination_admin_assets($hook) {
    // Only load on post edit pages
    if ($hook != 'post.php' && $hook != 'post-new.php') return;
    
    global $post;
    if (!$post) return;
    
    // Check if this post should have destination metaboxes
    $should_load = false;
    
    if ($post->post_type === 'page') {
        $template = get_page_template_slug($post->ID);
        if ($template === 'destination-template.php') {
            $should_load = true;
        }
    } elseif ($post->post_type === 'post') {
        $should_load = get_post_meta($post->ID, '_is_destination', true);
    } elseif ($post->post_type === 'destination') {
        $should_load = true;
    }
    
    if (!$should_load) return;
    
    // Enqueue WordPress media scripts
    wp_enqueue_media();
    
    // Add inline script for metabox functionality
    wp_add_inline_script('jquery', '
    jQuery(document).ready(function($) {
        // Map Image Upload
        $(document).on("click", ".upload-image", function(e) {
            e.preventDefault();
            var button = $(this);
            var frame = wp.media({
                title: "Select Map Image",
                button: { text: "Use this image" },
                multiple: false
            });
            frame.on("select", function() {
                var attachment = frame.state().get("selection").first().toJSON();
                button.siblings(".image-id").val(attachment.id);
                button.siblings(".image-preview").html("<img src=\"" + attachment.url + "\" style=\"max-width:100%;\"/>");
                button.siblings(".remove-image").show();
            });
            frame.open();
        });
        
        $(document).on("click", ".remove-image", function() {
            $(this).siblings(".image-id").val("");
            $(this).siblings(".image-preview").html("");
            $(this).hide();
        });

        // Places & Activities
        $(document).on("click", "#add-places-activity", function() {
            var index = Date.now();
            var html = `
            <div class="item" style="border:1px solid #ddd; padding:15px; margin-bottom:15px;">
                <p>
                    <label>Title:</label>
                    <input type="text" name="places_activities[${index}][title]" style="width:100%;">
                </p>
                <p>
                    <label>Description:</label>
                    <textarea name="places_activities[${index}][description]" style="width:100%; height:100px;"></textarea>
                </p>
                <div class="image-upload">
                    <label>Image:</label>
                    <div class="image-preview"></div>
                    <input type="hidden" name="places_activities[${index}][image_id]" class="image-id">
                    <button type="button" class="button upload-item-image">Select Image</button>
                    <button type="button" class="button remove-item">Remove Item</button>
                </div>
            </div>`;
            $("#places-activities-container").append(html);
        });

        $(document).on("click", ".upload-item-image", function() {
            var button = $(this);
            var frame = wp.media({
                title: "Select Image",
                button: { text: "Use this image" },
                multiple: false
            });
            frame.on("select", function() {
                var attachment = frame.state().get("selection").first().toJSON();
                button.siblings(".image-id").val(attachment.id);
                button.siblings(".image-preview").html("<img src=\"" + attachment.url + "\" style=\"max-width:100%;\"/>");
            });
            frame.open();
        });

        $(document).on("click", ".remove-item", function() {
            $(this).closest(".item").remove();
        });

        // Popular Packages
        $(document).on("click", "#add-popular-package", function() {
            var index = Date.now();
            var html = `
            <div class="item" style="border:1px solid #ddd; padding:15px; margin-bottom:15px;">
                <p>
                    <label>Package Title:</label>
                    <input type="text" name="popular_packages[${index}][title]" style="width:100%;">
                </p>
                <p>
                    <label>Description:</label>
                    <textarea name="popular_packages[${index}][description]" style="width:100%; height:100px;"></textarea>
                </p>
                <p>
                    <label>Price:</label>
                    <input type="text" name="popular_packages[${index}][price]" style="width:100%;">
                </p>
                <p>
                    <label>Link URL:</label>
                    <input type="url" name="popular_packages[${index}][link]" style="width:100%;">
                </p>
                <button type="button" class="button remove-package">Remove Package</button>
            </div>`;
            $("#popular-packages-container").append(html);
        });

        $(document).on("click", ".remove-package", function() {
            $(this).closest(".item").remove();
        });

        // Useful Information
        $(document).on("click", "#add-useful-info", function() {
            var index = Date.now();
            var html = `
            <div class="item" style="border:1px solid #ddd; padding:15px; margin-bottom:15px;">
                <p>
                    <label>Title:</label>
                    <input type="text" name="useful_info[${index}][title]" style="width:100%;">
                </p>
                <p>
                    <label>Content:</label>
                    <textarea name="useful_info[${index}][content]" style="width:100%; height:120px;"></textarea>
                </p>
                <button type="button" class="button remove-info">Remove Item</button>
            </div>`;
            $("#useful-info-container").append(html);
        });

        $(document).on("click", ".remove-info", function() {
            $(this).closest(".item").remove();
        });
    });
    ');
    
    // Add some basic styling for metaboxes
    wp_add_inline_style('wp-admin', '
    .destination-metabox .image-preview img {
        max-width: 200px;
        height: auto;
        margin: 10px 0;
        border: 1px solid #ddd;
    }
    .destination-metabox .item {
        background: #f9f9f9;
    }
    .destination-metabox .item:nth-child(even) {
        background: #fff;
    }
    ');
}

// ======================
// 5. TEMPLATE DISPLAY (Keep existing frontend code)
// ======================

get_header();
include get_template_directory() . '/inc/menus/menu.php';

$current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview';
$post_id = get_the_ID();
?>

<!-- Frontend CSS and HTML remains the same as your original -->
<style>
/* MAIN CONTAINER */
.destination-template {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    font-family: 'Raleway', sans-serif;
    color: #333;
    line-height: 1.6;
}

/* TABS NAVIGATION */
.destination-tabs {
    margin-bottom: 30px;
    background: #f8f9fa;
    border-radius: 8px;
    padding: 10px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.05);
}

.destination-tabs ul {
    display: flex;
    list-style: none;
    padding: 0;
    margin: 0;
    flex-wrap: wrap;
}

.destination-tabs li {
    margin-right: 5px;
    margin-bottom: 5px;
}

.destination-tabs a {
    display: block;
    padding: 10px 20px;
    text-decoration: none;
    color: #495057;
    font-weight: 600;
    border-radius: 4px;
    transition: all 0.3s;
    background: #e9ecef;
}

.destination-tabs a:hover {
    background: #dee2e6;
    color: #212529;
}

.destination-tabs a.active {
    background: #1e73be;
    color: white;
}

/* CONTENT SECTIONS */
.destination-section {
    margin-bottom: 40px;
    padding: 25px;
    background: white;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.section-title {
    margin-top: 0;
    color: #1e73be;
    border-bottom: 2px solid #e9ecef;
    padding-bottom: 10px;
}

/* MAP SECTION */
.map-container {
    margin-bottom: 25px;
    text-align: center;
}

.map-container img {
    max-width: 100%;
    height: auto;
    border-radius: 6px;
    box-shadow: 0 3px 10px rgba(0,0,0,0.1);
}

/* PLACES & ACTIVITIES */
.places-activities-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    gap: 25px;
    margin-top: 20px;
}

.places-activity-card {
    border: 1px solid #e9ecef;
    border-radius: 8px;
    overflow: hidden;
    transition: transform 0.3s, box-shadow 0.3s;
}

.places-activity-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}

.places-activity-image {
    height: 200px;
    overflow: hidden;
}

.places-activity-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.5s;
}

.places-activity-card:hover .places-activity-image img {
    transform: scale(1.05);
}

.places-activity-content {
    padding: 20px;
}

.places-activity-content h3 {
    margin-top: 0;
    margin-bottom: 10px;
    color: #212529;
}

/* POPULAR PACKAGES */
.packages-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 25px;
    margin-top: 20px;
}

.package-card {
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 25px;
    transition: all 0.3s;
}

.package-card:hover {
    border-color: #1e73be;
    box-shadow: 0 5px 15px rgba(30, 115, 190, 0.1);
}

.package-title {
    margin-top: 0;
    color: #212529;
}

.package-price {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1e73be;
    margin: 15px 0;
}

.package-btn {
    display: inline-block;
    padding: 10px 20px;
    background: #1e73be;
    color: white;
    text-decoration: none;
    border-radius: 4px;
    font-weight: 600;
    transition: background 0.3s;
}

.package-btn:hover {
    background: #155a9e;
    color: white;
}

/* USEFUL INFORMATION */
.useful-info-accordion {
    margin-top: 20px;
}

.info-item {
    border: 1px solid #e9ecef;
    border-radius: 6px;
    margin-bottom: 15px;
    overflow: hidden;
}

.info-title {
    margin: 0;
    padding: 15px 20px;
    background: #f8f9fa;
    cursor: pointer;
    position: relative;
    font-size: 1.1rem;
}

.info-title:after {
    content: '+';
    position: absolute;
    right: 20px;
    top: 50%;
    transform: translateY(-50%);
    font-size: 1.2rem;
}

.info-item.active .info-title:after {
    content: '-';
}

.info-content {
    padding: 0 20px;
    max-height: 0;
    overflow: hidden;
    transition: all 0.3s;
}

.info-item.active .info-content {
    padding: 20px;
    max-height: 1000px;
}

/* RESPONSIVE */
@media (max-width: 768px) {
    .destination-tabs ul {
        flex-direction: column;
    }
    
    .destination-tabs li {
        margin-right: 0;
        margin-bottom: 5px;
    }
    
    .destination-tabs a {
        width: 100%;
    }
    
    .places-activities-grid,
    .packages-grid {
        grid-template-columns: 1fr;
    }
    
    .destination-section {
        padding: 15px;
    }
}
</style>

<!-- HTML TEMPLATE -->
<div class="destination-template">
    <!-- Tab Navigation -->
    <nav class="destination-tabs">
        <ul>
            <li><a href="?tab=overview" class="<?php echo $current_tab === 'overview' ? 'active' : ''; ?>">Overview</a></li>
            <li><a href="?tab=places-activities" class="<?php echo $current_tab === 'places-activities' ? 'active' : ''; ?>">Places & Activities</a></li>
            <li><a href="?tab=popular-packages" class="<?php echo $current_tab === 'popular-packages' ? 'active' : ''; ?>">Popular Packages</a></li>
            <li><a href="?tab=useful-info" class="<?php echo $current_tab === 'useful-info' ? 'active' : ''; ?>">Useful Information</a></li>
        </ul>
    </nav>

    <!-- Tab Content -->
    <div class="destination-content">
        <?php switch($current_tab):
            case 'overview': ?>
                <section class="destination-section">
                    <h2 class="section-title">Overview</h2>
                    <?php 
                    $map_image_id = get_post_meta($post_id, '_destination_map_image', true);
                    $overview = get_post_meta($post_id, '_destination_overview', true);
                    
                    if ($map_image_id): ?>
                        <div class="map-container">
                            <?php echo wp_get_attachment_image($map_image_id, 'large'); ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="overview-content">
                        <?php echo wpautop($overview); ?>
                    </div>
                </section>
                <?php break;
                
            case 'places-activities': ?>
                <section class="destination-section">
                    <h2 class="section-title">Places & Activities</h2>
                    <?php 
                    $items = get_post_meta($post_id, '_destination_places_activities', true);
                    
                    if (!empty($items)): ?>
                        <div class="places-activities-grid">
                            <?php foreach ($items as $item): ?>
                                <div class="places-activity-card">
                                    <?php if (!empty($item['image_id'])): ?>
                                        <div class="places-activity-image">
                                            <?php echo wp_get_attachment_image($item['image_id'], 'large'); ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="places-activity-content">
                                        <h3><?php echo esc_html($item['title']); ?></h3>
                                        <p><?php echo wpautop(esc_html($item['description'])); ?></p>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p>No places & activities added yet.</p>
                    <?php endif; ?>
                </section>
                <?php break;
                
            case 'popular-packages': ?>
                <section class="destination-section">
                    <h2 class="section-title">Popular Packages</h2>
                    <?php 
                    $packages = get_post_meta($post_id, '_destination_popular_packages', true);
                    
                    if (!empty($packages)): ?>
                        <div class="packages-grid">
                            <?php foreach ($packages as $pkg): ?>
                                <div class="package-card">
                                    <h3 class="package-title"><?php echo esc_html($pkg['title']); ?></h3>
                                    <?php if (!empty($pkg['description'])): ?>
                                        <div class="package-description">
                                            <?php echo wpautop(esc_html($pkg['description'])); ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($pkg['price'])): ?>
                                        <div class="package-price"><?php echo esc_html($pkg['price']); ?></div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($pkg['link'])): ?>
                                        <a href="<?php echo esc_url($pkg['link']); ?>" class="package-btn" target="_blank">View Package</a>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p>No popular packages added yet.</p>
                    <?php endif; ?>
                </section>
                <?php break;
                
            case 'useful-info': ?>
                <section class="destination-section">
                    <h2 class="section-title">Useful Information</h2>
                    <?php 
                    $infos = get_post_meta($post_id, '_destination_useful_info', true);
                    
                    if (!empty($infos)): ?>
                        <div class="useful-info-accordion">
                            <?php foreach ($infos as $info): ?>
                                <div class="info-item">
                                    <h3 class="info-title"><?php echo esc_html($info['title']); ?></h3>
                                    <div class="info-content">
                                        <?php echo wpautop($info['content']); ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <script>
                        jQuery(document).ready(function($) {
                            $('.info-title').click(function() {
                                $(this).parent().toggleClass('active');
                            });
                        });
                        </script>
                    <?php else: ?>
                        <p>No useful information added yet.</p>
                    <?php endif; ?>
                </section>
                <?php break;
                
        endswitch; ?>
    </div>
</div>

<?php get_footer(); ?>