<?php
/**
 * Template Name: Single Tour Package
 * Template Post Type: tour_package
 */

get_header(); 
include get_template_directory() . '/inc/menus/menu.php'; 

function generate_sequential_booking_number() {
    // Get the last booking number from database
    $last_booking = get_posts(array(
        'post_type' => 'booking',
        'posts_per_page' => 1,
        'orderby' => 'date',
        'order' => 'DESC'
    ));
    
    if (empty($last_booking)) {
        // First booking
        $sequence = 1;
    } else {
        $last_booking_number = get_post_meta($last_booking[0]->ID, '_booking_number', true);
        // Extract sequence number from last booking
        preg_match('/BOOK-\d+-(\d+)/', $last_booking_number, $matches);
        $sequence = isset($matches[1]) ? intval($matches[1]) + 1 : 1;
    }
    
    return 'BOOK-' . date('Ymd') . '-' . str_pad($sequence, 4, '0', STR_PAD_LEFT);
}


function render_inclusions_exclusions($items, $default_items = []) {
    if (!empty($items) && is_array($items)) {
        echo '<ul class="inclusions-exclusions-list">';
        foreach ($items as $item) {
            if (!empty($item['item'])) {
                echo '<li>' . esc_html($item['item']) . '</li>';
            }
        }
        echo '</ul>';
    } elseif (!empty($default_items)) {
        echo '<ul class="inclusions-exclusions-list">';
        foreach ($default_items as $item) {
            echo '<li>' . esc_html($item) . '</li>';
        }
        echo '</ul>';
    }
}

// -------------------
// Fetch post meta
// -------------------
$post_id = get_the_ID();
$duration       = get_post_meta($post_id, '_package_duration', true);
$destination    = get_post_meta($post_id, '_destination', true);
$price_adult    = get_post_meta($post_id, '_package_price_adult', true);
$price_child    = get_post_meta($post_id, '_package_price_child', true);
$price_ugandan  = get_post_meta($post_id, '_package_price_ugandan', true);
$discount       = get_post_meta($post_id, '_package_discount', true);
$itinerary      = get_post_meta($post_id, '_package_itinerary', true);
$inclusions     = get_post_meta($post_id, '_package_inclusions', true);
$exclusions     = get_post_meta($post_id, '_package_exclusions', true);
$contact_email  = get_post_meta($post_id, '_contact_email', true);
$contact_phone  = get_post_meta($post_id, '_contact_phone', true);
$bank_name      = get_post_meta($post_id, '_bank_name', true);
$account_name   = get_post_meta($post_id, '_account_name', true);
$account_number = get_post_meta($post_id, '_account_number', true);
$swift_code     = get_post_meta($post_id, '_swift_code', true);
$address        = get_post_meta($post_id, '_address', true);
$featured_image = get_the_post_thumbnail_url($post_id, 'full');

// Set host email
$host_email = 'info@exploreug.com';

// Process booking form
$booking_success = false;
$booking_error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['booking_submit'])) {
    // Verify nonce for security
    if (!wp_verify_nonce($_POST['booking_nonce'], 'booking_form')) {
        $booking_error = 'Security check failed. Please try again.';
    } else {
        // Sanitize and validate input
        $full_name = sanitize_text_field($_POST['full_name']);
        $email = sanitize_email($_POST['email']);
        $phone = sanitize_text_field($_POST['phone']);
        $tour_package = sanitize_text_field($_POST['tour_package']);
        $num_adults = intval($_POST['num_adults']);
        $num_children = intval($_POST['num_children']);
        $num_ugandans = intval($_POST['num_ugandans']);
        $tour_date = sanitize_text_field($_POST['tour_date']);
        
        // Validate required fields
        if (empty($full_name) || empty($email) || empty($phone) || empty($tour_date)) {
            $booking_error = 'Please fill in all required fields.';
        } else {
            // Calculate total cost based on prices
            $total_cost = 0;
            $price_details = [];
            
            // Calculate costs for each category
            if (!empty($price_adult) && $num_adults > 0) {
                $adult_price = floatval(str_replace(['$', ','], '', $price_adult));
                $total_cost += $num_adults * $adult_price;
                $price_details[] = "International Adults: $num_adults x $$price_adult";
            }
            
            if (!empty($price_ugandan) && $num_ugandans > 0) {
                $ugandan_price = floatval(str_replace(['$', ','], '', $price_ugandan));
                $total_cost += $num_ugandans * $ugandan_price;
                $price_details[] = "Ugandan Residents: $num_ugandans x $$price_ugandan";
            }
            
            // Add children cost if available
            if (!empty($price_child) && $num_children > 0) {
                $child_price = floatval(str_replace(['$', ','], '', $price_child));
                $total_cost += $num_children * $child_price;
                $price_details[] = "Children: $num_children x $$price_child";
            }
            
            // Apply discount if available
            $original_cost = $total_cost;
            if (!empty($discount) && $total_cost > 0) {
                $discount_amount = $total_cost * ($discount / 100);
                $total_cost = $total_cost - $discount_amount;
            }

            // Generate unique booking number
            $booking_number = generate_sequential_booking_number();

            // Create booking as custom post type
            $booking_post = array(
                'post_title'    => $booking_number . ' - ' . $full_name,
                'post_content'  => '',
                'post_status'   => 'publish',
                'post_type'     => 'booking',
                'meta_input'    => array(
                    '_booking_number' => $booking_number,
                    '_customer_name' => $full_name,
                    '_customer_email' => $email,
                    '_customer_phone' => $phone,
                    '_tour_package' => $tour_package,
                    '_num_adults' => $num_adults,
                    '_num_children' => $num_children,
                    '_num_ugandans' => $num_ugandans,
                    '_tour_date' => $tour_date,
                    '_total_cost' => $total_cost,
                    '_original_cost' => $original_cost,
                    '_discount_amount' => isset($discount_amount) ? $discount_amount : 0,
                    '_booking_status' => 'pending',
                    '_deposit_amount' => $total_cost * 0.5,
                    '_balance_amount' => $total_cost * 0.5
                )
            );

            // Insert the booking post
            $booking_id = wp_insert_post($booking_post);

            if ($booking_id && !is_wp_error($booking_id)) {
                // Prepare price breakdown for email
                $price_breakdown = implode("\n", $price_details);
                if (!empty($discount) && $original_cost > 0) {
                    $price_breakdown .= "\nDiscount: $discount%";
                    $price_breakdown .= "\nOriginal Total: $" . number_format($original_cost, 2);
                }
                $price_breakdown .= "\nFinal Total: $" . number_format($total_cost, 2);
                
                // Calculate payment amounts
                $deposit_amount = $total_cost * 0.5;
                $balance_amount = $total_cost - $deposit_amount;

                $customer_subject = "Booking Confirmation - $booking_number";
                $customer_message = "
                <!DOCTYPE html>
                <html>
                <head>
                  <meta charset='UTF-8'>
                  <title>Booking Confirmation</title>
                  <style>
                    body {
                      margin: 0;
                      padding: 0;
                      background: #f8f8f8;
                      font-family: 'Arial', sans-serif;
                    }
                    table {
                      border-collapse: collapse;
                      width: 100%;
                    }
                    .container {
                      max-width: 600px;
                      margin: 0 auto;
                      background: #ffffff;
                      border-radius: 10px;
                      overflow: hidden;
                      box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                    }
                    .header {
                      background: linear-gradient(135deg, #710b0d, #8a0f12);
                      color: #ffffff;
                      text-align: center;
                      padding: 40px 20px;
                    }
                    .header h1 {
                      font-size: 24px;
                      margin: 0;
                    }
                    .booking-number {
                      font-size: 18px;
                      color: #f7d508;
                      margin-top: 8px;
                      font-weight: 400;
                    }
                    .content {
                      padding: 30px;
                      color: #333333;
                    }
                    .section-title {
                      color: #710b0d;
                      border-bottom: 2px solid #f7d508;
                      padding-bottom: 5px;
                      margin-bottom: 15px;
                      font-size: 18px;
                    }
                    .details-table {
                      width: 100%;
                      border: 1px solid #eee;
                      border-radius: 6px;
                      margin-bottom: 25px;
                    }
                    .details-table td {
                      padding: 10px;
                      border-bottom: 1px solid #f1f1f1;
                    }
                    .details-table td.label {
                      font-weight: bold;
                      color: #710b0d;
                      width: 40%;
                    }
                    .highlight {
                      background: #fff3cd;
                      padding: 3px 8px;
                      border-radius: 4px;
                      color: #710b0d;
                      font-weight: 400;
                    }
                    .footer {
                      background: #710b0d;
                      color: #fff;
                      text-align: center;
                      padding: 25px 20px;
                      font-size: 14px;
                    }
                
                    /* MOBILE STYLES */
                    @media only screen and (max-width: 600px) {
                      .container {
                        width: 100% !important;
                        border-radius: 0;
                      }
                      .content {
                        padding: 20px !important;
                      }
                      .header h1 {
                        font-size: 20px !important;
                      }
                      .booking-number {
                        font-size: 16px !important;
                      }
                      .details-table td {
                        display: block;
                        width: 100% !important;
                      }
                      .details-table td.label {
                        font-size: 14px;
                        color: #710b0d;
                      }
                      .section-title {
                        font-size: 16px;
                      }
                    }
                  </style>
                </head>
                <body>
                  <table class='container'>
                    <tr>
                      <td class='header'>
                        <h1>Booking Confirmed</h1>
                        <div class='booking-number'>Booking #$booking_number</div>
                        <p>Thank you for choosing Explore Uganda.</p>
                      </td>
                    </tr>
                    <tr>
                      <td class='content'>
                        <p>Dear <strong>$full_name</strong>,</p>
                        <p>Your booking has been successfully received. Below are your details:</p>
                
                        <div class='section-title'>Booking Summary</div>
                        <table class='details-table'>
                          <tr><td class='label'>Tour Package</td><td>$tour_package</td></tr>
                          <tr><td class='label'>Tour Date</td><td>$tour_date</td></tr>
                          <tr><td class='label'>International Adults</td><td>$num_adults</td></tr>
                          " . ($num_ugandans > 0 ? "<tr><td class='label'>Ugandan Residents</td><td>$num_ugandans</td></tr>" : "") . "
                          " . ($num_children > 0 ? "<tr><td class='label'>Children</td><td>$num_children</td></tr>" : "") . "
                        </table>
                
                        <div class='section-title'>Payment Summary</div>
                        <table class='details-table'>
                          <tr><td class='label'>Deposit (50%) and above</td><td><span class='highlight'>$" . number_format($deposit_amount, 2) . "</span></td></tr>
                          <tr><td class='label'>Balance Due</td><td><span class='highlight'>$" . number_format($balance_amount, 2) . "</span></td></tr>
                          <tr><td class='label'>Tour Date</td><td>$tour_date</td></tr>
                        </table>
                
                        <div class='section-title'>Bank Details</div>
                        <table class='details-table'>
                          <tr><td class='label'>Bank</td><td>$bank_name</td></tr>
                          <tr><td class='label'>Account Name</td><td>$account_name</td></tr>
                          <tr><td class='label'>Account Number</td><td>$account_number</td></tr>
                          " . (!empty($swift_code) ? "<tr><td class='label'>SWIFT Code</td><td>$swift_code</td></tr>" : "") . "
                        </table>
                
                        <p><strong>Amount to Pay Now is 50% and Above:</strong> <span class='highlight'>$" . number_format($deposit_amount, 2) . "</span></p>
                        <p><strong>Payment Reference:</strong> <span class='highlight'>$booking_number</span></p>
                
                        <div class='section-title'>Next Steps</div>
                        <ol>
                          <li>Make the 50% and above deposit payment using the bank details above.</li>
                          <li>Email your payment receipt to <strong>$contact_email</strong>.</li>
                          <li>We’ll confirm your booking within 24 hours after payment verification.</li>
                        </ol>
                
                        <p>If you have any questions, contact us at:<br>
                        <strong>Email:</strong> <a href='mailto:$contact_email' style='color:#710b0d;'>$contact_email</a><br>
                        <strong>Phone:</strong> <a href='tel:$contact_phone' style='color:#710b0d;'>$contact_phone</a></p>
                      </td>
                    </tr>
                    <tr>
                      <td class='footer'>
                        <p>Thank you for trusting Explore Uganda.</p>
                        <p><em>Warm regards,<br>The Explore Uganda Team</em></p>
                      </td>
                    </tr>
                  </table>
                </body>
                </html>
                ";


                $admin_subject = "New Booking - $booking_number";
                $admin_message = "
                <!DOCTYPE html>
                <html>
                <head>
                  <meta charset='UTF-8'>
                  <title>New Booking Notification</title>
                  <style>
                    body {
                      margin: 0;
                      padding: 0;
                      background: #f8f8f8;
                      font-family: 'Arial', sans-serif;
                    }
                    table {
                      border-collapse: collapse;
                      width: 100%;
                    }
                    .container {
                      max-width: 600px;
                      margin: 0 auto;
                      background: #ffffff;
                      border-radius: 10px;
                      overflow: hidden;
                      box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                    }
                    .header {
                      background: linear-gradient(135deg, #710b0d, #8a0f12);
                      color: #ffffff;
                      text-align: center;
                      padding: 35px 20px;
                    }
                    .header h1 {
                      font-size: 22px;
                      margin: 0;
                    }
                    .content {
                      padding: 30px;
                      color: #333333;
                    }
                    .section-title {
                      color: #710b0d;
                      border-bottom: 2px solid #f7d508;
                      padding-bottom: 5px;
                      margin-bottom: 15px;
                      font-size: 18px;
                    }
                    .details-table {
                      width: 100%;
                      border: 1px solid #eee;
                      border-radius: 6px;
                      margin-bottom: 25px;
                    }
                    .details-table td {
                      padding: 10px;
                      border-bottom: 1px solid #f1f1f1;
                    }
                    .details-table td.label {
                      font-weight: 400;
                      color: #710b0d;
                      width: 40%;
                    }
                    .footer {
                      background: #710b0d;
                      color: #fff;
                      text-align: center;
                      padding: 25px 20px;
                      font-size: 14px;
                    }
                    .alert {
                      background: #fff3cd;
                      border: 1px solid #ffeaa7;
                      padding: 10px;
                      border-radius: 5px;
                      margin-bottom: 20px;
                      color: #856404;
                    }
                
                    /* MOBILE STYLES */
                    @media only screen and (max-width: 600px) {
                      .container {
                        width: 100% !important;
                        border-radius: 0;
                      }
                      .content {
                        padding: 20px !important;
                      }
                      .header h1 {
                        font-size: 18px !important;
                      }
                      .details-table td {
                        display: block;
                        width: 100% !important;
                      }
                      .details-table td.label {
                        font-size: 14px;
                      }
                      .section-title {
                        font-size: 16px;
                      }
                    }
                  </style>
                </head>
                <body>
                  <table class='container'>
                    <tr>
                      <td class='header'>
                        <h1>New Booking Received</h1>
                        <p>Booking #$booking_number</p>
                      </td>
                    </tr>
                    <tr>
                      <td class='content'>
                        <div class='alert'>
                          <strong>Action Required:</strong> Monitor for payment confirmation and contact the customer.
                        </div>
                
                        <div class='section-title'>Customer Information</div>
                        <table class='details-table'>
                          <tr><td class='label'>Name</td><td>$full_name</td></tr>
                          <tr><td class='label'>Email</td><td><a href='mailto:$email'>$email</a></td></tr>
                          <tr><td class='label'>Phone</td><td><a href='tel:$phone'>$phone</a></td></tr>
                        </table>
                
                        <div class='section-title'>Tour Details</div>
                        <table class='details-table'>
                          <tr><td class='label'>Package</td><td>$tour_package</td></tr>
                          <tr><td class='label'>Date</td><td>$tour_date</td></tr>
                          <tr><td class='label'>Travelers</td>
                              <td>$num_adults International Adults" .
                              ($num_ugandans > 0 ? ", $num_ugandans Ugandan Residents" : "") .
                              ($num_children > 0 ? ", $num_children Children" : "") .
                              "</td></tr>
                        </table>
                
                        <div class='section-title'>Financial Details</div>
                        <table class='details-table'>
                          <tr><td class='label'>Total Cost</td><td>$" . number_format($total_cost, 2) . "</td></tr>
                          <tr><td class='label'>Deposit (50%)</td><td>$" . number_format($deposit_amount, 2) . "</td></tr>
                          <tr><td class='label'>Balance</td><td>$" . number_format($balance_amount, 2) . "</td></tr>
                          <tr><td class='label'>Price Breakdown</td><td>" . nl2br($price_breakdown) . "</td></tr>
                          <tr><td class='label'>Booking Time</td><td>" . date('F j, Y \a\t g:i A') . "</td></tr>
                        </table>
                
                        <p><strong>Next Steps:</strong></p>
                        <ol>
                          <li>Wait for customer to make deposit payment.</li>
                          <li>Verify payment when received.</li>
                          <li>Contact customer to confirm booking.</li>
                          <li>Update booking status in the WordPress admin panel.</li>
                        </ol>
                      </td>
                    </tr>
                    <tr>
                      <td class='footer'>
                        <p>This is an automated message from your booking system.</p>
                      </td>
                    </tr>
                  </table>
                </body>
                </html>
                ";


                 // Set proper email headers to prevent spam
                $customer_headers = array(
                    'Content-Type: text/html; charset=UTF-8',
                    'From: Explore Uganda Uganda With Us <info@exploreug.com>',
                    'Reply-To: info@exploreug.com',
                    'X-Mailer: PHP/' . phpversion(),
                    'X-Priority: 3'
                );

                $admin_headers = array(
                    'Content-Type: text/html; charset=UTF-8',
                    'From: Explore Uganda With Us Booking System <noreply@exploreug.com>',
                    'Reply-To: ' . $email,
                    'X-Mailer: PHP/' . phpversion(),
                    'X-Priority: 1'
                );

                // Send emails
                $customer_email_sent = wp_mail($email, $customer_subject, $customer_message, $customer_headers);
                $admin_email_sent = wp_mail($host_email, $admin_subject, $admin_message, $admin_headers);


                // Set success flag
                $booking_success = true;

            } else {
                $booking_error = 'Sorry, there was an error processing your booking. Please try again.';
            }
        }
    }
}
?>

<div class="explore-uganda-single-package">
    <?php while (have_posts()) : the_post(); ?>

    <!-- Hero Section -->
    <section class="package-hero">
        <img src="<?php echo $featured_image ? esc_url($featured_image) : 'https://via.placeholder.com/1200x400/006b3e/ffffff?text=' . urlencode(get_the_title()); ?>" alt="<?php the_title(); ?>" class="package-hero-image">
        <div class="hero-overlay">
            <div class="hero-content">
                <h1><?php the_title(); ?></h1>
                <div class="trip-meta">
                    <?php if($duration): ?><span>⏱ <?php echo esc_html($duration); ?></span><?php endif; ?>
                    <?php if($destination): ?><span>📍 <?php echo esc_html($destination); ?></span><?php endif; ?>
                </div>
            </div>
        </div>
    </section>

    <div class="package-content">
        <div class="package-main">

            <!-- Overview Section -->
            <section class="package-section">
                <h2>Overview</h2>
                <div class="overview-content">
                    <?php the_content(); ?>
                </div>
            </section>

            <!-- Itinerary Section -->
            <?php if(!empty($itinerary) && is_array($itinerary)): ?>
            <section class="package-section">
                <h2>Itinerary</h2>
                <?php 
                $day_count = 0;
                foreach($itinerary as $index => $day): 
                    $day_count++;
                    if (empty($day['title']) && empty($day['description'])) continue;
                ?>
                <div class="itinerary-day">
                    <h3>Day <?php echo $day_count; ?>: <?php echo !empty($day['title']) ? esc_html($day['title']) : 'Day ' . $day_count; ?></h3>
                    <div class="day-content">
                        <?php if(!empty($day['description'])): ?>
                        <p><?php echo nl2br(esc_html($day['description'])); ?></p>
                        <?php endif; ?>
                        <?php if(!empty($day['meals'])): ?>
                        <p><strong>Meals:</strong> <?php echo esc_html($day['meals']); ?></p>
                        <?php endif; ?>
                        <?php if(!empty($day['accommodation'])): ?>
                        <p><strong>Accommodation:</strong> <?php echo esc_html($day['accommodation']); ?></p>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </section>
            <?php endif; ?>

            <!-- Inclusions & Exclusions -->
            <section class="package-section">
                <h2>What's Included</h2>
                <div class="includes-excludes">
                    <div class="includes">
                        <h3>Inclusions</h3>
                        <?php 
                            render_inclusions_exclusions(
                                $inclusions,
                                [
                                    'Professional English-speaking guide',
                                    'All transportation in suitable vehicle',
                                    'All entry fees',
                                    'Drinking water throughout the tour'
                                ]
                            );
                        ?>
                    </div>
                    <div class="excludes">
                        <h3>Exclusions</h3>
                        <?php 
                            render_inclusions_exclusions(
                                $exclusions,
                                [
                                    'International flights and visas',
                                    'Travel insurance',
                                    'Personal expenses (souvenirs, drinks, etc.)',
                                    'Tips for guides and drivers',
                                    'Optional activities not mentioned in itinerary'
                                ]
                            );
                        ?>
                    </div>
                </div>
            </section>
                        <!-- Customer Reviews & Ratings -->
<section class="package-section ratings-section">
    <h2>Customer Reviews</h2>
    
    <?php
    // Get average rating for this tour package
    $tour_package_name = get_the_title();
    $average_rating = get_tour_package_average_rating($tour_package_name);
    $total_reviews = get_tour_package_review_count($tour_package_name);
    ?>
    
    <div class="overall-rating">
        <div class="rating-stars">
            <?php echo generate_star_rating($average_rating); ?>
        </div>
        <div class="rating-details">
            <span class="average-rating"><?php echo number_format($average_rating, 1); ?> out of 5</span>
            <span class="total-reviews"><?php echo $total_reviews; ?> reviews</span>
        </div>
    </div>
    
    <?php if ($total_reviews > 0): ?>
    <div class="reviews-list">
        <h3>Recent Reviews</h3>
        <?php display_recent_reviews($tour_package_name, 3); ?>
    </div>
</section>
                <style>
.ratings-section {
    background: #ffffff;
    border: 1px solid #e0e0e0;
    border-radius: 10px;
    padding: 30px;
    margin-top: 20px;
}

.ratings-section h2 {
    color: #710b0d;
    margin-bottom: 25px;
    font-size: 24px;
    border-bottom: 2px solid #f7d508;
    padding-bottom: 10px;
}

.overall-rating {
    display: flex;
    align-items: center;
    gap: 25px;
    margin-bottom: 15px;
    padding: 5px;
    background: linear-gradient(135deg, #f8f8f8, #f8f8f8);
    border-left: 4px solid #f7d508;
}

.rating-stars-large {
    font-size: 28px;
    line-height: 1;
}

.rating-details {
    display: flex;
    flex-direction: column;
    gap: 5px;
}

.average-score {
    font-size: 32px;
    font-weight: bold;
    line-height: 1;
    color:#fff;
}

.total-reviews {
    color: #666;
    font-size: 14px;
    font-weight: 500;
}

.reviews-list h3 {
    color: #333;
    margin-bottom: 20px;
    font-size: 20px;
}

.review-item {
    background: #f8f9fa;
    padding: 5px;
    margin-bottom: 10px;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}

.review-item:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.review-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 15px;
}

.reviewer-info {
    display: flex;
    align-items: center;
    gap: 12px;
}

.reviewer-avatar {
    width: 40px;
    height: 40px;
    background: #710b0d;
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 16px;
}

.reviewer-details {
    display: flex;
    flex-direction: column;
    gap: 3px;
}

.reviewer-name {
    color: #333;
    font-size: 16px;
}

.review-date {
    color: #666;
    font-size: 12px;
}

.review-rating {
    font-size: 18px;
}

.review-content p {
    margin-bottom: 10px;
    line-height: 1.6;
    color: #333;
    font-size: 15px;
}

.tour-date {
    color: #888;
    font-size: 12px;
    font-style: italic;
}

.view-all-reviews {
    text-align: center;
    margin-top: 25px;
}

.view-reviews-btn {
    display: inline-block;
    background: #710b0d;
    color: white;
    padding: 12px 25px;
    text-decoration: none;
    border-radius: 5px;
    font-weight: 500;
    transition: background 0.3s ease;
}

.view-reviews-btn:hover {
    background: #8a0f12;
    color: white;
}

.no-reviews {
    text-align: center;
    padding: 40px 20px;
    background: #f8f9fa;
    border-radius: 8px;
    border: 2px dashed #ddd;
}

.no-reviews-icon {
    font-size: 48px;
    margin-bottom: 15px;
}

.no-reviews h3 {
    color: #710b0d;
    margin-bottom: 10px;
}

.no-reviews p {
    color: #666;
    margin: 0;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .ratings-section {
        padding: 20px;
    }
    
    .overall-rating {
        flex-direction: column;
        text-align: center;
        gap: 15px;
        padding: 20px;
    }
    
    .rating-stars-large {
        font-size: 24px;
    }
    
    .average-score {
        font-size: 28px;
    }
    
    .review-header {
        flex-direction: column;
        gap: 10px;
    }
    
    .reviewer-info {
        justify-content: center;
    }
    
    .review-item {
        padding: 20px;
    }
}</style>
                <?php endif; ?>
                
            </section>
        </div>

        <div class="package-sidebar">
            <!-- Pricing Information -->
            <section class="package-section pricing-info">
                <h2>Pricing (Per person)</h2>
                <div class="price-categories">
                    <?php if(!empty($price_adult)): ?>
                    <div class="price-category">
                        <span class="category-name">Non Ugandan</span>
                        <span class="category-price"><?php echo esc_html($price_adult); ?></span>
                    </div>
                    <?php endif; ?>
                    
                    <?php if(!empty($price_ugandan)): ?>
                    <div class="price-category">
                        <span class="category-name">Ugandan</span>
                        <span class="category-price"><?php echo esc_html($price_ugandan); ?></span>
                    </div>
                    <?php endif; ?>
                    
                    <?php if(!empty($price_child)): ?>
                    <div class="price-category">
                        <span class="category-name">Child</span>
                        <span class="category-price"><?php echo esc_html($price_child); ?></span>
                    </div>
                    <?php endif; ?>
                </div>
                
                <?php if(!empty($discount)): ?>
                <div class="discount-banner">
                    🎉 Special Discount: <?php echo esc_html($discount); ?>% Off
                </div>
                <?php endif; ?>
            </section>

            <!-- Booking Form -->
            <section class="package-section booking-form-section">
                <h2>Book This Tour</h2>
                
                <?php if ($booking_success): ?>
                    <div class="booking-success">
                        <h3>✅ Booking Successful!</h3>
                        <p>Thank you for your booking! We've sent a confirmation email with payment instructions to your email address.</p>
                        <p><strong>Next Steps:</strong></p>
                        <ol>
                            <li>Check your email for booking confirmation</li>
                            <li>Make the 50% deposit payment using the provided bank details</li>
                            <li>We'll confirm your booking once payment is received</li>
                        </ol>
                        <p>If you don't see the email within 5 minutes, please check your spam folder.</p>
                    </div>
                <?php elseif (!empty($booking_error)): ?>
                    <div class="booking-error">
                        <h3>❌ Booking Error</h3>
                        <p><?php echo esc_html($booking_error); ?></p>
                    </div>
                <?php endif; ?>
                
                <?php if (!$booking_success): ?>
                <form method="post" class="booking-form" id="bookingForm">
                    <?php wp_nonce_field('booking_form', 'booking_nonce'); ?>
                    
                    <div class="form-group">
                        <label for="full_name">Full Name *</label>
                        <input type="text" id="full_name" name="full_name" value="<?php echo isset($_POST['full_name']) ? esc_attr($_POST['full_name']) : ''; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="email">Email Address *</label>
                        <input type="email" id="email" name="email" value="<?php echo isset($_POST['email']) ? esc_attr($_POST['email']) : ''; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="phone">Phone Number *</label>
                        <input type="tel" id="phone" name="phone" value="<?php echo isset($_POST['phone']) ? esc_attr($_POST['phone']) : ''; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="tour_package">Tour Package *</label>
                        <input type="text" id="tour_package" name="tour_package" value="<?php echo esc_attr(get_the_title()); ?>" readonly>
                    </div>
                    
                    <div class="form-row">
                        <?php if(!empty($price_adult)): ?>
                        <div class="form-group">
                            <label for="num_adults">International Adults</label>
                            <select id="num_adults" name="num_adults">
                                <option value="0">0</option>
                                <?php for ($i = 1; $i <= 20; $i++): ?>
                                    <option value="<?php echo $i; ?>" <?php echo (isset($_POST['num_adults']) && $_POST['num_adults'] == $i) ? 'selected' : ''; ?>><?php echo $i; ?></option>
                                <?php endfor; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                        
                        <?php if(!empty($price_ugandan)): ?>
                        <div class="form-group">
                            <label for="num_ugandans">Ugandan Residents</label>
                            <select id="num_ugandans" name="num_ugandans">
                                <option value="0">0</option>
                                <?php for ($i = 1; $i <= 20; $i++): ?>
                                    <option value="<?php echo $i; ?>" <?php echo (isset($_POST['num_ugandans']) && $_POST['num_ugandans'] == $i) ? 'selected' : ''; ?>><?php echo $i; ?></option>
                                <?php endfor; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                    </div>
                        
                    <?php if(!empty($price_child)): ?>
                    <div class="form-group">
                        <label for="num_children">Children</label>
                        <select id="num_children" name="num_children">
                            <option value="0">0</option>
                            <?php for ($i = 1; $i <= 10; $i++): ?>
                                <option value="<?php echo $i; ?>" <?php echo (isset($_POST['num_children']) && $_POST['num_children'] == $i) ? 'selected' : ''; ?>><?php echo $i; ?></option>
                            <?php endfor; ?>
                        </select>
                    </div>
                    <?php endif; ?>
                    
                    <div class="form-group">
                        <label for="tour_date">Tour Date *</label>
                        <input type="date" id="tour_date" name="tour_date" value="<?php echo isset($_POST['tour_date']) ? esc_attr($_POST['tour_date']) : ''; ?>" min="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    
                    <div class="price-summary">
                        <h4>Price Summary</h4>
                        <div id="priceBreakdown">
                            <!-- Price breakdown will be populated by JavaScript -->
                        </div>
                        <?php if(!empty($discount)): ?>
                        <div class="price-line discount">
                            <span>Discount (<?php echo esc_html($discount); ?>%):</span>
                            <span id="discountAmount">-$0</span>
                        </div>
                        <?php endif; ?>
                        <div class="price-line total">
                            <strong>Total Cost:</strong>
                            <strong id="totalCost">$0</strong>
                        </div>
                        <div class="payment-instruction">
                            <p><small>A 50% deposit is required to secure your booking. The balance is due 30 days before your tour date.</small></p>
                        </div>
                    </div>
                    
                    <button type="submit" name="booking_submit" class="submit-booking">Book Now</button>
                </form>
                <?php endif; ?>
            </section>
            
            <!-- Contact Information -->
            <?php if($contact_email || $contact_phone): ?>
            <section class="package-section contact-details">
                <h2>Contact Information</h2>
                <div class="contact-info">
                    <?php if($contact_email): ?><p><strong>Email:</strong> <a href="mailto:<?php echo esc_attr($contact_email); ?>"><?php echo esc_html($contact_email); ?></a></p><?php endif; ?>
                    <?php if($contact_phone): ?><p><strong>Phone:</strong> <a href="tel:<?php echo esc_attr($contact_phone); ?>"><?php echo esc_html($contact_phone); ?></a></p><?php endif; ?>
                </div>
            </section>
            <?php endif; ?>
        </div>
    </div>

    <?php endwhile; ?>
</div>



<script>
jQuery(document).ready(function($) {
    // Price calculation function
    function calculateTotal() {
        var adults = parseInt($('#num_adults').val()) || 0;
        var ugandans = parseInt($('#num_ugandans').val()) || 0;
        var children = parseInt($('#num_children').val()) || 0;
        
        // Get prices from PHP variables
        var adultPrice = <?php echo !empty($price_adult) ? floatval(str_replace(['$', ','], '', $price_adult)) : 0; ?>;
        var ugandanPrice = <?php echo !empty($price_ugandan) ? floatval(str_replace(['$', ','], '', $price_ugandan)) : 0; ?>;
        var childPrice = <?php echo !empty($price_child) ? floatval(str_replace(['$', ','], '', $price_child)) : 0; ?>;
        var discount = <?php echo !empty($discount) ? $discount : 0; ?>;
        
        var total = 0;
        var priceBreakdown = '';
        
        // Calculate costs for each category
        if (adultPrice > 0 && adults > 0) {
            var adultTotal = adults * adultPrice;
            total += adultTotal;
            priceBreakdown += '<div class="price-line"><span>International Adults (x' + adults + '):</span><span>$' + adultTotal.toFixed(2) + '</span></div>';
        }
        
        if (ugandanPrice > 0 && ugandans > 0) {
            var ugandanTotal = ugandans * ugandanPrice;
            total += ugandanTotal;
            priceBreakdown += '<div class="price-line"><span>Ugandan Residents (x' + ugandans + '):</span><span>$' + ugandanTotal.toFixed(2) + '</span></div>';
        }
        
        if (childPrice > 0 && children > 0) {
            var childTotal = children * childPrice;
            total += childTotal;
            priceBreakdown += '<div class="price-line"><span>Children (x' + children + '):</span><span>$' + childTotal.toFixed(2) + '</span></div>';
        }
        
        var originalTotal = total;
        var discountAmount = 0;
        
        // Apply discount
        if (discount > 0 && total > 0) {
            discountAmount = total * (discount / 100);
            total = total - discountAmount;
        }
        
        // Update display
        if (priceBreakdown) {
            $('#priceBreakdown').html(priceBreakdown);
        } else {
            $('#priceBreakdown').html('<div class="price-line"><span>Select participants to see pricing</span><span>$0</span></div>');
        }
        
        if (discount > 0 && originalTotal > 0) {
            $('#discountAmount').text('-$' + discountAmount.toFixed(2));
            $('.price-line.discount').show();
        } else {
            $('.price-line.discount').hide();
        }
        
        $('#totalCost').text('$' + total.toFixed(2));
    }
    
    // Bind events
    $('#num_adults, #num_ugandans, #num_children').change(calculateTotal);
    
    // Initialize
    calculateTotal();
    
    // Form validation
    $('#bookingForm').on('submit', function(e) {
        var tourDate = new Date($('#tour_date').val());
        var today = new Date();
        today.setHours(0, 0, 0, 0);
        
        if (tourDate <= today) {
            alert('Please select a future date for the tour.');
            e.preventDefault();
            return false;
        }
        
        var adults = parseInt($('#num_adults').val()) || 0;
        var ugandans = parseInt($('#num_ugandans').val()) || 0;
        var children = parseInt($('#num_children').val()) || 0;
        
        if (adults + ugandans + children < 1) {
            alert('Please select at least one participant.');
            e.preventDefault();
            return false;
        }
        
        return true;
    });
    
    // Set minimum date for tour date
    var today = new Date().toISOString().split('T')[0];
    $('#tour_date').attr('min', today);
});
</script>

<script>
jQuery(document).ready(function($) {
    // Itinerary Accordion Functionality
    $('.itinerary-day').each(function(index) {
        var $day = $(this);
        var $heading = $day.find('h3');
        var $content = $day.find('.day-content');
        
        // Add accordion classes and structure
        $heading.addClass('itinerary-accordion-header');
        $content.addClass('itinerary-accordion-content');
        
        // Create toggle button
        var toggleIcon = '<span class="accordion-toggle-icon">+</span>';
        $heading.append(toggleIcon);
        
        // Initially hide all content except first day
        if (index === 0) {
            $content.show();
            $heading.find('.accordion-toggle-icon').text('−');
            $heading.addClass('active');
        } else {
            $content.hide();
        }
        
        // Click event for accordion headers
        $heading.on('click', function() {
            var $this = $(this);
            var $content = $this.next('.itinerary-accordion-content');
            var $icon = $this.find('.accordion-toggle-icon');
            
            // Toggle current content
            $content.slideToggle(300);
            $this.toggleClass('active');
            
            // Update icon
            if ($this.hasClass('active')) {
                $icon.text('−');
            } else {
                $icon.text('+');
            }
            
            // Optional: Close other accordions when one opens
            // Uncomment below if you want only one open at a time
            
            if ($this.hasClass('active')) {
                $('.itinerary-accordion-header').not($this).removeClass('active');
                $('.itinerary-accordion-header').not($this).find('.accordion-toggle-icon').text('+');
                $('.itinerary-accordion-content').not($content).slideUp(300);
            }
            
        });
    });
    
    // Optional: Add keyboard accessibility
    $('.itinerary-accordion-header').on('keydown', function(e) {
        if (e.key === 'Enter' || e.key === ' ') {
            e.preventDefault();
            $(this).trigger('click');
        }
    });
});
</script>

<?php get_footer(); ?>