<?php
/*
Plugin Name: Slider CPT with AJAX
Description: Custom post type for Sliders with full image management
Version: 1.0
Author: You
*/

// =============================
// 1. REGISTER SLIDER POST TYPE
// =============================
add_action('init', function() {
    $labels = [
        'name'          => 'Sliders',
        'singular_name' => 'Slider',
        'menu_name'     => 'Sliders',
        'add_new'       => 'Add New',
        'add_new_item'  => 'Add New Slider',
        'edit_item'     => 'Edit Slider',
        'new_item'      => 'New Slider',
        'view_item'     => 'View Slider',
        'search_items'  => 'Search Sliders',
        'not_found'     => 'No sliders found',
    ];
    $args = [
        'labels'             => $labels,
        'public'             => true,
        'has_archive'        => false,
        'menu_icon'          => 'dashicons-images-alt2',
        'supports'           => ['title'],
        'show_in_menu'       => true,
        'exclude_from_search'=> true,
        'publicly_queryable' => false,
    ];
    register_post_type('slider', $args);
});

// =============================
// 2. ENQUEUE ADMIN SCRIPTS
// =============================
add_action('admin_enqueue_scripts', function($hook) {
    global $post;
    
    if (($hook == 'post-new.php' || $hook == 'post.php') && isset($post->post_type) && $post->post_type === 'slider') {
        wp_enqueue_media();
        wp_enqueue_script('jquery');
        
        // Inline admin JS
        add_action('admin_footer', 'slider_admin_inline_js');
    }
});

function slider_admin_inline_js() {
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        let frame;
        const $imagesInput = $('#slider_images');
        const $preview = $('#slider_images_preview');
        const $saveStatus = $('#slider_save_status');

        // Open media library
        $('#add_slider_images').on('click', function(e) {
            e.preventDefault();

            if (frame) {
                frame.open();
                return;
            }

            frame = wp.media({
                title: 'Select or Upload Slider Images',
                button: { text: 'Add to Slider' },
                multiple: true,
                library: { type: 'image' }
            });

            frame.on('select', function() {
                const attachments = frame.state().get('selection').toJSON();
                const existingIds = $imagesInput.val() ? $imagesInput.val().split(',') : [];
                const newIds = [];

                attachments.forEach(function(attachment) {
                    if (existingIds.indexOf(attachment.id.toString()) === -1) {
                        existingIds.push(attachment.id);
                        newIds.push(attachment.id);
                    }
                });

                // Update hidden field
                $imagesInput.val(existingIds.filter(id => id !== '').join(','));

                // Add new images to preview
                newIds.forEach(function(id) {
                    if (id) {
                        const imageUrl = wp.media.attachment(id).get('sizes').thumbnail ? wp.media.attachment(id).get('sizes').thumbnail.url : wp.media.attachment(id).get('url');
                        const $previewItem = $(
                            '<div class="slider-image-preview" data-image-id="' + id + '" style="display:inline-block;margin:5px;text-align:center;position:relative;">' +
                            '<img src="' + imageUrl + '" style="width:100px;height:100px;object-fit:cover;border:2px solid #0073aa;">' +
                            '<div style="font-size:11px;margin-top:5px;">ID: ' + id + '</div>' +
                            '<button type="button" class="button button-small remove-single-image" style="position:absolute;top:-5px;right:-5px;background:#dc3232;color:white;border:none;border-radius:50%;width:20px;height:20px;font-size:12px;line-height:1;cursor:pointer;">×</button>' +
                            '</div>'
                        );
                        $preview.append($previewItem);
                    }
                });

                // Remove "no images" message if it exists
                $preview.find('p').remove();
                
                // Auto-save
                saveSliderImages();
            });

            frame.open();
        });

        // Clear all images
        $('#clear_slider_images').on('click', function() {
            if (confirm('Are you sure you want to remove all images from this slider?')) {
                $imagesInput.val('');
                $preview.empty().append('<p style="color:#666;text-align:center;margin:20px 0;">No images yet. Click "Add Images".</p>');
                saveSliderImages();
            }
        });

        // Remove single image (delegated event for dynamically added elements)
        $preview.on('click', '.remove-single-image', function() {
            const $previewItem = $(this).closest('.slider-image-preview');
            const imageId = $previewItem.data('image-id');
            
            let currentIds = $imagesInput.val() ? $imagesInput.val().split(',') : [];
            currentIds = currentIds.filter(id => id != imageId && id !== '');
            
            $imagesInput.val(currentIds.join(','));
            $previewItem.remove();
            
            // Show "no images" message if all are removed
            if (currentIds.length === 0) {
                $preview.append('<p style="color:#666;text-align:center;margin:20px 0;">No images yet. Click "Add Images".</p>');
            }
            
            saveSliderImages();
        });

        // AJAX save function
        function saveSliderImages() {
            $saveStatus.text('Saving...').css('color', '#666');
            
            $.ajax({
                url: '<?php echo admin_url('admin-ajax.php'); ?>',
                type: 'POST',
                data: {
                    action: 'save_slider_images',
                    security: '<?php echo wp_create_nonce('save_slider_images_nonce'); ?>',
                    post_id: $('#post_ID').val(),
                    images: $imagesInput.val()
                },
                success: function(response) {
                    if (response.success) {
                        $saveStatus.text('Saved! ' + response.data.count + ' images').css('color', '#46b450');
                        setTimeout(function() {
                            $saveStatus.fadeOut(300, function() {
                                $saveStatus.text('').show();
                            });
                        }, 2000);
                    } else {
                        $saveStatus.text('Error saving').css('color', '#dc3232');
                    }
                },
                error: function() {
                    $saveStatus.text('Error saving').css('color', '#dc3232');
                }
            });
        }
    });
    </script>
    <?php
}

// =============================
// 3. ADMIN METABOX
// =============================
add_action('add_meta_boxes', function() {
    add_meta_box('slider_images_box', 'Slider Images', 'render_slider_images_box', 'slider', 'normal', 'high');
    add_meta_box('slider_shortcode_box', 'Slider Shortcode', function($post) {
        echo '<p>Copy and paste this shortcode into any page or post:</p>';
        echo '<code>[slider id="' . esc_attr($post->ID) . '"]</code>';
        echo '<p class="description">You can also use the slider ID in template files with: <code>echo do_shortcode(\'[slider id="' . esc_attr($post->ID) . '"]\');</code></p>';
    }, 'slider', 'side', 'high');
});

function render_slider_images_box($post) {
    wp_nonce_field('slider_images_nonce', 'slider_images_nonce_field');
    $images = get_post_meta($post->ID, '_slider_images', true);
    ?>
    <div id="slider_images_preview" style="border:2px dashed #ddd;padding:10px;min-height:60px;margin-bottom:15px;">
        <?php if ($images) :
            $ids = array_filter(explode(',', $images));
            foreach ($ids as $id): 
                if (!empty($id) && wp_attachment_is_image($id)) : 
                    $image_url = wp_get_attachment_image_url($id, 'thumbnail');
                    ?>
                    <div class="slider-image-preview" data-image-id="<?php echo esc_attr($id); ?>" style="display:inline-block;margin:5px;text-align:center;position:relative;">
                        <img src="<?php echo esc_url($image_url); ?>" style="width:100px;height:100px;object-fit:cover;border:2px solid #0073aa;">
                        <div style="font-size:11px;margin-top:5px;">ID: <?php echo $id; ?></div>
                        <button type="button" class="button button-small remove-single-image" style="position:absolute;top:-5px;right:-5px;background:#dc3232;color:white;border:none;border-radius:50%;width:20px;height:20px;font-size:12px;line-height:1;cursor:pointer;">×</button>
                    </div>
                <?php endif;
            endforeach;
        else: ?>
            <p style="color:#666;text-align:center;margin:20px 0;">No images yet. Click "Add Images".</p>
        <?php endif; ?>
    </div>

    <input type="hidden" id="slider_images" name="slider_images" value="<?php echo esc_attr($images); ?>">
    
    <p>
        <button type="button" class="button button-primary" id="add_slider_images">Add Images</button>
        <button type="button" class="button button-secondary" id="clear_slider_images">Clear All</button>
        <span id="slider_save_status" style="margin-left:15px;font-style:italic;color:#666;"></span>
    </p>
    <p class="description">
        You can add multiple images at once. Click individual images to remove them, or clear all at once.
    </p>
    <?php
}

// =============================
// 4. AJAX SAVE HANDLER
// =============================
add_action('wp_ajax_save_slider_images', function() {
    check_ajax_referer('save_slider_images_nonce', 'security');
    
    $post_id = intval($_POST['post_id']);
    $images = sanitize_text_field($_POST['images']);
    
    if (!current_user_can('edit_post', $post_id)) {
        wp_send_json_error(['message' => 'Permission denied']);
    }
    
    $image_ids = array_filter(explode(',', $images));
    $valid_ids = [];
    
    foreach ($image_ids as $image_id) {
        if (is_numeric($image_id) && wp_attachment_is_image($image_id)) {
            $valid_ids[] = $image_id;
        }
    }
    
    $clean_images = implode(',', $valid_ids);
    update_post_meta($post_id, '_slider_images', $clean_images);
    
    wp_send_json_success(['message' => 'Images saved successfully', 'count' => count($valid_ids)]);
});

// =============================
// 5. SAVE ON REGULAR POST SAVE
// =============================
add_action('save_post_slider', function($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!isset($_POST['slider_images_nonce_field']) || !wp_verify_nonce($_POST['slider_images_nonce_field'], 'slider_images_nonce')) return;
    if (!current_user_can('edit_post', $post_id)) return;

    if (isset($_POST['slider_images'])) {
        $images = sanitize_text_field($_POST['slider_images']);
        update_post_meta($post_id, '_slider_images', $images);
    }
});

// =============================
// 6. SHORTCODE DISPLAY
// =============================
add_shortcode('slider', function($atts) {
    $atts = shortcode_atts(['id'=>''], $atts, 'slider');
    if (empty($atts['id'])) return '<p>No slider ID provided.</p>';

    $images = get_post_meta($atts['id'], '_slider_images', true);
    if (!$images) return '<p>No images in this slider.</p>';

    $ids = explode(',', $images);
    ob_start(); ?>

    <div class="cpt-slider-main" id="cpt-slider-<?php echo esc_attr($atts['id']); ?>">
        <div class="cpt-slides-wrapper">
            <?php foreach ($ids as $index => $id): 
                if (!wp_attachment_is_image($id)) continue;
                $image_url = wp_get_attachment_image_url($id, 'large');
                $image_alt = get_post_meta($id, '_wp_attachment_image_alt', true);
                if (!$image_alt) $image_alt = 'Slider Image ' . ($index + 1);
                ?>
                <div class="cpt-slide-item <?php echo $index === 0 ? 'cpt-slide-current' : ''; ?>" data-slide-index="<?php echo $index; ?>">
                    <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($image_alt); ?>" class="cpt-slide-img">
                </div>
            <?php endforeach; ?>
        </div>
        
        <button type="button" class="cpt-slider-btn cpt-slider-prev-btn" aria-label="Previous slide"><span>‹</span></button>
        <button type="button" class="cpt-slider-btn cpt-slider-next-btn" aria-label="Next slide"><span>›</span></button>
        
        <div class="cpt-slider-dots">
            <?php foreach ($ids as $index => $id): 
                if (!wp_attachment_is_image($id)) continue; ?>
                <button type="button" class="cpt-slider-dot <?php echo $index === 0 ? 'cpt-dot-active' : ''; ?>" data-slide-index="<?php echo $index; ?>" aria-label="Go to slide <?php echo $index + 1; ?>"></button>
            <?php endforeach; ?>
        </div>
    </div>

    <style>
    .cpt-slider-main { 
        position: relative; 
        max-width: 100%; 
        margin: 20px auto; 
        overflow: hidden; 
        border-radius: 20px; 
        box-shadow: 0 4px 12px rgba(0,0,0,0.15); 
        background: #000; 
    }
    .cpt-slides-wrapper { 
        position: relative; 
        width: 100%; 
        height: 630px; 
    }
    .cpt-slide-item { 
        position: absolute; 
        top: 0; 
        left: 0; 
        width: 100%; 
        height: 100%; 
        opacity: 0; 
        transition: opacity 0.6s ease-in-out; 
        z-index: 1; 
    }
    .cpt-slide-item.cpt-slide-current { 
        opacity: 1; 
        z-index: 2; 
    }
    .cpt-slide-img { 
        width: 100%; 
        height: 100%; 
        object-fit: cover; 
        display: block; 
    }
    .cpt-slider-btn { 
        position: absolute;
        top: 50%;
        transform: translateY(-50%);
        background: rgba(0,0,0,0.5);
        color: white;
        border: none;
        width: 50px;
        height: 50px;
        border-radius: 50%;
        font-size: 24px;
        cursor: pointer;
        z-index: 10;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: background 0.3s ease;
    }
    .cpt-slider-btn:hover {
        background: rgba(0,0,0,0.8);
    }
    .cpt-slider-prev-btn {
        left: 20px;
    }
    .cpt-slider-next-btn {
        right: 20px;
    }
    .cpt-slider-dots { 
        position: absolute; 
        bottom: 20px; 
        left: 0; 
        right: 0; 
        display: flex; 
        justify-content: center; 
        gap: 10px; 
        z-index: 100; 
    }
    .cpt-slider-dot { 
        width: 14px; 
        height: 14px; 
        border-radius: 50%; 
        border: 2px solid white; 
        background: transparent; 
        cursor: pointer; 
        transition: all 0.3s ease; 
        padding: 0; 
    }
    .cpt-slider-dot.cpt-dot-active { 
        background: white; 
        transform: scale(1.2); 
    }
    .cpt-slider-dot:hover { 
        background: rgba(255,255,255,0.7); 
    }
    @media (max-width: 768px) { 
        .cpt-slides-wrapper { height: 400px; } 
        .cpt-slider-btn { width: 40px; height: 40px; font-size: 20px; }
        .cpt-slider-prev-btn { left: 10px; }
        .cpt-slider-next-btn { right: 10px; }
    }
    @media (max-width: 480px) { 
        .cpt-slides-wrapper { height: 300px; } 
        .cpt-slider-btn { width: 35px; height: 35px; font-size: 18px; }
    }
    </style>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const slider = document.getElementById('cpt-slider-<?php echo esc_attr($atts['id']); ?>');
        if (!slider) return;

        const slides = slider.querySelectorAll('.cpt-slide-item');
        const prevBtn = slider.querySelector('.cpt-slider-prev-btn');
        const nextBtn = slider.querySelector('.cpt-slider-next-btn');
        const dots = slider.querySelectorAll('.cpt-slider-dot');
        let current = 0;
        let autoSlideInterval;

        function showSlide(index) {
            slides[current].classList.remove('cpt-slide-current');
            dots[current].classList.remove('cpt-dot-active');
            current = index;
            slides[current].classList.add('cpt-slide-current');
            dots[current].classList.add('cpt-dot-active');
        }

        function nextSlide() { 
            showSlide((current + 1) % slides.length); 
        }
        function prevSlide() { 
            showSlide((current - 1 + slides.length) % slides.length); 
        }

        function startAutoSlide() {
            autoSlideInterval = setInterval(nextSlide, 5000);
        }

        function stopAutoSlide() {
            clearInterval(autoSlideInterval);
        }

        // Event listeners
        if (prevBtn) prevBtn.addEventListener('click', function() {
            prevSlide();
            stopAutoSlide();
            startAutoSlide();
        });
        
        if (nextBtn) nextBtn.addEventListener('click', function() {
            nextSlide();
            stopAutoSlide();
            startAutoSlide();
        });
        
        dots.forEach((dot, index) => {
            dot.addEventListener('click', function() {
                showSlide(index);
                stopAutoSlide();
                startAutoSlide();
            });
        });

        // Pause on hover
        slider.addEventListener('mouseenter', stopAutoSlide);
        slider.addEventListener('mouseleave', startAutoSlide);

        // Start autoplay
        startAutoSlide();
    });
    </script>
    <?php
    return ob_get_clean();
});

// =============================
// 7. ALWAYS ACTIVE - NO DEACTIVATION
// =============================
register_activation_hook(__FILE__, function() {
    // Flush rewrite rules on activation
    flush_rewrite_rules();
});

register_deactivation_hook(__FILE__, function() {
    // Don't remove data on deactivation since it should always be active
    // flush_rewrite_rules();
});
?>