<?php
/**
 * Kipya functions and definitions.
 *
 * @link https://afiego.org
 *
 * @package Kipya
 */
set_time_limit(300); // Set to a higher value if needed
if ( ! function_exists( 'kipya_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 */
	function kipya_setup() {
		// Make the theme available for translation.
		load_theme_textdomain( 'kipya', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

        //Upload Logo in website.
        add_theme_support('custom-logo');

		// Let WordPress manage the document title.
		add_theme_support( 'title-tag' );

		// Enable support for Post Thumbnails on posts and pages.
		add_theme_support( 'post-thumbnails' );


		// Register navigation menus.
		register_nav_menus( array(
			'primary'   => esc_html__( 'Primary Menu', 'kipya' ),
			'top-menu'  => esc_html__( 'Top Menu', 'kipya' ),
            'sidebar'   => esc_html__( 'Sidebar Menu', 'kipya' ),
            'footer'    => esc_html__( 'Footer Menu', 'kipya' ),
		) );

		// Switch default core markup for search form, comment form, and comments to output valid HTML5.
		add_theme_support( 'kipya', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'kipya_custom_background_args', array(
			'default-color' => 'ffffff',
			'default-image' => '',
		) ) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

        /**
		 * Enable support for the following post formats:
		 * aside, gallery, quote, image, and video
		 */
        add_theme_support( 'post-formats', array( 'aside', 'gallery', 'quote', 'image', 'video' ) );
	}
endif;
add_action( 'after_setup_theme', 'kipya_setup' );

// Enqueue styles.
function kipya_styles() {
    wp_register_style('kipya-style', get_template_directory_uri() . '/style.css', array(), '1.0.0', 'all');
    wp_register_style('custom', get_template_directory_uri() . '/assets/css/custom.css', array(), '1.0.0', 'all');
    wp_register_style('bootstrap.min', get_template_directory_uri() . '/assets/vendor/bootstrap/css/bootstrap.min.css', array(), '5.3.2', 'all');
    wp_register_style('bootstrap-icons', get_template_directory_uri() . '/assets/vendor/bootstrap-icons/font/bootstrap-icons.min.css', array(), '1.11.3', 'all');
    wp_register_style('aos', get_template_directory_uri() . '/assets/vendor/aos/aos.css', array(), '2.3.4', 'all');
    wp_register_style('swiper', get_template_directory_uri() . '/assets/vendor/swiper/swiper.min.css', array(), '5.4.5', 'all');
    wp_register_style('toastr.min', get_template_directory_uri() . '/assets/vendor/toastr/toastr.min.css', array(), '2.1.3', 'all');
    wp_register_style('datatables.min', get_template_directory_uri() . '/assets/vendor/DataTables/datatables.min.css', array(), '1.13.3', 'all');



    wp_enqueue_style( 'kipya-style');
    wp_enqueue_style( 'custom');
    wp_enqueue_style( 'bootstrap.min');
    wp_enqueue_style( 'bootstrap-icons');
    wp_enqueue_style( 'aos');
    wp_enqueue_style( 'swiper');
    wp_enqueue_style( 'toastr.min');
    wp_enqueue_style( 'datatables.min');
}
add_action( 'wp_enqueue_scripts', 'kipya_styles' );

// Enqueue scripts.
function kipya_scripts() {

    wp_enqueue_script( 'bootstrap-script', get_template_directory_uri() . '/assets/vendor/bootstrap/js/bootstrap.bundle.min.js', array(), '4.5.2', 'all');
    wp_enqueue_script( 'aos-script', get_template_directory_uri() . '/assets/vendor/aos/aos.js', array(), '2.3.4', 'all');
    wp_enqueue_script( 'swiper-script', get_template_directory_uri() . '/assets/vendor/swiper/swiper.min.js', array(), '5.4.5', 'all');
    wp_enqueue_script( 'toastr-script', get_template_directory_uri() . '/assets/vendor/toastr/toastr.min.js', array(), '2.1.3', 'all');
    wp_enqueue_script( 'jquery-script', get_template_directory_uri() . '/assets/vendor/jquery/jquery-3.7.1.min.js', array( ), '3.7.1', 'all');
    wp_enqueue_script( 'kipya-script', get_template_directory_uri() . '/assets/js/main.js', array(), '1.0.0', 'all');
    wp_enqueue_script( 'search-script', get_template_directory_uri() . '/assets/js/search.js', array(), '1.0.0', 'all');
    wp_enqueue_script( 'datatables.min', get_template_directory_uri() . '/assets/vendor/DataTables/datatables.min.js', array(), '1.13.3', 'all');

function enqueue_glightbox() {
    wp_enqueue_script('glightbox', 'https://cdn.jsdelivr.net/npm/glightbox/dist/js/glightbox.min.js', array(), null, true);
    wp_enqueue_style('glightbox-css', 'https://cdn.jsdelivr.net/npm/glightbox/dist/css/glightbox.min.css');
}
add_action('wp_enqueue_scripts', 'enqueue_glightbox');


function force_gallery_links_to_media($block_content, $block) {
    if ($block['blockName'] === 'core/gallery' && !empty($block['innerBlocks'])) {
        foreach ($block['innerBlocks'] as &$innerBlock) {
            if ($innerBlock['blockName'] === 'core/image' && isset($innerBlock['attrs']['id'])) {
                $image_id = $innerBlock['attrs']['id'];
                $image_url = wp_get_attachment_url($image_id);
                if ($image_url) {
                    $innerBlock['attrs']['linkDestination'] = 'media';
                    $innerBlock['attrs']['href'] = $image_url;
                }
            }
        }
    }
    return $block_content;
}
add_filter('render_block', 'force_gallery_links_to_media', 10, 2);



    // Pass the ajaxurl to search.js
    wp_localize_script('search-script', 'kipya_ajax_object', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
    ));
}
//add_action( 'wp_enqueue_scripts', 'kipya_scripts' );
add_action('wp_footer','kipya_scripts');

// ligtbox-gallery
function enqueue_custom_scripts() {
    // Enqueue lightbox library
    wp_enqueue_style('lightbox-css', 'https://cdnjs.cloudflare.com/ajax/libs/lightbox2/2.11.3/css/lightbox.min.css');
    wp_enqueue_script('lightbox-js', 'https://cdnjs.cloudflare.com/ajax/libs/lightbox2/2.11.3/js/lightbox.min.js', array('jquery'), null, true);

    // Enqueue custom JavaScript
    wp_enqueue_script('custom-gallery-lightbox', get_template_directory_uri() . '/assets/js/gallery-lightbox.js', array('jquery'), null, true);

    // Enqueue custom AJAX script for projects page
    if (is_page('projects-page')) {
        wp_enqueue_script('custom-ajax-script', get_template_directory_uri() . '/assets/js/custom-ajax-script.js', array('jquery'), null, true);
        wp_localize_script('custom-ajax-script', 'ajax_object', array('ajax_url' => admin_url('admin-ajax.php')));
    }
}
add_action('wp_enqueue_scripts', 'enqueue_custom_scripts');

// If Dynamic Sidebar Exists
if (function_exists('register_sidebar'))
{
    // Define Sidebar Widget Area 1
    register_sidebar(array(
        'name' => __('Widget Area 1', 'kipya'),
        'description' => __('Description for this widget-area...', 'kipya'),
        'id' => 'widget-area-1',
        'before_widget' => '<div id="%1$s" class="%2$s">',
        'after_widget' => '</div>',
        'before_title' => '<h3>',
        'after_title' => '</h3>'
    ));

    // Define Sidebar Widget Area 2
    register_sidebar(array(
        'name' => __('Widget Area 2', 'kipya'),
        'description' => __('Description for this widget-area...', 'kipya'),
        'id' => 'widget-area-2',
        'before_widget' => '<div id="%1$s" class="%2$s">',
        'after_widget' => '</div>',
        'before_title' => '<h3>',
        'after_title' => '</h3>'
    ));
    
    register_sidebar( array(
        'name'          => esc_html__( 'Top Left', 'kipya' ),
        'id'            => 'topl',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
	register_sidebar( array(
        'name'          => esc_html__( 'Top Right', 'kipya' ),
        'id'            => 'topr',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
	register_sidebar( array(
        'name'          => esc_html__( 'Search Area', 'kipya' ),
        'id'            => 'searchi',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
	register_sidebar( array(
        'name'          => esc_html__( 'Quick Number', 'kipya' ),
        'id'            => 'inno',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
    register_sidebar( array(
        'name'          => esc_html__( 'Menu Right', 'kipya' ),
        'id'            => 'nav-right',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
    register_sidebar(array(
        'name' => __('Services Sidebar', 'kipya'),
        'description' => __('Description for this widget-area...', 'kipya'),
        'id' => 'servsidebar',
        'before_widget' => '<div id="%1$s" class="%2$s">',
        'after_widget' => '</div>',
        'before_title' => '<h3>',
        'after_title' => '</h3>'
    ));
    register_sidebar( array(
        'name'          => esc_html__( 'Advert Top', 'kipya' ),
        'id'            => 'advtop',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
     register_sidebar( array(
        'name'          => esc_html__( 'Newsletter', 'kipya' ),
        'id'            => 'footernewz',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
    register_sidebar( array(
        'name'          => esc_html__( 'Footer Starter', 'kipya' ),
        'id'            => 'footeri',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
	register_sidebar( array(
        'name'          => esc_html__( 'Footer A', 'kipya' ),
        'id'            => 'footera',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
	register_sidebar( array(
        'name'          => esc_html__( 'Footer B', 'kipya' ),
        'id'            => 'footerb',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
	register_sidebar( array(
        'name'          => esc_html__( 'Footer C', 'kipya' ),
        'id'            => 'footerc',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
    register_sidebar( array(
        'name'          => esc_html__( 'Footer D', 'kipya' ),
        'id'            => 'footerd',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
     register_sidebar( array(
        'name'          => esc_html__( 'Copyright Links', 'kipya' ),
        'id'            => 'footer-links',
        'description'   => esc_html__( 'Add widgets here.', 'kipya' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ) );
}

//Activate Excerpt in PAGES
function enable_page_excerpts() {
    add_post_type_support('page', 'excerpt');
}
add_action('init', 'enable_page_excerpts');




//Custom BOOTSTRAP Menu 
include get_template_directory() . '/inc/bootstrap-5-wordpress-navbar-walker.php';
include get_template_directory() . '/inc/custom-nav-walker.php';

//Custom Post Type :: SEO / SEARCH / PAGE CATS
include get_template_directory() . '/inc/seo-post-types.php';
include get_template_directory() . '/inc/accordion.php';
include get_template_directory() . '/inc/ajax-search.php';
include get_template_directory() . '/inc/add_page_category.php';

//Custom Post Type :: SLIDING PHOTOS 
//include get_template_directory() . '/inc/slides-post-types.php';
include get_template_directory() . '/inc/text-slider-post-type.php';
include get_template_directory() . '/inc/photo.php';

//Custom Post Type :: UPCOMING EVENTS
include get_template_directory() . '/inc/upcoming-events-post-types.php';
include get_template_directory() . '/inc/more-upcoming-events.php';
include get_template_directory() . '/inc/past-events.php';

//Custom Post Type :: DOWNLOADS
include get_template_directory() . '/inc/downloads-post-types.php';

//Custom Post Type :: TEAMS
include get_template_directory() . '/inc/team-post-types.php';
include get_template_directory() . '/inc/contact-form.php';

//Custom Post Type :: PARTNERS
include get_template_directory() . '/inc/partners-post-types.php';

// Shortcode :: AOS
include get_template_directory() . '/inc/aos-animation.php'; 
include get_template_directory() . '/inc/breadcrumbs.php'; 




// include get_template_directory() . '/inc/destinations-posts.php';

include get_template_directory() . '/inc/tour-packages-shortcode.php';
include get_template_directory() . '/inc/packages.php';
include get_template_directory() . '/inc/all_news-shortcode.php';


//Duplicate Pages
include get_template_directory() . '/inc/duplicate-page.php'; 

//ALL Associates Shortcode
include get_template_directory() . '/inc/all_partners-page.php'; 





// Add Video Subtitle Meta Box
function explore_uganda_add_video_subtitle_meta_box() {
    add_meta_box(
        'video_subtitle_box',
        'Video Subtitle',
        'explore_uganda_video_subtitle_meta_box_html',
        'text_slider',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes', 'explore_uganda_add_video_subtitle_meta_box');

// Meta box HTML
function explore_uganda_video_subtitle_meta_box_html($post) {
    $value = get_post_meta($post->ID, '_video_subtitle', true);
    ?>
    <input type="text" name="video_subtitle" value="<?php echo esc_attr($value); ?>" style="width:100%; font-size:18px; padding:8px;">
    <?php
}

// Save the field
function explore_uganda_save_video_subtitle_meta_box($post_id) {
    if (array_key_exists('video_subtitle', $_POST)) {
        update_post_meta(
            $post_id,
            '_video_subtitle',
            sanitize_text_field($_POST['video_subtitle'])
        );
    }
}
add_action('save_post', 'explore_uganda_save_video_subtitle_meta_box');


//ALL Single Gallery Shortcode
function load_custom_single_template($template) {
    if (is_single() && in_category('photo-gallery')) {
        $new_template = locate_template(array('single-gallery.php'));
        if ($new_template) {
            return $new_template;
        }
    }
    if (is_singular('events')) {
        $eve_template = locate_template(array('single-event.php'));
        if ($eve_template) {
            return $eve_template;
        }
    }
    return $template;
}
add_filter('single_template', 'load_custom_single_template');

// //FORM PROCESSOR
// include get_template_directory() . '/inc/process-form.php'; 

//BACK BUTTON (previous page)
add_action( 'back_button', 'kipya_back_button' );
function kipya_back_button() {
    if ( wp_get_referer() ) {
        $back_text = __( '&laquo; BACK' );
        $button    = "\n<button id='my-back-button' class='btn button my-back-button' onclick='javascript:history.back()'>$back_text</button>";
        echo ( $button );
    }
}

// Pagination for paged posts, Page 1, Page 2, Page 3, with Next and Previous Links, No plugin
function html5wp_pagination()
{
    global $wp_query;
    $big = 999999999;
    echo paginate_links(array(
        'base' => str_replace($big, '%#%', get_pagenum_link($big)),
        'format' => '?paged=%#%',
        'current' => max(1, get_query_var('paged')),
        'total' => $wp_query->max_num_pages
    ));
}
function enqueue_datatables_assets() {
    wp_enqueue_style('datatables-css', 'https://cdn.datatables.net/1.11.3/css/jquery.dataTables.min.css');
    wp_enqueue_script('datatables-js', 'https://cdn.datatables.net/1.11.3/js/jquery.dataTables.min.js', array('jquery'), null, true);
}
add_action('wp_enqueue_scripts', 'enqueue_datatables_assets');

// Custom Excerpts
function html5wp_index($length) // Create 20 Word Callback for Index page Excerpts, call using html5wp_excerpt('html5wp_index');
{
    return 50;
}

// Create 40 Word Callback for Custom Post Excerpts, call using html5wp_excerpt('html5wp_custom_post');
function html5wp_custom_post($length)
{
    return 50;
}

//Allow upload of SVG Images
function enable_svg_support($mimes) {
    $mimes['svg'] = 'image/svg+xml';
    return $mimes;
}
add_filter('upload_mimes', 'enable_svg_support');

// Create the Custom Excerpts callback
function html5wp_excerpt($length_callback = '', $more_callback = '')
{
    global $post;
    if (function_exists($length_callback)) {
        add_filter('excerpt_length', $length_callback);
    }
    if (function_exists($more_callback)) {
        add_filter('excerpt_more', $more_callback);
    }
    $output = get_the_excerpt();
    $output = apply_filters('wptexturize', $output);
    $output = apply_filters('convert_chars', $output);
    $output = '<p>' . $output . '</p>';
    echo $output;
}

// Custom View Article link to Post
function html5_blank_view_article($more)
{
    global $post;
    return '... <a class="view-article" href="' . get_permalink($post->ID) . '">' . __('Read More', 'kipya') . '</a>';
}

// Remove Admin bar
function remove_admin_bar() {
    return false;
}
add_filter('show_admin_bar', 'remove_admin_bar');

// Remove 'text/css' from our enqueued stylesheet
function html5_style_remove($tag)
{
    return preg_replace('~\s+type=["\'][^"\']++["\']~', '', $tag);
}

function enqueue_destination_styles() {
    wp_enqueue_style(
        'destination-custom-css',
        get_stylesheet_directory_uri() . '/assets/css/custom.css',
        array(),
        filemtime(get_stylesheet_directory() . '/assets/css/custom.css')
    );
}
add_action('wp_enqueue_scripts', 'enqueue_destination_styles');

add_filter('single_template', function($template) {
    global $post;
    if ($post->post_type === 'destination') {
        return locate_template('single-destination.php');
    }
    return $template;
});


// Remove thumbnail width and height dimensions that prevent fluid images in the_thumbnail
function remove_thumbnail_dimensions( $html )
{
    $html = preg_replace('/(width|height)=\"\d*\"\s/', "", $html);
    return $html;
}
function load_bootstrap_icons_footer() {
    ?>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons/font/bootstrap-icons.css" type="text/css" />
    <?php
}
add_action('wp_footer', 'load_bootstrap_icons_footer');



// Create tour bookings table on theme activation
function create_tour_bookings_table() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'tour_bookings';
    
    // Check if table already exists
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            booking_number VARCHAR(50) NOT NULL,
            full_name VARCHAR(100) NOT NULL,
            email VARCHAR(100) NOT NULL,
            phone VARCHAR(30) NOT NULL,
            tour_package VARCHAR(255) NOT NULL,
            num_adults INT(11) DEFAULT 0,
            num_children INT(11) DEFAULT 0,
            num_ugandans INT(11) DEFAULT 0,
            tour_date DATE NOT NULL,
            total_cost DECIMAL(10,2) NOT NULL,
            status VARCHAR(20) DEFAULT 'pending',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY booking_number (booking_number),
            KEY email (email),
            KEY status (status),
            KEY tour_date (tour_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        error_log('Tour bookings table created successfully');
    }
}

// Hook to create table when theme is activated
add_action('after_switch_theme', 'create_tour_bookings_table');





//*********************** The Booking and Rating Posttype*****************************************************//



// Register Booking Custom Post Type
function register_booking_post_type() {
    $labels = array(
        'name' => 'Bookings',
        'singular_name' => 'Booking',
        'menu_name' => 'Bookings',
        'add_new' => 'Add New Booking',
        'add_new_item' => 'Add New Booking',
        'edit_item' => 'View Booking',
        'new_item' => 'New Booking',
        'view_item' => 'View Booking',
        'search_items' => 'Search Bookings',
        'not_found' => 'No bookings found',
        'not_found_in_trash' => 'No bookings found in Trash'
    );

    $args = array(
        'labels' => $labels,
        'public' => false,
        'show_ui' => true,
        'show_in_menu' => true,
        'capability_type' => 'post',
        'capabilities' => array(
            'create_posts' => false, // Removes "Add New"
        ),
        'map_meta_cap' => true,
        'hierarchical' => false,
        'supports' => array('title'),
        'menu_icon' => 'dashicons-calendar-alt',
    );

    register_post_type('booking', $args);
}
add_action('init', 'register_booking_post_type');

// Remove "Add New" from admin menu
function remove_booking_add_new() {
    remove_submenu_page('edit.php?post_type=booking', 'post-new.php?post_type=booking');
}
add_action('admin_menu', 'remove_booking_add_new');

// Make bookings non-editable but add completion actions
function make_bookings_view_only() {
    global $pagenow, $post;
    
    if ($pagenow == 'post.php' && $post->post_type == 'booking') {
        // Remove publish meta box
        remove_meta_box('submitdiv', 'booking', 'side');
        
        // Remove all other meta boxes except custom ones
        global $wp_meta_boxes;
        $keep_boxes = array('booking_details', 'booking_actions');
        
        foreach ($wp_meta_boxes['booking'] as $context => $priority_boxes) {
            foreach ($priority_boxes as $priority => $boxes) {
                foreach ($boxes as $id => $box) {
                    if (!in_array($id, $keep_boxes)) {
                        remove_meta_box($id, 'booking', $context);
                    }
                }
            }
        }
    }
}
add_action('admin_head', 'make_bookings_view_only');

// Add custom meta boxes for booking details and actions
function add_booking_meta_boxes() {
    add_meta_box(
        'booking_details',
        'Booking Details',
        'display_booking_details',
        'booking',
        'normal',
        'high'
    );
    
    add_meta_box(
        'booking_actions',
        'Booking Actions',
        'display_booking_actions',
        'booking',
        'side',
        'high'
    );
}
add_action('add_meta_boxes', 'add_booking_meta_boxes');

function display_booking_details($post) {
    // Get all booking meta
    $booking_number = get_post_meta($post->ID, '_booking_number', true);
    $customer_name = get_post_meta($post->ID, '_customer_name', true);
    $customer_email = get_post_meta($post->ID, '_customer_email', true);
    $customer_phone = get_post_meta($post->ID, '_customer_phone', true);
    $tour_package = get_post_meta($post->ID, '_tour_package', true);
    $num_adults = get_post_meta($post->ID, '_num_adults', true);
    $num_children = get_post_meta($post->ID, '_num_children', true);
    $num_ugandans = get_post_meta($post->ID, '_num_ugandans', true);
    $tour_date = get_post_meta($post->ID, '_tour_date', true);
    $tour_completed_date = get_post_meta($post->ID, '_tour_completed_date', true);
    $total_cost = get_post_meta($post->ID, '_total_cost', true);
    $original_cost = get_post_meta($post->ID, '_original_cost', true);
    $discount_amount = get_post_meta($post->ID, '_discount_amount', true);
    $booking_status = get_post_meta($post->ID, '_booking_status', true);
    $deposit_amount = get_post_meta($post->ID, '_deposit_amount', true);
    $balance_amount = get_post_meta($post->ID, '_balance_amount', true);
    $tour_rating = get_post_meta($post->ID, '_tour_rating', true);
    
    echo '<div class="booking-details">';
    echo '<table class="widefat">';
    echo '<tr><td><strong>Booking Number:</strong></td><td>' . esc_html($booking_number) . '</td></tr>';
    echo '<tr><td><strong>Customer Name:</strong></td><td>' . esc_html($customer_name) . '</td></tr>';
    echo '<tr><td><strong>Email:</strong></td><td>' . esc_html($customer_email) . '</td></tr>';
    echo '<tr><td><strong>Phone:</strong></td><td>' . esc_html($customer_phone) . '</td></tr>';
    echo '<tr><td><strong>Tour Package:</strong></td><td>' . esc_html($tour_package) . '</td></tr>';
    echo '<tr><td><strong>Scheduled Tour Date:</strong></td><td>' . esc_html($tour_date) . '</td></tr>';
    echo '<tr><td><strong>Actual Completion Date:</strong></td><td>' . ($tour_completed_date ? esc_html($tour_completed_date) : '<em>Not completed yet</em>') . '</td></tr>';
    echo '<tr><td><strong>Participants:</strong></td><td>';
    echo 'International Adults: ' . esc_html($num_adults);
    if ($num_ugandans > 0) echo ', Ugandan Residents: ' . esc_html($num_ugandans);
    if ($num_children > 0) echo ', Children: ' . esc_html($num_children);
    echo '</td></tr>';
    echo '<tr><td><strong>Total Cost:</strong></td><td>$' . number_format($total_cost, 2) . '</td></tr>';
    echo '<tr><td><strong>Deposit (50%):</strong></td><td>$' . number_format($deposit_amount, 2) . '</td></tr>';
    echo '<tr><td><strong>Balance:</strong></td><td>$' . number_format($balance_amount, 2) . '</td></tr>';
    echo '<tr><td><strong>Booking Status:</strong></td><td>' . esc_html($booking_status) . '</td></tr>';
    
    // Interactive rating stars
echo '<tr><td><strong>Customer Rating:</strong></td><td>';
echo '<div class="star-rating" data-post-id="' . esc_attr($post->ID) . '" data-current-rating="' . esc_attr($tour_rating ? $tour_rating : 0) . '">';
for ($i = 1; $i <= 5; $i++) {
    $filled = ($tour_rating && $i <= $tour_rating) ? 'filled' : '';
    echo '<span class="star ' . $filled . '" data-value="' . $i . '">&#9733;</span>';
}
echo '</div>';
$tour_review = get_post_meta($post->ID, '_tour_review', true);
if ($tour_review) {
    echo '<br><em>"' . esc_html($tour_review) . '"</em>';
}
echo '</td></tr>';

    
    echo '</table>';
    echo '</div>';
    
    // Add some CSS
    echo '<style>
        .booking-details table { width: 100%; border-collapse: collapse; }
        .booking-details td { padding: 12px; border-bottom: 1px solid #eee; }
        .booking-details td:first-child { width: 200px; font-weight: bold; }
    </style>';
    
    // Add interactive star CSS and JS
echo '<style>
.star-rating {
    display: inline-flex;
    flex-direction: row; /* ensures stars fill left-to-right */
    cursor: pointer;
}
.star {
    font-size: 24px;
    color: #ccc;
    transition: color 0.2s;
    padding: 2px;
}
.star.filled {
    color: #ffcc00; /* gold when filled */
}
.star:hover,
.star:hover ~ .star {
    color: #ccc; /* disable weird hover inheritance */
}
.star:hover,
.star:hover ~ .star {
    color: #ffeb80;
}

</style>

<script>
document.addEventListener("DOMContentLoaded", function() {
    document.querySelectorAll(".star-rating").forEach(rating => {
        const stars = rating.querySelectorAll(".star");
        const postId = rating.dataset.postId;
        let currentRating = parseInt(rating.dataset.currentRating) || 0;

        // Initialize stars based on saved rating
        updateStars(stars, currentRating);

        stars.forEach(star => {
            star.addEventListener("mouseenter", function() {
                updateStars(stars, this.dataset.value);
            });

            star.addEventListener("mouseleave", function() {
                updateStars(stars, currentRating);
            });

            star.addEventListener("click", function() {
                currentRating = parseInt(this.dataset.value);
                updateStars(stars, currentRating);

                // Save rating via AJAX
                const data = new FormData();
                data.append("action", "save_tour_rating");
                data.append("post_id", postId);
                data.append("rating", currentRating);
                data.append("_ajax_nonce", "' . wp_create_nonce('save_tour_rating_nonce') . '");

                fetch(ajaxurl, {
                    method: "POST",
                    body: data
                })
                .then(res => res.text())
                .then(response => console.log("Saved rating:", response))
                .catch(err => console.error("Error:", err));
            });
        });
    });

    function updateStars(stars, rating) {
        stars.forEach(star => {
            const value = parseInt(star.dataset.value);
            star.classList.toggle("filled", value <= rating);
        });
    }
});

</script>';

}

// Replace the existing display_booking_actions function with this:
function display_booking_actions($post) {
    $tour_completed_date = get_post_meta($post->ID, '_tour_completed_date', true);
    $tour_rating = get_post_meta($post->ID, '_tour_rating', true);
    $rating_token = get_post_meta($post->ID, '_rating_token', true);
    
    echo '<div class="booking-actions">';
    
    if (!$tour_completed_date) {
        echo '<div class="action-section">';
        echo '<h4>Tour Completion</h4>';
        echo '<p>Mark this tour as completed to send rating request:</p>';
        echo '<form method="post" style="margin-top: 10px;">';
        wp_nonce_field('mark_tour_completed', 'tour_completion_nonce');
        echo '<input type="hidden" name="mark_completed" value="' . $post->ID . '">';
        echo '<button type="submit" class="button button-primary" name="complete_tour" style="width:100%;">✅ Mark Tour as Completed</button>';
        echo '</form>';
        echo '</div>';
    } else {
        echo '<div class="action-section completed">';
        echo '<h4>✅ Tour Completed</h4>';
        echo '<p>Completed on: ' . date('M j, Y', strtotime($tour_completed_date)) . '</p>';
        
        if ($tour_rating) {
            echo '<div class="rating-display">';
            echo '<p><strong>Current Rating:</strong> ' . str_repeat('⭐', $tour_rating) . ' (' . $tour_rating . '/5)</p>';
            
            // Add delete button
            $delete_url = admin_url('post.php?post=' . $post->ID . '&action=edit&delete_rating=' . $post->ID);
            $delete_url = wp_nonce_url($delete_url, 'delete_rating_' . $post->ID);
            
            echo '<a href="' . $delete_url . '" class="button button-secondary" style="width:100%; background:#dc3232; color:white; text-align:center; display:block; margin-top: 10px;" onclick="return confirm(\'Are you sure you want to delete this rating?\\\\n\\\\nThis cannot be undone.\')">🗑️ Delete Rating</a>';
            echo '</div>';
        } elseif (!$tour_rating && !$rating_token) {
            echo '<form method="post" style="margin-top: 10px;">';
            wp_nonce_field('send_rating_request', 'rating_request_nonce');
            echo '<input type="hidden" name="send_rating_request" value="' . $post->ID . '">';
            echo '<button type="submit" class="button" name="request_rating" style="width:100%; background:#2196F3; color:white;">📧 Send Rating Request</button>';
            echo '</form>';
        }
        echo '</div>';
    }
    
    echo '</div>';
    
    echo '<style>
        .booking-actions .action-section { padding: 15px; margin: -12px -12px 12px -12px; }
        .booking-actions .action-section:not(:last-child) { border-bottom: 1px solid #ccd0d4; }
        .booking-actions .action-section.completed { background: #f0f9ff; }
        .booking-actions .action-section.rated { background: #f0fff4; }
        .booking-actions h4 { margin: 0 0 8px 0; }
    </style>';
}

// Add rating deletion tool
function add_rating_deletion_tool() {
    if (isset($_GET['delete_rating']) && current_user_can('manage_options')) {
        $booking_id = intval($_GET['delete_rating']);
        
        if ($booking_id > 0) {
            // Verify nonce for security
            if (!wp_verify_nonce($_GET['_wpnonce'], 'delete_rating_' . $booking_id)) {
                wp_die('Security check failed');
            }
            
            // Delete all rating-related meta
            delete_post_meta($booking_id, '_tour_rating');
            delete_post_meta($booking_id, '_tour_review');
            delete_post_meta($booking_id, '_rating_date');
            delete_post_meta($booking_id, '_rating_token');
            
            // Redirect back with success message
            wp_redirect(admin_url('post.php?post=' . $booking_id . '&action=edit&rating_deleted=1'));
            exit;
        }
    }
}
add_action('init', 'add_rating_deletion_tool');

// Add admin notice when rating is deleted
function rating_deleted_notice() {
    if (isset($_GET['rating_deleted'])) {
        echo '<div class="notice notice-success is-dismissible"><p>🗑️ Rating deleted successfully!</p></div>';
    }
}
add_action('admin_notices', 'rating_deleted_notice');

// Handle booking actions
function handle_booking_actions() {
    if (isset($_POST['complete_tour']) && wp_verify_nonce($_POST['tour_completion_nonce'], 'mark_tour_completed')) {
        $booking_id = intval($_POST['mark_completed']);
        if ($booking_id > 0) {
            update_post_meta($booking_id, '_tour_completed_date', current_time('mysql'));
            update_post_meta($booking_id, '_booking_status', 'completed');
            
            // Send rating request immediately
            send_rating_request_email($booking_id);
            
            // Add admin notice
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>✅ Tour marked as completed and rating request sent!</p></div>';
            });
        }
    }
    
    if (isset($_POST['request_rating']) && wp_verify_nonce($_POST['rating_request_nonce'], 'send_rating_request')) {
        $booking_id = intval($_POST['send_rating_request']);
        if ($booking_id > 0) {
            send_rating_request_email($booking_id);
            
            // Add admin notice
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>📧 Rating request sent to customer!</p></div>';
            });
        }
    }
}
add_action('admin_init', 'handle_booking_actions');

// Register rating custom fields for bookings
function register_rating_fields() {
    register_meta('post', '_tour_rating', array(
        'type' => 'number',
        'description' => 'Tour rating from 1-5',
        'single' => true,
        'show_in_rest' => true,
    ));
    
    register_meta('post', '_tour_review', array(
        'type' => 'string',
        'description' => 'Tour review text',
        'single' => true,
        'show_in_rest' => true,
    ));
    
    register_meta('post', '_rating_date', array(
        'type' => 'string',
        'description' => 'Date when rating was submitted',
        'single' => true,
        'show_in_rest' => true,
    ));
    
    register_meta('post', '_tour_completed_date', array(
        'type' => 'string',
        'description' => 'Date when tour was actually completed',
        'single' => true,
        'show_in_rest' => true,
    ));
    
    register_meta('post', '_rating_token', array(
        'type' => 'string',
        'description' => 'Secure token for rating requests',
        'single' => true,
        'show_in_rest' => true,
    ));
}
add_action('init', 'register_rating_fields');

// Handle AJAX rating save
function save_tour_rating() {
    check_ajax_referer('save_tour_rating_nonce');

    $post_id = intval($_POST['post_id']);
    $rating = intval($_POST['rating']);

    if ($post_id > 0 && $rating >= 1 && $rating <= 5 && current_user_can('edit_post', $post_id)) {
        update_post_meta($post_id, '_tour_rating', $rating);
        wp_send_json_success("Rating saved successfully: " . $rating);
    } else {
        wp_send_json_error("Invalid data or permission denied");
    }
}
add_action('wp_ajax_save_tour_rating', 'save_tour_rating');


// Get average rating for a tour package
function get_tour_package_average_rating($tour_package_name) {
    $bookings = get_posts(array(
        'post_type' => 'booking',
        'posts_per_page' => -1,
        'meta_query' => array(
            array(
                'key' => '_tour_package',
                'value' => $tour_package_name,
                'compare' => '='
            ),
            array(
                'key' => '_tour_rating',
                'compare' => 'EXISTS'
            )
        )
    ));
    
    if (empty($bookings)) return 0;
    
    $total_rating = 0;
    foreach ($bookings as $booking) {
        $rating = get_post_meta($booking->ID, '_tour_rating', true);
        $total_rating += floatval($rating);
    }
    
    return round($total_rating / count($bookings), 1);
}

// Get review count for a tour package
function get_tour_package_review_count($tour_package_name) {
    $bookings = get_posts(array(
        'post_type' => 'booking',
        'posts_per_page' => -1,
        'meta_query' => array(
            array(
                'key' => '_tour_package',
                'value' => $tour_package_name,
                'compare' => '='
            ),
            array(
                'key' => '_tour_rating',
                'compare' => 'EXISTS'
            )
        )
    ));
    
    return count($bookings);
}

// Send rating request email after tour completion
function send_rating_request_email($booking_id) {
    $customer_email = get_post_meta($booking_id, '_customer_email', true);
    $customer_name = get_post_meta($booking_id, '_customer_name', true);
    $tour_package = get_post_meta($booking_id, '_tour_package', true);
    $tour_date = get_post_meta($booking_id, '_tour_date', true);
    $booking_number = get_post_meta($booking_id, '_booking_number', true);
    
    // Create unique rating link
    $rating_token = wp_generate_password(32, false);
    update_post_meta($booking_id, '_rating_token', $rating_token);
    
    $rating_link = home_url('/rate-your-tour/?booking=' . $booking_id . '&token=' . $rating_token);
    
    $subject = "How was your $tour_package tour?";
    
    $message = "
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <title>Rate Your Tour Experience</title>
        <style>
            body { margin: 0; padding: 0; background: #f8f8f8; font-family: Arial, sans-serif; }
            .container { max-width: 600px; margin: 0 auto; background: #ffffff; border-radius: 10px; overflow: hidden; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .header { background: linear-gradient(135deg, #710b0d, #710b0d); color: #ffffff; text-align: center; padding: 40px 20px; }
            .content { padding: 30px; color: #333333; }
            .rating-link { display: inline-block; background: #710b0d; color: #fff !important; padding: 15px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; font-size: 16px; }
            .footer { background: #710b0d; color: #fff; text-align: center; padding: 25px 20px; font-size: 14px; }
        </style>
    </head>
    <body>
        <table class='container'>
            <tr>
                <td class='header'>
                    <h1>How Was Your Tour?</h1>
                    <p>We'd love to hear about your experience!</p>
                </td>
            </tr>
            <tr>
                <td class='content'>
                    <p>Dear $customer_name,</p>
                    <p>We hope you enjoyed your <strong>$tour_package</strong> tour on $tour_date!</p>
                    <p>Your feedback helps us improve our services and helps other travelers make informed decisions.</p>
                    
                    <div style='text-align: center;'>
                        <a href='$rating_link' class='rating-link'>Rate Your Tour Experience</a>
                    </div>
                    
                    <p>It only takes 2 minutes to share your experience!</p>
                    
                    <p>Thank you for choosing Explore Uganda!</p>
                </td>
            </tr>
            <tr>
                <td class='footer'>
                    <p>This is an automated message from Explore Uganda.</p>
                </td>
            </tr>
        </table>
    </body>
    </html>
    ";
    
    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: Explore Uganda <info@exploreug.com>'
    );
    
    return wp_mail($customer_email, $subject, $message, $headers);
}

// Schedule rating emails for completed tours only
function schedule_rating_emails() {
    if (!wp_next_scheduled('check_completed_tours')) {
        wp_schedule_event(time(), 'daily', 'check_completed_tours');
    }
}
add_action('wp', 'schedule_rating_emails');

// Check for ACTUALLY completed tours and send rating requests
function check_completed_tours_for_ratings() {
    $bookings = get_posts(array(
        'post_type' => 'booking',
        'posts_per_page' => -1,
        'meta_query' => array(
            array(
                'key' => '_tour_completed_date', // Only tours that were actually completed
                'value' => date('Y-m-d', strtotime('-3 days')),
                'compare' => '<=',
                'type' => 'DATE'
            ),
            array(
                'key' => '_rating_token',
                'compare' => 'NOT EXISTS' // Haven't sent rating request yet
            ),
            array(
                'key' => '_tour_rating',
                'compare' => 'NOT EXISTS' // Haven't rated yet
            )
        )
    ));
    
    foreach ($bookings as $booking) {
        send_rating_request_email($booking->ID);
    }
}
add_action('check_completed_tours', 'check_completed_tours_for_ratings');

// Add ratings column to bookings admin
function add_ratings_column($columns) {
    $columns['tour_status'] = 'Tour Status';
    $columns['rating'] = 'Rating';
    return $columns;
}
add_filter('manage_booking_posts_columns', 'add_ratings_column');

function show_ratings_column($column, $post_id) {
    if ($column === 'rating') {
        $rating = get_post_meta($post_id, '_tour_rating', true);
        if (!empty($rating)) {
            echo str_repeat('⭐', $rating) . ' (' . $rating . '/5)';
        } else {
            $rating_token = get_post_meta($post_id, '_rating_token', true);
            if ($rating_token) {
                echo '📧 Request Sent';
            } else {
                echo 'Not Rated';
            }
        }
    }
    
    if ($column === 'tour_status') {
        $completed_date = get_post_meta($post_id, '_tour_completed_date', true);
        if ($completed_date) {
            echo '✅ Completed';
        } else {
            $tour_date = get_post_meta($post_id, '_tour_date', true);
            $today = date('Y-m-d');
            if ($tour_date < $today) {
                echo '❌ Not Completed';
            } else {
                echo '⏰ Upcoming';
            }
        }
    }
}
add_action('manage_booking_posts_custom_column', 'show_ratings_column', 10, 2);

// Generate star rating HTML
function generate_star_rating($rating) {
    $full_stars = floor($rating);
    $half_star = ($rating - $full_stars) >= 0.5;
    $empty_stars = 5 - $full_stars - ($half_star ? 1 : 0);
    
    $stars = '';
    
    // Full stars
    for ($i = 0; $i < $full_stars; $i++) {
        $stars .= '⭐';
    }
    
    // Half star
    if ($half_star) {
        $stars .= '⭐';
    }
    
    // Empty stars
    for ($i = 0; $i < $empty_stars; $i++) {
        $stars .= '☆';
    }
    
    return $stars;
}

// Display recent reviews for a tour
function display_recent_reviews($tour_package_name, $limit = 3) {
    $bookings = get_posts(array(
        'post_type' => 'booking',
        'posts_per_page' => $limit,
        'meta_query' => array(
            array(
                'key' => '_tour_package',
                'value' => $tour_package_name,
                'compare' => '='
            ),
            array(
                'key' => '_tour_rating',
                'compare' => 'EXISTS'
            )
        ),
        'orderby' => 'date',
        'order' => 'DESC'
    ));
    
    foreach ($bookings as $booking) {
        $rating = get_post_meta($booking->ID, '_tour_rating', true);
        $review = get_post_meta($booking->ID, '_tour_review', true);
        $customer_name = get_post_meta($booking->ID, '_customer_name', true);
        $rating_date = get_post_meta($booking->ID, '_rating_date', true);
        $tour_date = get_post_meta($booking->ID, '_tour_date', true);
        
        // Anonymize name
        $names = explode(' ', $customer_name);
        $display_name = $names[0] . (isset($names[1]) ? ' ' . substr($names[1], 0, 1) . '.' : '');
        
        echo '
        <div class="review-item">
            <div class="review-header">
                <div class="reviewer-info">
                    <div class="reviewer-avatar">' . substr($display_name, 0, 1) . '</div>
                    <div class="reviewer-details">
                        <strong class="reviewer-name">' . esc_html($display_name) . '</strong>
                        <span class="review-date">Reviewed on ' . date('F j, Y', strtotime($rating_date)) . '</span>
                    </div>
                </div>
                <div class="review-rating">' . generate_star_rating($rating) . '</div>
            </div>
            <div class="review-content">
                <p>' . esc_html($review ?: 'Rated this tour ' . $rating . ' out of 5 stars.') . '</p>
                <div class="tour-date">Tour date: ' . date('F j, Y', strtotime($tour_date)) . '</div>
            </div>
        </div>';
    }
}

// Anonymize customer names for public display
function anonymize_name($full_name) {
    $names = explode(' ', $full_name);
    $first_name = $names[0];
    $last_initial = isset($names[1]) ? substr($names[1], 0, 1) . '.' : '';
    
    return $first_name . ' ' . $last_initial;
}

// Add completion status to existing bookings (one-time migration)
function migrate_existing_bookings_completion() {
    $bookings = get_posts(array(
        'post_type' => 'booking',
        'posts_per_page' => -1,
        'meta_query' => array(
            array(
                'key' => '_tour_completed_date',
                'compare' => 'NOT EXISTS'
            )
        )
    ));
    
    foreach ($bookings as $booking) {
        $tour_date = get_post_meta($booking->ID, '_tour_date', true);
        $today = date('Y-m-d');
        
        // If tour date is more than 3 days ago, assume it was completed
        if ($tour_date && $tour_date < date('Y-m-d', strtotime('-3 days'))) {
            update_post_meta($booking->ID, '_tour_completed_date', $tour_date);
        }
    }
}
// Uncomment the line below to run migration once, then comment it out again
add_action('init', 'migrate_existing_bookings_completion');





?>