<?php
/*
Plugin Name: Single Destination Widget
Description: A comprehensive widget for displaying destination information with multiple sections.
Version: 1.0
Author: Melissa Lokoroma
*/

// Register the widget
function register_single_destination_widget() {
    register_widget('Single_Destination_Widget');
}
add_action('widgets_init', 'register_single_destination_widget');

class Single_Destination_Widget extends WP_Widget {

    function __construct() {
        parent::__construct(
            'single_destination_widget',
            __('Single Destination Widget', 'text_domain'),
            array('description' => __('Displays comprehensive destination information with multiple sections', 'text_domain'))
        );
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Enqueue frontend scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
    }
    
    // Enqueue admin scripts
    public function enqueue_admin_scripts() {
        wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css');
        
        // WordPress media uploader
        wp_enqueue_media();
    }
    
    // Enqueue frontend scripts
    public function enqueue_frontend_scripts() {
        wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css');
    }

    // Widget backend form
    public function form($instance) {
        $defaults = array(
            'title' => '',
            'destination_name' => '',
            'destination_excerpt' => '',
            'featured_image' => '',
            'show_tour_overview' => true,
            'tour_overview_title' => 'Tour Overview',
            'tour_overview_content' => '',
            'show_itinerary' => true,
            'itinerary_title' => 'Itinerary',
            'itinerary_days' => array(),
            'show_price_details' => true,
            'price_includes_title' => 'Price Includes',
            'price_includes' => array(),
            'price_excludes_title' => 'Price Excludes',
            'price_excludes' => array(),
            'show_additional_sections' => true,
            'additional_sections' => array(),
            'sidebar_title' => 'Quick Facts',
            'sidebar_items' => array(),
            'show_whatsapp_button' => true,
            'whatsapp_number' => '',
            'whatsapp_message' => 'I\'m interested in this destination',
            'inquiry_button_text' => 'Inquire Now',
            'inquiry_button_link' => '#',
            'show_gallery' => true,
            'gallery_title' => 'Gallery',
            'gallery_images' => array()
        );
        
        $instance = wp_parse_args((array) $instance, $defaults);
        ?>
        
        <div class="single-destination-widget-admin">
            <!-- Basic Settings Tab -->
            <div class="tab-content active" id="tab-basic">
                <p>
                    <label for="<?php echo $this->get_field_id('title'); ?>"><?php _e('Widget Title:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" 
                           name="<?php echo $this->get_field_name('title'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['title']); ?>" />
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('destination_name'); ?>"><?php _e('Destination Name:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('destination_name'); ?>" 
                           name="<?php echo $this->get_field_name('destination_name'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['destination_name']); ?>" />
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('destination_excerpt'); ?>"><?php _e('Destination Excerpt:'); ?></label>
                    <textarea class="widefat" id="<?php echo $this->get_field_id('destination_excerpt'); ?>" 
                              name="<?php echo $this->get_field_name('destination_excerpt'); ?>" 
                              rows="3"><?php echo esc_textarea($instance['destination_excerpt']); ?></textarea>
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('featured_image'); ?>"><?php _e('Featured Image:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('featured_image'); ?>" 
                           name="<?php echo $this->get_field_name('featured_image'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['featured_image']); ?>" />
                    <button class="upload-image-button button"><?php _e('Upload Image'); ?></button>
                    <?php if ($instance['featured_image']) : ?>
                        <div class="image-preview">
                            <img src="<?php echo esc_attr($instance['featured_image']); ?>" style="max-width: 100%; height: auto; margin-top: 10px;" />
                        </div>
                    <?php endif; ?>
                </p>
            </div>
            
            <!-- Tour Overview Tab -->
            <div class="tab-content" id="tab-tour-overview">
                <p>
                    <label>
                        <input type="checkbox" id="<?php echo $this->get_field_id('show_tour_overview'); ?>" 
                               name="<?php echo $this->get_field_name('show_tour_overview'); ?>" 
                               value="1" <?php checked($instance['show_tour_overview'], 1); ?> />
                        <?php _e('Show Tour Overview Section'); ?>
                    </label>
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('tour_overview_title'); ?>"><?php _e('Tour Overview Title:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('tour_overview_title'); ?>" 
                           name="<?php echo $this->get_field_name('tour_overview_title'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['tour_overview_title']); ?>" />
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('tour_overview_content'); ?>"><?php _e('Tour Overview Content:'); ?></label>
                    <textarea class="widefat" id="<?php echo $this->get_field_id('tour_overview_content'); ?>" 
                              name="<?php echo $this->get_field_name('tour_overview_content'); ?>" 
                              rows="8"><?php echo esc_textarea($instance['tour_overview_content']); ?></textarea>
                </p>
            </div>
            
            <!-- Itinerary Tab -->
            <div class="tab-content" id="tab-itinerary">
                <p>
                    <label>
                        <input type="checkbox" id="<?php echo $this->get_field_id('show_itinerary'); ?>" 
                               name="<?php echo $this->get_field_name('show_itinerary'); ?>" 
                               value="1" <?php checked($instance['show_itinerary'], 1); ?> />
                        <?php _e('Show Itinerary Section'); ?>
                    </label>
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('itinerary_title'); ?>"><?php _e('Itinerary Section Title:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('itinerary_title'); ?>" 
                           name="<?php echo $this->get_field_name('itinerary_title'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['itinerary_title']); ?>" />
                </p>
                
                <div class="itinerary-days-container">
                    <h4><?php _e('Itinerary Days:'); ?></h4>
                    
                    <?php if (!empty($instance['itinerary_days'])) : ?>
                        <?php foreach ($instance['itinerary_days'] as $index => $day) : ?>
                            <div class="itinerary-day">
                                <button type="button" class="remove-day button">Remove Day</button>
                                <p>
                                    <label><?php _e('Day Title:'); ?></label>
                                    <input type="text" class="widefat" 
                                           name="<?php echo $this->get_field_name('itinerary_days'); ?>[<?php echo $index; ?>][title]" 
                                           value="<?php echo esc_attr($day['title']); ?>" />
                                </p>
                                <p>
                                    <label><?php _e('Day Description:'); ?></label>
                                    <textarea class="widefat" rows="4"
                                              name="<?php echo $this->get_field_name('itinerary_days'); ?>[<?php echo $index; ?>][description]"><?php echo esc_textarea($day['description']); ?></textarea>
                                </p>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <button type="button" class="add-itinerary-day button"><?php _e('Add Day'); ?></button>
            </div>
            
            <!-- Price Details Tab -->
            <div class="tab-content" id="tab-price-details">
                <p>
                    <label>
                        <input type="checkbox" id="<?php echo $this->get_field_id('show_price_details'); ?>" 
                               name="<?php echo $this->get_field_name('show_price_details'); ?>" 
                               value="1" <?php checked($instance['show_price_details'], 1); ?> />
                        <?php _e('Show Price Details Section'); ?>
                    </label>
                </p>
                
                <div class="price-details-grid">
                    <div class="price-includes">
                        <h4><?php _e('Price Includes:'); ?></h4>
                        
                        <p>
                            <label for="<?php echo $this->get_field_id('price_includes_title'); ?>"><?php _e('Includes Title:'); ?></label>
                            <input class="widefat" id="<?php echo $this->get_field_id('price_includes_title'); ?>" 
                                   name="<?php echo $this->get_field_name('price_includes_title'); ?>" type="text" 
                                   value="<?php echo esc_attr($instance['price_includes_title']); ?>" />
                        </p>
                        
                        <div class="includes-items">
                            <?php if (!empty($instance['price_includes'])) : ?>
                                <?php foreach ($instance['price_includes'] as $index => $item) : ?>
                                    <div class="price-item">
                                        <input type="text" class="widefat" 
                                               name="<?php echo $this->get_field_name('price_includes'); ?>[]" 
                                               value="<?php echo esc_attr($item); ?>" />
                                        <button type="button" class="remove-item button">Remove</button>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        
                        <button type="button" class="add-include-item button"><?php _e('Add Include Item'); ?></button>
                    </div>
                    
                    <div class="price-excludes">
                        <h4><?php _e('Price Excludes:'); ?></h4>
                        
                        <p>
                            <label for="<?php echo $this->get_field_id('price_excludes_title'); ?>"><?php _e('Excludes Title:'); ?></label>
                            <input class="widefat" id="<?php echo $this->get_field_id('price_excludes_title'); ?>" 
                                   name="<?php echo $this->get_field_name('price_excludes_title'); ?>" type="text" 
                                   value="<?php echo esc_attr($instance['price_excludes_title']); ?>" />
                        </p>
                        
                        <div class="excludes-items">
                            <?php if (!empty($instance['price_excludes'])) : ?>
                                <?php foreach ($instance['price_excludes'] as $index => $item) : ?>
                                    <div class="price-item">
                                        <input type="text" class="widefat" 
                                               name="<?php echo $this->get_field_name('price_excludes'); ?>[]" 
                                               value="<?php echo esc_attr($item); ?>" />
                                        <button type="button" class="remove-item button">Remove</button>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        
                        <button type="button" class="add-exclude-item button"><?php _e('Add Exclude Item'); ?></button>
                    </div>
                </div>
            </div>
            
            <!-- Additional Sections Tab -->
            <div class="tab-content" id="tab-additional-sections">
                <p>
                    <label>
                        <input type="checkbox" id="<?php echo $this->get_field_id('show_additional_sections'); ?>" 
                               name="<?php echo $this->get_field_name('show_additional_sections'); ?>" 
                               value="1" <?php checked($instance['show_additional_sections'], 1); ?> />
                        <?php _e('Show Additional Sections'); ?>
                    </label>
                </p>
                
                <div class="additional-sections-container">
                    <?php if (!empty($instance['additional_sections'])) : ?>
                        <?php foreach ($instance['additional_sections'] as $index => $section) : ?>
                            <div class="additional-section">
                                <button type="button" class="remove-section button">Remove Section</button>
                                <p>
                                    <label><?php _e('Section Title:'); ?></label>
                                    <input type="text" class="widefat" 
                                           name="<?php echo $this->get_field_name('additional_sections'); ?>[<?php echo $index; ?>][title]" 
                                           value="<?php echo esc_attr($section['title']); ?>" />
                                </p>
                                <p>
                                    <label><?php _e('Section Content:'); ?></label>
                                    <textarea class="widefat" rows="6"
                                              name="<?php echo $this->get_field_name('additional_sections'); ?>[<?php echo $index; ?>][content]"><?php echo esc_textarea($section['content']); ?></textarea>
                                </p>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <button type="button" class="add-additional-section button"><?php _e('Add Section'); ?></button>
            </div>
            
            <!-- Sidebar Tab -->
            <div class="tab-content" id="tab-sidebar">
                <p>
                    <label for="<?php echo $this->get_field_id('sidebar_title'); ?>"><?php _e('Sidebar Title:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('sidebar_title'); ?>" 
                           name="<?php echo $this->get_field_name('sidebar_title'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['sidebar_title']); ?>" />
                </p>
                
                <div class="sidebar-items-container">
                    <?php if (!empty($instance['sidebar_items'])) : ?>
                        <?php foreach ($instance['sidebar_items'] as $index => $item) : ?>
                            <div class="sidebar-item">
                                <button type="button" class="remove-sidebar-item button">Remove Item</button>
                                <p>
                                    <label><?php _e('Item Label:'); ?></label>
                                    <input type="text" class="widefat" 
                                           name="<?php echo $this->get_field_name('sidebar_items'); ?>[<?php echo $index; ?>][label]" 
                                           value="<?php echo esc_attr($item['label']); ?>" />
                                </p>
                                <p>
                                    <label><?php _e('Item Value:'); ?></label>
                                    <input type="text" class="widefat" 
                                           name="<?php echo $this->get_field_name('sidebar_items'); ?>[<?php echo $index; ?>][value]" 
                                           value="<?php echo esc_attr($item['value']); ?>" />
                                </p>
                                <p>
                                    <label><?php _e('Icon (Font Awesome class):'); ?></label>
                                    <input type="text" class="widefat" 
                                           name="<?php echo $this->get_field_name('sidebar_items'); ?>[<?php echo $index; ?>][icon]" 
                                           value="<?php echo esc_attr($item['icon']); ?>" placeholder="fas fa-calendar" />
                                </p>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <button type="button" class="add-sidebar-item button"><?php _e('Add Sidebar Item'); ?></button>
                
                <p>
                    <label for="<?php echo $this->get_field_id('inquiry_button_text'); ?>"><?php _e('Inquiry Button Text:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('inquiry_button_text'); ?>" 
                           name="<?php echo $this->get_field_name('inquiry_button_text'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['inquiry_button_text']); ?>" />
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('inquiry_button_link'); ?>"><?php _e('Inquiry Button Link:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('inquiry_button_link'); ?>" 
                           name="<?php echo $this->get_field_name('inquiry_button_link'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['inquiry_button_link']); ?>" />
                </p>
            </div>
            
            <!-- Gallery Tab -->
            <div class="tab-content" id="tab-gallery">
                <p>
                    <label>
                        <input type="checkbox" id="<?php echo $this->get_field_id('show_gallery'); ?>" 
                               name="<?php echo $this->get_field_name('show_gallery'); ?>" 
                               value="1" <?php checked($instance['show_gallery'], 1); ?> />
                        <?php _e('Show Gallery Section'); ?>
                    </label>
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('gallery_title'); ?>"><?php _e('Gallery Title:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('gallery_title'); ?>" 
                           name="<?php echo $this->get_field_name('gallery_title'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['gallery_title']); ?>" />
                </p>
                
                <div class="gallery-images-container">
                    <?php if (!empty($instance['gallery_images'])) : ?>
                        <?php foreach ($instance['gallery_images'] as $index => $image) : ?>
                            <div class="gallery-image">
                                <button type="button" class="remove-gallery-image button">Remove Image</button>
                                <input type="text" class="widefat image-url" 
                                       name="<?php echo $this->get_field_name('gallery_images'); ?>[]" 
                                       value="<?php echo esc_attr($image); ?>" />
                                <button type="button" class="upload-gallery-image button">Select Image</button>
                                <div class="image-preview">
                                    <?php if ($image) : ?>
                                        <img src="<?php echo esc_attr($image); ?>" style="max-width: 100%; height: auto; margin-top: 10px;" />
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <button type="button" class="add-gallery-image button"><?php _e('Add Image'); ?></button>
            </div>
            
            <!-- WhatsApp Tab -->
            <div class="tab-content" id="tab-whatsapp">
                <p>
                    <label>
                        <input type="checkbox" id="<?php echo $this->get_field_id('show_whatsapp_button'); ?>" 
                               name="<?php echo $this->get_field_name('show_whatsapp_button'); ?>" 
                               value="1" <?php checked($instance['show_whatsapp_button'], 1); ?> />
                        <?php _e('Show WhatsApp Button'); ?>
                    </label>
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('whatsapp_number'); ?>"><?php _e('WhatsApp Number:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('whatsapp_number'); ?>" 
                           name="<?php echo $this->get_field_name('whatsapp_number'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['whatsapp_number']); ?>" placeholder="e.g. 1234567890" />
                </p>
                
                <p>
                    <label for="<?php echo $this->get_field_id('whatsapp_message'); ?>"><?php _e('Default Message:'); ?></label>
                    <input class="widefat" id="<?php echo $this->get_field_id('whatsapp_message'); ?>" 
                           name="<?php echo $this->get_field_name('whatsapp_message'); ?>" type="text" 
                           value="<?php echo esc_attr($instance['whatsapp_message']); ?>" />
                </p>
            </div>
        </div>
        
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                // Tab switching
                $('.single-destination-widget-admin .tab-content').hide();
                $('.single-destination-widget-admin .tab-content:first').show();
                
                // Add itinerary day
                var itineraryDayIndex = <?php echo !empty($instance['itinerary_days']) ? count($instance['itinerary_days']) : 0; ?>;
                $('.add-itinerary-day').on('click', function() {
                    var html = '<div class="itinerary-day">' +
                        '<button type="button" class="remove-day button">Remove Day</button>' +
                        '<p><label><?php _e('Day Title:'); ?></label>' +
                        '<input type="text" class="widefat" name="<?php echo $this->get_field_name('itinerary_days'); ?>[' + itineraryDayIndex + '][title]" /></p>' +
                        '<p><label><?php _e('Day Description:'); ?></label>' +
                        '<textarea class="widefat" rows="4" name="<?php echo $this->get_field_name('itinerary_days'); ?>[' + itineraryDayIndex + '][description]"></textarea></p>' +
                        '</div>';
                    
                    $('.itinerary-days-container').append(html);
                    itineraryDayIndex++;
                });
                
                // Remove itinerary day
                $(document).on('click', '.remove-day', function() {
                    $(this).parent().remove();
                });
                
                // Add include item
                $('.add-include-item').on('click', function() {
                    var html = '<div class="price-item">' +
                        '<input type="text" class="widefat" name="<?php echo $this->get_field_name('price_includes'); ?>[]" />' +
                        '<button type="button" class="remove-item button">Remove</button>' +
                        '</div>';
                    
                    $('.includes-items').append(html);
                });
                
                // Add exclude item
                $('.add-exclude-item').on('click', function() {
                    var html = '<div class="price-item">' +
                        '<input type="text" class="widefat" name="<?php echo $this->get_field_name('price_excludes'); ?>[]" />' +
                        '<button type="button" class="remove-item button">Remove</button>' +
                        '</div>';
                    
                    $('.excludes-items').append(html);
                });
                
                // Remove price item
                $(document).on('click', '.remove-item', function() {
                    $(this).parent().remove();
                });
                
                // Add additional section
                var additionalSectionIndex = <?php echo !empty($instance['additional_sections']) ? count($instance['additional_sections']) : 0; ?>;
                $('.add-additional-section').on('click', function() {
                    var html = '<div class="additional-section">' +
                        '<button type="button" class="remove-section button">Remove Section</button>' +
                        '<p><label><?php _e('Section Title:'); ?></label>' +
                        '<input type="text" class="widefat" name="<?php echo $this->get_field_name('additional_sections'); ?>[' + additionalSectionIndex + '][title]" /></p>' +
                        '<p><label><?php _e('Section Content:'); ?></label>' +
                        '<textarea class="widefat" rows="6" name="<?php echo $this->get_field_name('additional_sections'); ?>[' + additionalSectionIndex + '][content]"></textarea></p>' +
                        '</div>';
                    
                    $('.additional-sections-container').append(html);
                    additionalSectionIndex++;
                });
                
                // Remove additional section
                $(document).on('click', '.remove-section', function() {
                    $(this).parent().remove();
                });
                
                // Add sidebar item
                var sidebarItemIndex = <?php echo !empty($instance['sidebar_items']) ? count($instance['sidebar_items']) : 0; ?>;
                $('.add-sidebar-item').on('click', function() {
                    var html = '<div class="sidebar-item">' +
                        '<button type="button" class="remove-sidebar-item button">Remove Item</button>' +
                        '<p><label><?php _e('Item Label:'); ?></label>' +
                        '<input type="text" class="widefat" name="<?php echo $this->get_field_name('sidebar_items'); ?>[' + sidebarItemIndex + '][label]" /></p>' +
                        '<p><label><?php _e('Item Value:'); ?></label>' +
                        '<input type="text" class="widefat" name="<?php echo $this->get_field_name('sidebar_items'); ?>[' + sidebarItemIndex + '][value]" /></p>' +
                        '<p><label><?php _e('Icon (Font Awesome class):'); ?></label>' +
                        '<input type="text" class="widefat" name="<?php echo $this->get_field_name('sidebar_items'); ?>[' + sidebarItemIndex + '][icon]" placeholder="fas fa-calendar" /></p>' +
                        '</div>';
                    
                    $('.sidebar-items-container').append(html);
                    sidebarItemIndex++;
                });
                
                // Remove sidebar item
                $(document).on('click', '.remove-sidebar-item', function() {
                    $(this).parent().remove();
                });
                
                // Add gallery image
                $('.add-gallery-image').on('click', function() {
                    var html = '<div class="gallery-image">' +
                        '<button type="button" class="remove-gallery-image button">Remove Image</button>' +
                        '<input type="text" class="widefat image-url" name="<?php echo $this->get_field_name('gallery_images'); ?>[]" />' +
                        '<button type="button" class="upload-gallery-image button">Select Image</button>' +
                        '<div class="image-preview"></div>' +
                        '</div>';
                    
                    $('.gallery-images-container').append(html);
                });
                
                // Remove gallery image
                $(document).on('click', '.remove-gallery-image', function() {
                    $(this).parent().remove();
                });
                
                // Image upload
                $(document).on('click', '.upload-image-button, .upload-gallery-image', function() {
                    var button = $(this);
                    var imageUrlField = button.siblings('input[type="text"]');
                    var imagePreview = button.siblings('.image-preview');
                    
                    // Create the media frame
                    var frame = wp.media({
                        title: 'Select or Upload Image',
                        button: {
                            text: 'Use this image'
                        },
                        multiple: false
                    });
                    
                    // When an image is selected, run a callback
                    frame.on('select', function() {
                        var attachment = frame.state().get('selection').first().toJSON();
                        imageUrlField.val(attachment.url);
                        
                        if (imagePreview.length) {
                            imagePreview.html('<img src="' + attachment.url + '" style="max-width: 100%; height: auto; margin-top: 10px;" />');
                        }
                    });
                    
                    // Finally, open the modal
                    frame.open();
                });
            });
        </script>
        <?php
    }

    // Update widget settings
    public function update($new_instance, $old_instance) {
        $instance = $old_instance;
        
        // Basic settings
        $instance['title'] = sanitize_text_field($new_instance['title']);
        $instance['destination_name'] = sanitize_text_field($new_instance['destination_name']);
        $instance['destination_excerpt'] = sanitize_textarea_field($new_instance['destination_excerpt']);
        $instance['featured_image'] = esc_url_raw($new_instance['featured_image']);
        
        // Tour overview
        $instance['show_tour_overview'] = isset($new_instance['show_tour_overview']) ? 1 : 0;
        $instance['tour_overview_title'] = sanitize_text_field($new_instance['tour_overview_title']);
        $instance['tour_overview_content'] = wp_kses_post($new_instance['tour_overview_content']);
        
        // Itinerary
        $instance['show_itinerary'] = isset($new_instance['show_itinerary']) ? 1 : 0;
        $instance['itinerary_title'] = sanitize_text_field($new_instance['itinerary_title']);
        $instance['itinerary_days'] = array();
        
        if (!empty($new_instance['itinerary_days'])) {
            foreach ($new_instance['itinerary_days'] as $day) {
                if (!empty($day['title']) || !empty($day['description'])) {
                    $instance['itinerary_days'][] = array(
                        'title' => sanitize_text_field($day['title']),
                        'description' => wp_kses_post($day['description'])
                    );
                }
            }
        }
        
        // Price details
        $instance['show_price_details'] = isset($new_instance['show_price_details']) ? 1 : 0;
        $instance['price_includes_title'] = sanitize_text_field($new_instance['price_includes_title']);
        $instance['price_excludes_title'] = sanitize_text_field($new_instance['price_excludes_title']);
        
        $instance['price_includes'] = array();
        if (!empty($new_instance['price_includes'])) {
            foreach ($new_instance['price_includes'] as $item) {
                if (!empty($item)) {
                    $instance['price_includes'][] = sanitize_text_field($item);
                }
            }
        }
        
        $instance['price_excludes'] = array();
        if (!empty($new_instance['price_excludes'])) {
            foreach ($new_instance['price_excludes'] as $item) {
                if (!empty($item)) {
                    $instance['price_excludes'][] = sanitize_text_field($item);
                }
            }
        }
        
        // Additional sections
        $instance['show_additional_sections'] = isset($new_instance['show_additional_sections']) ? 1 : 0;
        $instance['additional_sections'] = array();
        
        if (!empty($new_instance['additional_sections'])) {
            foreach ($new_instance['additional_sections'] as $section) {
                if (!empty($section['title']) || !empty($section['content'])) {
                    $instance['additional_sections'][] = array(
                        'title' => sanitize_text_field($section['title']),
                        'content' => wp_kses_post($section['content'])
                    );
                }
            }
        }
        
        // Sidebar
        $instance['sidebar_title'] = sanitize_text_field($new_instance['sidebar_title']);
        $instance['sidebar_items'] = array();
        
        if (!empty($new_instance['sidebar_items'])) {
            foreach ($new_instance['sidebar_items'] as $item) {
                if (!empty($item['label']) || !empty($item['value'])) {
                    $instance['sidebar_items'][] = array(
                        'label' => sanitize_text_field($item['label']),
                        'value' => sanitize_text_field($item['value']),
                        'icon' => sanitize_text_field($item['icon'])
                    );
                }
            }
        }
        
        $instance['inquiry_button_text'] = sanitize_text_field($new_instance['inquiry_button_text']);
        $instance['inquiry_button_link'] = esc_url_raw($new_instance['inquiry_button_link']);
        
        // Gallery
        $instance['show_gallery'] = isset($new_instance['show_gallery']) ? 1 : 0;
        $instance['gallery_title'] = sanitize_text_field($new_instance['gallery_title']);
        $instance['gallery_images'] = array();
        
        if (!empty($new_instance['gallery_images'])) {
            foreach ($new_instance['gallery_images'] as $image) {
                if (!empty($image)) {
                    $instance['gallery_images'][] = esc_url_raw($image);
                }
            }
        }
        
        // WhatsApp
        $instance['show_whatsapp_button'] = isset($new_instance['show_whatsapp_button']) ? 1 : 0;
        $instance['whatsapp_number'] = sanitize_text_field($new_instance['whatsapp_number']);
        $instance['whatsapp_message'] = sanitize_text_field($new_instance['whatsapp_message']);
        
        return $instance;
    }

    // Widget frontend display
    public function widget($args, $instance) {
        echo $args['before_widget'];
        
        if (!empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }
        
        // Output the widget content directly
        echo '<div class="single-destination-widget">
            <!-- Hero Section -->
            <section class="destination-hero" style="background-image: linear-gradient(rgba(0, 0, 0, 0.5), rgba(0, 0, 0, 0.5)), url(\'' . esc_url($instance['featured_image']) . '\');">
                <div class="hero-content">
                    <h1 class="destination-title">' . esc_html($instance['destination_name']) . '</h1>';
                    
        if (!empty($instance['destination_excerpt'])) {
            echo '<p class="destination-excerpt">' . esc_html($instance['destination_excerpt']) . '</p>';
        }

        echo '</div>
            </section>
        </div>';
            
        echo '<div class="destination-container">
            <div class="destination-main">
                <div class="destination-content">';

        // Tour Overview Section
        if ($instance['show_tour_overview']) {
            echo '<section class="tour-overview-section">
                <h2>' . esc_html($instance['tour_overview_title']) . '</h2>
                <div class="tour-overview-content">' .
                    wpautop(wp_kses_post($instance['tour_overview_content'])) . '
                </div>
            </section>';
        }

        // Itinerary Section
        if ($instance['show_itinerary'] && !empty($instance['itinerary_days'])) {
            echo '<section class="itinerary-section">
                <h2>' . esc_html($instance['itinerary_title']) . '</h2>';
            
            foreach ($instance['itinerary_days'] as $index => $day) {
                echo '<div class="itinerary-day">
                    <div class="itinerary-day-header">' .
                        esc_html($day['title']) . '
                    </div>
                    <div class="itinerary-day-content">' .
                        wpautop(wp_kses_post($day['description'])) . '
                    </div>
                </div>';
            }
            
            echo '</section>';
        }

        // Price Details Section
        if ($instance['show_price_details'] && (!empty($instance['price_includes']) || !empty($instance['price_excludes']))) {
            echo '<section class="price-details-section">
                <div class="price-details-grid">';
            
            if (!empty($instance['price_includes'])) {
                echo '<div class="price-includes">
                    <h3>' . esc_html($instance['price_includes_title']) . '</h3>
                    <ul>';
                
                foreach ($instance['price_includes'] as $item) {
                    echo '<li>' . esc_html($item) . '</li>';
                }
                
                echo '</ul>
                </div>';
            }
            
            if (!empty($instance['price_excludes'])) {
                echo '<div class="price-excludes">
                    <h3>' . esc_html($instance['price_excludes_title']) . '</h3>
                    <ul>';
                
                foreach ($instance['price_excludes'] as $item) {
                    echo '<li>' . esc_html($item) . '</li>';
                }
                
                echo '</ul>
                </div>';
            }
            
            echo '</div>
            </section>';
        }

        // Additional Sections
        if ($instance['show_additional_sections'] && !empty($instance['additional_sections'])) {
            foreach ($instance['additional_sections'] as $section) {
                echo '<section class="additional-section">
                    <h2>' . esc_html($section['title']) . '</h2>
                    <div class="section-content">' .
                        wpautop(wp_kses_post($section['content'])) . '
                    </div>
                </section>';
            }
        }

        // Gallery Section
        if ($instance['show_gallery'] && !empty($instance['gallery_images'])) {
            echo '<section class="gallery-section">
                <h2>' . esc_html($instance['gallery_title']) . '</h2>
                <div class="destination-gallery">';
            
            foreach ($instance['gallery_images'] as $image) {
                echo '<div class="gallery-item">
                    <img src="' . esc_url($image) . '" alt="' . esc_attr($instance['destination_name']) . '">
                </div>';
            }
            
            echo '</div>
            </section>';
        }

        echo '</div>
            </div>';

        // Sidebar
        echo '<div class="destination-sidebar">
            <div class="tour-overview-sidebar">
                <h3>' . esc_html($instance['sidebar_title']) . '</h3>';
                
        if (!empty($instance['sidebar_items'])) {
            foreach ($instance['sidebar_items'] as $item) {
                echo '<div class="sidebar-item">';
                
                if (!empty($item['icon'])) {
                    echo '<i class="' . esc_attr($item['icon']) . '"></i>';
                }
                
                echo '<div class="sidebar-item-content">
                    <span class="sidebar-item-label">' . esc_html($item['label']) . '</span>
                    <span class="sidebar-item-value">' . esc_html($item['value']) . '</span>
                </div>
                </div>';
            }
        }

        echo '<a href="' . esc_url($instance['inquiry_button_link']) . '" class="inquiry-button">' .
                esc_html($instance['inquiry_button_text']) . '
            </a>
            </div>
        </div>';

        // WhatsApp Button
        if ($instance['show_whatsapp_button'] && !empty($instance['whatsapp_number'])) {
            $whatsapp_url = 'https://wa.me/' . $instance['whatsapp_number'] . '?text=' . urlencode($instance['whatsapp_message']);
            echo '<a href="' . esc_url($whatsapp_url) . '" class="whatsapp-button" target="_blank">
                <i class="fab fa-whatsapp"></i>
            </a>';
        }

        echo '</div>';

        // Output the CSS
        echo '<style>
        .single-destination-widget {
            font-family: \'Poppins\', sans-serif;
            color: #333;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .destination-hero {
            height: 400px;
            background-size: cover;
            background-position: center;
            display: flex;
            align-items: center;
            justify-content: center;
            text-align: center;
            color: white;
            margin-bottom: 40px;
            border-radius: 10px;
            overflow: hidden;
        }

        .hero-content {
            max-width: 800px;
            padding: 20px;
        }

        .destination-title {
            font-size: 3rem;
            margin-bottom: 20px;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.5);
        }

        .destination-excerpt {
            font-size: 1.2rem;
            line-height: 1.6;
            text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.5);
        }

        .destination-container {
            display: flex;
            gap: 30px;
        }

        .destination-main {
            flex: 2;
        }

        .destination-sidebar {
            flex: 1;
        }

        .tour-overview-section,
        .itinerary-section,
        .price-details-section,
        .additional-section,
        .gallery-section {
            margin-bottom: 40px;
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        }

        h2 {
            color: #2c5282;
            font-size: 2rem;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid #f6ad55;
        }

        .itinerary-day {
            margin-bottom: 20px;
            border: 1px solid #eee;
            border-radius: 8px;
            overflow: hidden;
        }

        .itinerary-day-header {
            background: #2c5282;
            color: white;
            padding: 15px 20px;
            cursor: pointer;
            font-weight: 600;
            transition: background 0.3s;
        }

        .itinerary-day-header:hover {
            background: #1e3a5f;
        }

        .itinerary-day-content {
            padding: 20px;
            display: none;
        }

        .itinerary-day.active .itinerary-day-content {
            display: block;
        }

        .price-details-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        .price-includes,
        .price-excludes {
            padding: 20px;
            border-radius: 8px;
        }

        .price-includes {
            border-left: 4px solid #28a745;
            background: rgba(40, 167, 69, 0.05);
        }

        .price-excludes {
            border-left: 4px solid #dc3545;
            background: rgba(220, 53, 69, 0.05);
        }

        .price-includes h3 {
            color: #28a745;
        }

        .price-excludes h3 {
            color: #dc3545;
        }

        .price-includes ul,
        .price-excludes ul {
            list-style: none;
            padding: 0;
        }

        .price-includes li:before {
            content: "✓";
            color: #28a745;
            margin-right: 10px;
        }

        .price-excludes li:before {
            content: "✗";
            color: #dc3545;
            margin-right: 10px;
        }

        .tour-overview-sidebar {
            background: linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%);
            color: white;
            padding: 25px;
            border-radius: 10px;
            position: sticky;
            top: 20px;
        }

        .tour-overview-sidebar h3 {
            color: #f6ad55;
            text-align: center;
            margin-bottom: 25px;
            font-size: 1.5rem;
        }

        .sidebar-item {
            display: flex;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .sidebar-item i {
            font-size: 1.5rem;
            margin-right: 15px;
            color: #f6ad55;
        }

        .sidebar-item-content {
            display: flex;
            flex-direction: column;
        }

        .sidebar-item-label {
            font-weight: 600;
            color: #f6ad55;
            margin-bottom: 5px;
        }

        .sidebar-item-value {
            font-size: 0.9rem;
        }

        .inquiry-button {
            display: block;
            background: #f6ad55;
            color: #1e3a5f;
            text-align: center;
            padding: 15px;
            border-radius: 5px;
            font-weight: 600;
            text-decoration: none;
            margin-top: 30px;
            transition: all 0.3s;
        }

        .inquiry-button:hover {
            background: #e69c42;
            transform: translateY(-3px);
        }

        .destination-gallery {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 15px;
        }

        .gallery-item img {
            width: 100%;
            height: 200px;
            object-fit: cover;
            border-radius: 8px;
            transition: transform 0.3s;
        }

        .gallery-item img:hover {
            transform: scale(1.05);
        }

        .whatsapp-button {
            position: fixed;
            bottom: 30px;
            right: 30px;
            background: #25D366;
            color: white;
            width: 60px;
            height: 60px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2rem;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
            transition: all 0.3s;
            z-index: 1000;
        }

        .whatsapp-button:hover {
            transform: scale(1.1);
            color: white;
        }

        @media (max-width: 992px) {
            .destination-container {
                flex-direction: column;
            }
            
            .price-details-grid {
                grid-template-columns: 1fr;
            }
            
            .destination-hero {
                height: 300px;
            }
            
            .destination-title {
                font-size: 2.5rem;
            }
        }

        @media (max-width: 768px) {
            .destination-hero {
                height: 250px;
            }
            
            .destination-title {
                font-size: 2rem;
            }
            
            h2 {
                font-size: 1.5rem;
            }
        }
        </style>';

        // Output the JavaScript
        echo '<script>
        jQuery(document).ready(function($) {
            // Itinerary accordion
            $(\'.itinerary-day-header\').click(function() {
                $(this).next(\'.itinerary-day-content\').slideToggle();
                $(this).parent().toggleClass(\'active\');
            });
            
            // Initialize first itinerary day as open
            $(\'.itinerary-day:first-child .itinerary-day-content\').show();
            $(\'.itinerary-day:first-child\').addClass(\'active\');
        });
        </script>';

        // Admin CSS
        echo '<style>
        .single-destination-widget-admin {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
        }

        .single-destination-widget-admin .tab-content {
            display: none;
            padding: 15px;
            background: #f9f9f9;
            border-radius: 5px;
            margin-bottom: 15px;
        }

        .single-destination-widget-admin .tab-content.active {
            display: block;
        }

        .single-destination-widget-admin .itinerary-day,
        .single-destination-widget-admin .additional-section,
        .single-destination-widget-admin .sidebar-item,
        .single-destination-widget-admin .gallery-image {
            background: white;
            padding: 15px;
            border: 1px solid #ddd;
            border-radius: 5px;
            margin-bottom: 15px;
            position: relative;
        }

        .single-destination-widget-admin .price-details-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        .single-destination-widget-admin .price-item {
            display: flex;
            gap: 10px;
            margin-bottom: 10px;
        }

        .single-destination-widget-admin .price-item input {
            flex: 1;
        }

        .single-destination-widget-admin .remove-day,
        .single-destination-widget-admin .remove-section,
        .single-destination-widget-admin .remove-sidebar-item,
        .single-destination-widget-admin .remove-gallery-image,
        .single-destination-widget-admin .remove-item {
            position: absolute;
            top: 10px;
            right: 10px;
            color: #dc3545;
            cursor: pointer;
        }

        .single-destination-widget-admin .image-preview img {
            max-width: 100%;
            height: auto;
            margin-top: 10px;
        }

        @media (max-width: 782px) {
            .single-destination-widget-admin .price-details-grid {
                grid-template-columns: 1fr;
            }
        }
        </style>';

        echo $args['after_widget'];
    }
}