$(document).ready(function() {
    // Set up AJAX defaults with CSRF token
    $.ajaxSetup({
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-CSRF-TOKEN': csrfTokenValue // Using the global variable
        }
    });

    // Load current permissions when accordion opens
    // Only attempt to load current permissions when the group accordion opens.
    // Extract the trailing numeric id to avoid ids like "flush-collapse3" becoming "flush-3".
    $('.collapse').on('show.bs.collapse', function() {
        const id = $(this).attr('id') || '';
        const match = id.match(/(\d+)$/);
        if (!match) return; // no numeric id found, skip
        const groupId = match[1];
        loadCurrentPermissions(groupId);
    });

    // Handle permission checkbox changes
    $('.permissions-grid').on('change', '.perm-checkbox', function() {
        const $checkbox = $(this);
        const groupId = $checkbox.data('group');
        const permId = $checkbox.data('perm');
        const isChecked = $checkbox.is(':checked');
        
        $('#confirmMessage').text(`Are you sure you want to ${isChecked ? 'add' : 'remove'} this permission?`);
        $('#confirmModal').modal('show');
        
        // Handle confirm
        $('#confirmAction').off('click').on('click', function() {
            updatePermission(groupId, permId, isChecked ? 'add' : 'remove', $checkbox);
            $('#confirmModal').modal('hide');
        });
        
        // Handle cancel - revert checkbox state
        $('#confirmModal').off('hidden.bs.modal').on('hidden.bs.modal', function() {
            if (!$checkbox.data('update-in-progress')) {
                $checkbox.prop('checked', !isChecked);
            }
        });
    });

    function loadCurrentPermissions(groupId) {
        $.get( base_url + 'administrator/settings/show-group-permissions/' + groupId, function(data) {
            let html = '<div class="d-flex flex-wrap">';
            if (data.length > 0) {
                data.forEach(perm => {
                    html += `<span class="badge bg-primary m-1 p-2">
                                ${perm.name} 
                                <small>(${perm.module})</small>
                                <button class="btn btn-xs p-1 btn-danger ml-2 remove-perm" 
                                        data-group="${groupId}" 
                                        data-perm="${perm.id}">
                                    <i class="bi bi-x"></i>
                                </button>
                            </span>`;
                });
            } else {
                html += '<span class="text-muted">No permissions assigned</span>';
            }
            html += '</div>';
            $(`#currentPermissions${groupId}`).html(html);
            
            // Note: Checkboxes are now pre-checked from server-side PHP
            // No need to check them again via JavaScript
        });
    }

    function updatePermission(groupId, permId, action, $checkbox) {
        // Mark update in progress
        if ($checkbox) {
            $checkbox.data('update-in-progress', true);
        }
        
        const postData = {
            group_id: groupId,
            permission_id: permId,
            action: action
        };

        // Include CSRF token under the correct token name expected by CodeIgniter
        try {
            if (window.csrfTokenName && window.csrfTokenValue) {
                postData[window.csrfTokenName] = window.csrfTokenValue;
            }
        } catch (e) {
            // noop
        }
        

        $.ajax({
            url: base_url + 'administrator/settings/update-group-permission',
            type: 'POST',
            data: postData,
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    loadCurrentPermissions(groupId);
                    toastr.success('Permission updated successfully');
                    // Update succeeded, keep checkbox state
                    if ($checkbox) {
                        $checkbox.data('update-in-progress', false);
                    }
                    // Update CSRF token values if sent by server
                    if (response.csrfName && response.csrfHash) {
                        try {
                            window.csrfTokenName = response.csrfName;
                            window.csrfTokenValue = response.csrfHash;
                            $.ajaxSetup({ headers: { 'X-CSRF-TOKEN': window.csrfTokenValue } });
                        } catch (e) {}
                    }
                } else {
                    toastr.error(response.message || 'Operation failed');
                    // Revert checkbox on failure
                    if ($checkbox) {
                        $checkbox.prop('checked', action !== 'add');
                        $checkbox.data('update-in-progress', false);
                    }
                }
            },
            error: function(xhr) {
                const resp = xhr.responseJSON || {};
                toastr.error('Error: ' + (resp.message || xhr.statusText));
                // Revert checkbox on error
                if ($checkbox) {
                    $checkbox.prop('checked', action !== 'add');
                    $checkbox.data('update-in-progress', false);
                }
                // If server sent new CSRF token on error, update it
                try {
                    if (resp.csrfName && resp.csrfHash) {
                        window.csrfTokenName = resp.csrfName;
                        window.csrfTokenValue = resp.csrfHash;
                        $.ajaxSetup({ headers: { 'X-CSRF-TOKEN': window.csrfTokenValue } });
                    }
                } catch (e) {}
            }
        });
    }

    // Handle remove permission buttons
    $(document).on('click', '.remove-perm', function() {
        const groupId = $(this).data('group');
        const permId = $(this).data('perm');
        
        // Find and uncheck the corresponding checkbox
        const $checkbox = $(`#perm_${groupId}_${permId}`);
        
        $('#confirmMessage').text('Are you sure you want to remove this permission?');
        $('#confirmModal').modal('show');
        
        $('#confirmAction').off('click').on('click', function() {
            updatePermission(groupId, permId, 'remove', $checkbox);
            if ($checkbox.length) {
                $checkbox.prop('checked', false);
            }
            $('#confirmModal').modal('hide');
        });
        
        // Handle cancel - do nothing
        $('#confirmModal').off('hidden.bs.modal');
    });
});