// Company Settings CRUD Operations - Fixed Version
class CompanyManager {
    constructor() {
        this.modal = null;
        this.form = null;
        this.baseUrl = window.base_url || '';
        this.init();
    }

    init() {
        this.initializeModal();
        this.bindEvents();
        this.loadCountryStatesCities();
    }

    initializeModal() {
        const modalElement = document.getElementById('companyModal');
        if (modalElement && typeof bootstrap !== 'undefined') {
            this.modal = new bootstrap.Modal(modalElement);
            this.form = document.getElementById('companyForm');
        }
    }

    bindEvents() {
        // Edit company button
        $(document).on('click', '.edit-company', (e) => {
            e.preventDefault();
            const companyId = $(e.currentTarget).data('id');
            this.loadCompanyData(companyId);
        });

        // Add company button
        $('[data-bs-target="#companyModal"]').on('click', () => {
            this.resetForm();
        });

        // Form submission
        if (this.form) {
            this.form.addEventListener('submit', (e) => {
                e.preventDefault();
                this.submitForm();
            });
        }

        // Country change event - filter states locally
        $(document).on('change', '#country_id', (e) => {
            const countryId = e.target.value;
            console.log('Country changed to:', countryId);
            this.filterStates(countryId);
            // Reset city dropdown
            $('#city_id').html('<option value="">-</option>');
            // Currency remains completely independent - NO LINKING
        });

        // State change event - filter cities locally
        $(document).on('change', '#state_id', (e) => {
            const stateId = e.target.value;
            console.log('State changed to:', stateId);
            this.filterCities(stateId);
        });

        // Image preview
        $(document).on('change', 'input[name="photo"]', (e) => {
            this.previewImage(e.target);
        });
    }

    resetForm() {
        if (this.form) {
            this.form.reset();
            $('input[name="id"]').val('');
            $('#companyModalLabel').text('Add Company Details');
            $('.sbtn').text('Save Company');
            $('#display-image').empty();
            this.clearValidation();
            
            // Reset state and city dropdowns only
            $('#state_id').html('<option value="">-</option>');
            $('#city_id').html('<option value="">-</option>');
            
            // Currency dropdown remains untouched - independent selection
        }
    }

    clearValidation() {
        $('.is-invalid').removeClass('is-invalid');
        $('.invalid-feedback').remove();
    }

    async loadCompanyData(companyId) {
        try {
            const response = await $.get(`${this.baseUrl}administrator/settings/company`, { id: companyId });
            
            if (response.success && response.data) {
                this.populateForm(response.data);
                this.modal.show();
            } else {
                this.showError('Failed to load company data');
            }
        } catch (error) {
            console.error('Error loading company data:', error);
            this.showError('Error loading company data');
        }
    }

    populateForm(company) {
        console.log('Populating form with company data:', company);
        
        // Basic info
        $('input[name="id"]').val(company.id);
        $('input[name="name"]').val(company.name);
        $('input[name="email"]').val(company.email);
        $('input[name="tel"]').val(company.tel);
        $('input[name="mobile"]').val(company.mobile);
        $('input[name="tax_id"]').val(company.tax_id);
        $('input[name="website"]').val(company.website);
        
        // Currency - COMPLETELY INDEPENDENT from country
        // Use the stored currency ID from the database
        $('select[name="default_currency_id"]').val(company.default_currency_id);
        console.log('Setting currency to:', company.default_currency_id);
        
        // Address
        $('input[name="address_1"]').val(company.address_1);
        $('input[name="address_2"]').val(company.address_2);
        $('input[name="area_code"]').val(company.area_code);
        
        // Location - filter states and cities locally
        // Only set location fields if they have valid values (not 0)
        if (company.country_id && company.country_id != 0) {
            $('select[name="country_id"]').val(company.country_id);
            console.log('Setting country to:', company.country_id);

            // Filter states for the selected country
            this.filterStates(company.country_id);

            if (company.state_id && company.state_id != 0) {
                $('select[name="state_id"]').val(company.state_id);
                console.log('Setting state to:', company.state_id);

                // Filter cities for the selected state
                this.filterCities(company.state_id);

                if (company.city_id && company.city_id != 0) {
                    $('select[name="city_id"]').val(company.city_id);
                    console.log('Setting city to:', company.city_id);
                }
            }
        }
        
        // Text areas
        $('textarea[name="vision"]').val(company.vision);
        $('textarea[name="mission"]').val(company.mission);
        $('textarea[name="objectives"]').val(company.objectives);
        $('textarea[name="core_values"]').val(company.core_values);
        $('textarea[name="description"]').val(company.description);
        
        // Social media
        $('input[name="facebook"]').val(company.facebook);
        $('input[name="twitter"]').val(company.twitter);
        $('input[name="linkedin"]').val(company.linkedin);
        $('input[name="instagram"]').val(company.instagram);
        $('input[name="whatsapp"]').val(company.whatsapp);
        
        // Display image - handle both photo URL and photo filename
        const displayImage = $('#display-image');
        displayImage.empty();
        
        if (company.photo_url) {
            // Use the photo_url provided by the controller
            const img = $('<img>', {
                class: 'img-fluid rounded',
                src: company.photo_url,
                alt: 'Company Logo',
                style: 'max-height: 200px;'
            });
            displayImage.append(img);
        } else if (company.photo) {
            // Fallback: construct URL from filename
            const baseUrl = window.base_url || '';
            const photoUrl = baseUrl + 'media/files/' + company.photo;
            const img = $('<img>', {
                class: 'img-fluid rounded',
                src: photoUrl,
                alt: 'Company Logo',
                style: 'max-height: 200px;'
            });
            displayImage.append(img);
        }
        
        $('#companyModalLabel').text('Edit Company Details');
        $('.sbtn').text('Update Company');
        
        console.log('Form populated successfully - Currency and Country are independent');
    }

    async submitForm() {
        console.log('Submitting form...');
        
        // Debug: Show current form values before submission
        const formData = new FormData(this.form);
        console.log('Currency value:', $('select[name="default_currency_id"]').val());
        console.log('Country value:', $('select[name="country_id"]').val());
        
        // Add CSRF token
        if (window.csrfTokenName && window.csrfTokenValue) {
            formData.append(window.csrfTokenName, window.csrfTokenValue);
        }

        // Show loading state
        const submitBtn = $('.sbtn');
        const originalText = submitBtn.text();
        submitBtn.prop('disabled', true).text('Saving...');

        try {
            const response = await $.ajax({
                url: `${this.baseUrl}administrator/settings/company`,
                method: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json'
            });

            console.log('Form submission response:', response);

            if (response.success) {
                this.showSuccess(response.message);
                if (this.modal) {
                    this.modal.hide();
                }
                setTimeout(() => {
                    window.location.reload();
                }, 1500);
            } else {
                this.displayValidationErrors(response.errors);
                this.showError(response.message || 'Please check the form for errors');
            }
        } catch (error) {
            console.error('Error submitting form:', error);
            this.showError('Server error occurred. Please try again.');
        } finally {
            // Re-enable button
            submitBtn.prop('disabled', false).text(originalText);
        }
    }

    displayValidationErrors(errors) {
        this.clearValidation();
        
        if (errors) {
            Object.keys(errors).forEach(field => {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                
                // Remove existing error message
                input.next('.invalid-feedback').remove();
                
                // Add new error message
                const errorDiv = $('<div>', {
                    class: 'invalid-feedback',
                    text: errors[field]
                });
                input.after(errorDiv);
            });
        }
    }

    previewImage(input) {
        const displayImage = $('#display-image');
        displayImage.empty();
        
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            reader.onload = (e) => {
                const img = $('<img>', {
                    class: 'img-fluid rounded',
                    src: e.target.result,
                    alt: 'Preview',
                    style: 'max-height: 200px;'
                });
                displayImage.append(img);
            };
            reader.readAsDataURL(input.files[0]);
        }
    }

    filterStates(countryId) {
        const stateSelect = $('#state_id');
        stateSelect.html('<option value="">-</option>');

        if (!countryId || countryId == 0) {
            $('#city_id').html('<option value="">-</option>');
            return;
        }

        // Filter states based on data-country-id attribute
        $('#state_id option').each(function() {
            const option = $(this);
            const stateCountryId = option.data('country-id');
            if (stateCountryId == countryId) {
                option.show();
            } else {
                option.hide();
            }
        });
    }

    filterCities(stateId) {
        const citySelect = $('#city_id');
        citySelect.html('<option value="">-</option>');

        if (!stateId || stateId == 0) {
            return;
        }

        // Filter cities based on data-state-id attribute
        $('#city_id option').each(function() {
            const option = $(this);
            const cityStateId = option.data('state-id');
            if (cityStateId == stateId) {
                option.show();
            } else {
                option.hide();
            }
        });
    }

    loadCountryStatesCities() {
        // For edit mode, filter states and cities based on pre-selected values
        const countryId = $('#country_id').val();
        if (countryId && countryId != 0) {
            this.filterStates(countryId);

            // Also filter cities if state is already selected
            const stateId = $('#state_id').val();
            if (stateId && stateId != 0) {
                this.filterCities(stateId);
            }
        }
    }

    showSuccess(message) {
        console.log('Success:', message);
        if (typeof toastr !== 'undefined') {
            toastr.success(message);
        } else {
            alert('Success: ' + message);
        }
    }

    showError(message) {
        console.error('Error:', message);
        if (typeof toastr !== 'undefined') {
            toastr.error(message);
        } else {
            alert('Error: ' + message);
        }
    }
}

// Initialize when document is ready
$(document).ready(() => {
    console.log('Initializing CompanyManager...');
    console.log('Currency and Country are COMPLETELY INDEPENDENT');
    new CompanyManager();
});