(function () {
  // Minimal, non-invasive media intercept.
  // - Shows modal with Local vs Library
  // - For Library: fetches /administrator/media, parses #file-table rows to extract file URLs
  // - Converts selected media to File objects and assigns to original input via DataTransfer

  if (window.mediaInterceptInitialized) return;
  window.mediaInterceptInitialized = true;

  const modalEl = document.getElementById('media-intercept-modal');
  const bsModal = modalEl ? new bootstrap.Modal(modalEl, { backdrop: 'static' }) : null;
  const btnLocal = document.getElementById('media-intercept-local');
  const btnLibrary = document.getElementById('media-intercept-library');
  const libraryPane = document.getElementById('media-intercept-library-pane');
  const grid = document.getElementById('media-intercept-grid');
  const loading = document.getElementById('media-intercept-loading');
  const insertBtn = document.getElementById('media-intercept-insert');

  let originalInput = null;
  let cachedMedia = null; // array of {thumb,url,name,size}
  let lastFolderData = null;
  let selectedItems = new Set();
  let showRandomized = false;
  // If we're inside the Media module view, don't intercept (the file manager has its own upload flow)
  const insideMediaModule = !!document.querySelector('.file-manager');
  const showRandomCheckbox = document.getElementById('media-intercept-show-random');
  const breadcrumbLabel = document.getElementById('media-intercept-breadcrumb');

  function isLikelyRandomName(name) {
    if (!name || typeof name !== 'string') return false;
    // pattern: numeric prefix + underscore + long hex/hash + ext e.g. 1761931093_f5740371e8bb9c5a1815.jpeg
    if (/^\d+_[a-f0-9]{8,}/i.test(name)) return true;
    // or contains underscore + long hex
    if (/_[a-f0-9]{8,}/i.test(name)) return true;
    // or very long filename with many hex chars
    const cleaned = name.replace(/[^a-z0-9]/gi, '');
    if (cleaned.length > 30 && /^[a-f0-9]+$/i.test(cleaned)) return true;
    return false;
  }

  function resetModal() {
    libraryPane.style.display = 'none';
    loading.style.display = 'none';
    grid.innerHTML = '';
    selectedItems.clear();
    insertBtn.disabled = true;
  }

  // Load a media folder view from the admin Media Manager and render it into the modal grid
  function loadMediaFolder(path) {
    const base = (window.base_url || '').replace(/\/$/, '');
    const url = (base ? base : '') + '/administrator/media' + (path ? '?p=' + encodeURIComponent(path) : '');
    fetch(url, { credentials: 'include' })
      .then(r => {
        if (!r.ok) throw new Error('Failed to load media manager: ' + r.status);
        return r.text();
      })
      .then(html => {
        const parser = new DOMParser();
        const doc = parser.parseFromString(html, 'text/html');
        // extract breadcrumb (if present) and file table rows and folder links
        const breadcrumb = Array.from(doc.querySelectorAll('.breadcrumb li')).map(li => li.textContent.trim()).filter(Boolean);
        const rows = Array.from(doc.querySelectorAll('#file-table tbody tr'));
        const folders = [];
        const files = [];

        rows.forEach(r => {
          const link = r.querySelector('td .file-name a');
          const isFolder = !!(link && link.getAttribute('href') && link.getAttribute('href').includes('?p='));
          const cb = r.querySelector('input.file-checkbox');
          const previewImg = r.querySelector('img') ? r.querySelector('img').getAttribute('src') : null;
          const name = link ? (link.textContent || link.innerText).trim() : (r.querySelector('.file-name span') ? r.querySelector('.file-name span').textContent.trim() : '');
          if (isFolder) {
            // extract folder name from href ?p=...
            const href = link.getAttribute('href');
            const m = href.match(/[?&]p=([^&]+)/);
            const folderName = m ? decodeURIComponent(m[1]) : name;
            folders.push({ name: folderName, displayName: name });
          } else {
            // file URL: try checkbox id or the preview link
            let fileUrl = cb && cb.id ? cb.id : null;
            if (!fileUrl) {
              const fileLink = r.querySelector('button.link') || r.querySelector('.file-name a');
              if (fileLink && fileLink.getAttribute('onclick')) {
                // try to extract URL from onclick copyFilePath or similar (copyFilePath(url, name))
                const onclick = fileLink.getAttribute('onclick');
                const m = onclick.match(/copyFilePath\('\s*([^']+)\s*'\s*,\s*'([^']+)'\)/);
                if (m) fileUrl = m[1] + m[2];
              }
            }
            // canonical identifier is the filename itself (as shown in the list)
            const identifier = name;
            files.push({ name: name, url: fileUrl, thumb: previewImg, identifier: identifier });
          }
        });
        renderMediaFolder({ path: path, breadcrumb: breadcrumb, folders: folders, files: files });
      })
      .catch(err => {
        console.error('media-intercept: failed to load folder', err);
        loading.style.display = 'none';
        libraryPane.style.display = '';
        grid.innerHTML = '<div class="col-12 text-danger">Failed to load media library.</div>';
      });
  }

  function renderMediaFolder(data) {
    loading.style.display = 'none';
    libraryPane.style.display = '';
    grid.innerHTML = '';
    lastFolderData = data;
    // Breadcrumb
    if (breadcrumbLabel) breadcrumbLabel.textContent = 'Folder: ' + (data.breadcrumb.join(' / ') || 'Media');

    // Folders grid
    if (data.folders && data.folders.length) {
      const row = document.createElement('div');
      row.className = 'row mb-2';
      data.folders.forEach(f => {
        const col = document.createElement('div'); col.className = 'col-3';
        const card = document.createElement('div'); card.className = 'card p-2 text-center';
        const btn = document.createElement('button'); btn.className = 'btn btn-light w-100'; btn.innerHTML = '<i class="ri-folder-line me-1"></i>' + f.displayName;
        btn.addEventListener('click', function () { loadMediaFolder(f.name); });
        card.appendChild(btn); col.appendChild(card); row.appendChild(col);
      });
      grid.appendChild(row);
    }

    // Files grid
    const filesRow = document.createElement('div'); filesRow.className = 'row g-2';
    const filesToRender = (data.files || []).filter(f => {
      if (showRandomCheckbox) showRandomized = !!showRandomCheckbox.checked;
      if (showRandomized) return true;
      if (isLikelyRandomName(f.name)) {
        console.info('media-intercept: hiding randomized file', f.name);
        return false;
      }
      return true;
    });

    if (!filesToRender.length) {
      const no = document.createElement('div'); no.className = 'col-12'; no.textContent = 'No files found in this folder.'; filesRow.appendChild(no);
    } else {
      filesToRender.forEach(f => {
        const col = document.createElement('div'); col.className = 'col-3';
        const card = document.createElement('div'); card.className = 'card p-2 text-center';
        const img = document.createElement('img'); img.src = f.thumb || (f.url || ''); img.style.maxWidth = '100%'; img.style.maxHeight = '110px';
        const cap = document.createElement('div'); cap.className = 'small text-truncate mt-1'; cap.textContent = f.name;
        card.appendChild(img); card.appendChild(cap); col.appendChild(card); filesRow.appendChild(col);
        // click to select (toggle). Always require explicit Insert to fetch/insert.
        col.style.cursor = 'pointer';
        col.addEventListener('click', function () {
          const key = f.identifier || f.name || f.url;
          if (!originalInput.multiple) {
            // single-file: clear previous selection and select this
            selectedItems.clear();
            // visually clear previous
            grid.querySelectorAll('.col-3').forEach(c => c.classList.remove('border','border-primary'));
            selectedItems.add(key);
            col.classList.add('border','border-primary');
          } else {
            if (selectedItems.has(key)) { selectedItems.delete(key); col.classList.remove('border','border-primary'); } else { selectedItems.add(key); col.classList.add('border','border-primary'); }
          }
          insertBtn.disabled = selectedItems.size === 0;
          console.info('media-intercept: selected items', Array.from(selectedItems));
        });
      });
    }
    grid.appendChild(filesRow);
    // wire up showRandom checkbox change to re-render without refetching
    if (showRandomCheckbox) {
      showRandomCheckbox.onchange = function () {
        // re-render from lastFolderData
        if (lastFolderData) renderMediaFolder(lastFolderData);
      };
    }
  }
  function findFileInputFromTarget(target) {
    if (!target) return null;
    if (target.tagName === 'INPUT' && target.type === 'file') return target;
    if (target.tagName === 'LABEL' && target.htmlFor) {
      const el = document.getElementById(target.htmlFor);
      if (el && el.type === 'file') return el;
    }
    // data attribute heuristics
    const selector = target.getAttribute && (target.getAttribute('data-target') || target.getAttribute('data-open-upload'));
    if (selector) {
      const el = document.querySelector(selector);
      if (el && el.type === 'file') return el;
    }
    // walk up to find nearest input[type=file]
    const fileEl = target.closest ? target.closest('input[type=file]') : null;
    if (fileEl) return fileEl;
    return null;
  }

  // Prevent re-intercept when we programmatically open native picker
  window.__mediaInterceptSkip = null;

  function onCaptureClick(e) {
    try {
      if (insideMediaModule) return; // do not intercept events inside the media manager UI
      if (window.__mediaInterceptSkip && (e.target === window.__mediaInterceptSkip || e.target.closest && e.target.closest('[name="'+ (window.__mediaInterceptSkip.name || '') +'"]'))) {
        // allow this one and clear
        window.__mediaInterceptSkip = null;
        return;
      }
      const input = findFileInputFromTarget(e.target);
      if (!input) return;
      // Only intercept real user events (mouse left click or key activation)
      if (e.type === 'click' && e.button !== 0) return;

      // Intercept
      e.preventDefault();
      e.stopPropagation();
      originalInput = input;
      resetModal();
      if (bsModal) bsModal.show();
    } catch (err) {
      console.error('media-intercept: error on click', err);
    }
  }

  document.addEventListener('click', onCaptureClick, true);
  document.addEventListener('keydown', function (e) {
    if (e.key === 'Enter' || e.key === ' ') {
      onCaptureClick(e);
    }
  }, true);

  // Local button
  if (btnLocal) {
    btnLocal.addEventListener('click', function () {
      if (!originalInput) return;
      // mark skip so native opener isn't intercepted
      window.__mediaInterceptSkip = originalInput;
      // open native picker under a real user gesture
      setTimeout(function () { try { originalInput.click(); } catch (e) { console.error(e); } }, 50);
      safeHideModal();
    });
  }

  if (btnLibrary) {
    btnLibrary.addEventListener('click', function () {
      if (!originalInput) return;
      // Show a navigable Media Library view inside the modal (AJAX + parse), starting at root
      libraryPane.style.display = 'none';
      loading.style.display = '';
      selectedItems.clear();
      insertBtn.disabled = true;
      // start at root (no p parameter)
      loadMediaFolder('');
    });
  }

  function renderMediaGrid(items) {
    grid.innerHTML = '';
    if (!items || !items.length) {
      grid.innerHTML = '<div class="col-12">No media found</div>';
      insertBtn.disabled = true;
      return;
    }
    items.forEach((it, idx) => {
      const col = document.createElement('div');
      col.className = 'col-3';
      col.style.cursor = 'pointer';
      const card = document.createElement('div');
      card.className = 'card';
      card.style.height = '100%';
      const body = document.createElement('div');
      body.className = 'card-body p-2 text-center';
      const img = document.createElement('img');
      img.src = it.url;
      img.style.maxWidth = '100%';
      img.style.maxHeight = '110px';
      img.alt = it.name || '';
      const caption = document.createElement('div');
      caption.className = 'small text-truncate mt-1';
      caption.textContent = it.name;
      body.appendChild(img);
      body.appendChild(caption);
      card.appendChild(body);
      col.appendChild(card);
      grid.appendChild(col);

      col.addEventListener('click', function () {
        if (selectedItems.has(idx)) {
          selectedItems.delete(idx);
          col.classList.remove('border', 'border-primary');
        } else {
          // if original input is single-file, clear previous selection
          if (!originalInput.multiple) {
            // clear visual selected
            grid.querySelectorAll('.col-3').forEach((c, i) => { if (i!==idx) c.classList.remove('border','border-primary'); });
            selectedItems.clear();
          }
          selectedItems.add(idx);
          col.classList.add('border', 'border-primary');
        }
        insertBtn.disabled = selectedItems.size === 0;
      });
    });
  }

  insertBtn.addEventListener('click', function () {
    if (!originalInput) return;
    // First prefer URLs the user selected in our modal (selectedItems contains URLs)
    let urls = Array.from(selectedItems).filter(Boolean);
    // If no selectedItems (legacy), try to read iframe selections
    if (!urls.length) {
      const iframe = document.getElementById('media-intercept-iframe');
      if (iframe) {
        try {
          const doc = iframe.contentDocument || iframe.contentWindow.document;
          const rows = Array.from(doc.querySelectorAll('#file-table tbody tr'));
          selectedItems.forEach(i => {
            const row = rows[i];
            if (!row) return;
            const cb = row.querySelector('input.file-checkbox');
            if (cb && cb.id) urls.push(cb.id);
          });
        } catch (err) {
          console.error('media-intercept: reading iframe selection failed', err);
        }
      }
    }
    if (!urls.length) return;
    insertBtn.disabled = true;
    fetchAndInsert(urls).finally(() => { safeHideModal(); });
  });

  // helper to fetch URLs and insert into input
  function fetchAndInsert(urls) {
    // Try each URL with a set of fallbacks until one succeeds per item.
    const base = (window.base_url || window.location.origin).replace(/\/$/, '');

    const tryFetchWithFallbacks = async (u) => {
      const filename = (u + '').split('/').pop() || (u + '');
      const candidates = [];

      // If u looks like a full URL or absolute path, consider it, but avoid admin UI page routes
      const looksAbsolute = /^https?:\/\//i.test(u) || (typeof u === 'string' && u.startsWith('/'));
      if (looksAbsolute) {
        // If it's an admin UI page (e.g. /administrator/pages/...), skip it as a fetch target
        if (u.indexOf('/administrator/') === -1 || u.indexOf('/administrator/media/serve/') !== -1) {
          candidates.push(u);
        } else {
          console.info('media-intercept: skipping admin UI url as fetch candidate', u);
        }
      } else {
        // treat u as an identifier/filename candidate (may be just the filename)
        console.info('media-intercept: treating selection as identifier/filename:', u);
      }

      // canonical public filesystem locations (try in order)
      candidates.push(base + '/media/files/' + encodeURIComponent(filename));
    
      console.info('media-intercept: candidate list for', u, candidates);

      for (const c of candidates) {
        try {
          console.info('media-intercept: trying', c);
          const resp = await fetch(c, { credentials: 'include' });
          if (!resp.ok) {
            console.warn('media-intercept: fetch failed', c, resp.status);
            continue;
          }
          const blob = await resp.blob();
          console.info('media-intercept: fetch succeeded', c);
          return { blob, url: c };
        } catch (err) {
          console.warn('media-intercept: fetch error for', c, err && err.message ? err.message : err);
          continue;
        }
      }
      throw new Error('fetch failed (all fallbacks) for ' + u);
    };

    const fetchPromises = urls.map(u => tryFetchWithFallbacks(u));

    return Promise.all(fetchPromises)
      .then(results => {
        const dt = new DataTransfer();
        results.forEach(r => {
          try {
            const filename = (r.url.split('/').pop()) || 'file';
            const file = new File([r.blob], decodeURIComponent(filename), { type: r.blob.type || 'application/octet-stream' });
            dt.items.add(file);
          } catch (err) {
            console.error('media-intercept: File creation failed', err);
          }
        });
        try {
          originalInput.files = dt.files;
          originalInput.dispatchEvent(new Event('change', { bubbles: true }));
        } catch (err) {
          console.error('media-intercept: assigning files failed', err);
          // fallback hidden input
          const hidden = document.createElement('input');
          hidden.type = 'hidden';
          hidden.name = originalInput.name + '_media_url';
          hidden.value = urls[0] || '';
          hidden.dataset.mediaRef = '1';
          originalInput.parentNode && originalInput.parentNode.insertBefore(hidden, originalInput.nextSibling);
          alert('Selected media has been attached as a reference. If upload is required server-side support may be necessary.');
        }
      })
      .catch(err => { console.error('media-intercept: error fetching media files', err); alert('Failed to fetch one or more media files. See console for details.'); });
  }

  // Helper to safely hide modal without causing aria-hidden focus issues
  function safeHideModal() {
    try {
      if (document.activeElement && document.activeElement !== document.body) {
        try { document.activeElement.blur(); } catch (e) { /* ignore */ }
      }
    } catch (e) { /* ignore */ }
    if (bsModal) {
      try { bsModal.hide(); } catch (e) { console.error('media-intercept: hide failed', e); }
    }
  }

  // MutationObserver to capture dynamically added inputs (optional enhancement)
  const mo = new MutationObserver((mutations) => {
    // no-op for now; capture-phase document listeners handle added nodes too
  });
  mo.observe(document.documentElement || document.body, { childList: true, subtree: true });

})();
