<?php

namespace Modules\Settings\Controllers;

use App\Controllers\Auth\AuthController;

class CompanySettings extends AuthController
{
    public function initController($request, $response, $logger)
    {
        parent::initController($request, $response, $logger);
    }

    public function index()
    {
        $this->data['title']    = 'Company Settings';
        $this->data['bcrumb']   = 'Company Settings';

        $this->data['gateways'] = $this->paymentgatewaysModel->getAll();
        $this->data['company']  = $this->companysettingsModel->getCompanyDetails();
        $this->data['policies'] = $this->policysettingsModel->getAll();
        
        // Load required data
        $this->data['currencies'] = $this->companysettingsModel->getCurrencies();
        $this->data['countries'] = $this->companysettingsModel->getCountries();
        $this->data['states'] = $this->stateModel->getAll();
        $this->data['cities'] = $this->cityModel->getAll();

        return view('Modules\Settings\Views\company\index', $this->data);
    }

    public function setCompany()
    {
        if ($this->request->getMethod() === 'POST') {
            $id = $this->request->getPost('id');
            
            $rules = [
                'name'      => ['label' => 'Company Name', 'rules' => 'required|max_length[100]'],
                'email'     => ['label' => 'Company Email', 'rules' => 'required|valid_email'],
                'tel'       => ['label' => 'Company Phone', 'rules' => 'required'],
                'address_1' => ['label' => 'Address', 'rules' => 'required'],
                'country_id' => ['label' => 'Country', 'rules' => 'required|numeric']
            ];

            $messages = [
                'photo' => [
                    'uploaded' => 'Please upload a valid company logo',
                    'mime_in' => 'Only JPG, PNG images are allowed',
                    'max_size' => 'Image size should not exceed 2MB'
                ]
            ];
            
            if (!$this->validate($rules, $messages)) {
                return $this->response->setJSON([
                    'success' => false,
                    'errors' => $this->validator->getErrors(),
                    'message' => 'Validation failed'
                ]);
            }

            // Handle file upload - store only filename
            $logo = $this->request->getFile('photo');
            $logoFileName = null;
            
            if ($logo->isValid() && !$logo->hasMoved()) {
                $newName = $logo->getRandomName();
                $logo->move(ROOTPATH . 'media/files', $newName);
                $logoFileName = $newName;
            }

            $data = [
                'name'          => $this->request->getPost('name'),
                'email'         => $this->request->getPost('email'),
                'tel'           => $this->request->getPost('tel'),
                'mobile'        => $this->request->getPost('mobile'),
                'vision'        => $this->request->getPost('vision'),
                'mission'       => $this->request->getPost('mission'),
                'core_values'   => $this->request->getPost('core_values'),
                'objectives'    => $this->request->getPost('objectives'),
                'address_1'     => $this->request->getPost('address_1'),
                'address_2'     => $this->request->getPost('address_2'),
                'tax_id'        => $this->request->getPost('tax_id'),
                'default_currency_id' => $this->request->getPost('default_currency_id'),
                'area_code'     => $this->request->getPost('area_code'),
                'country_id'    => $this->request->getPost('country_id'),
                'state_id'      => $this->request->getPost('state_id'),
                'city_id'       => $this->request->getPost('city_id'),
                'description'   => $this->request->getPost('description'),
                'website'       => $this->request->getPost('website'),
                'facebook'      => $this->request->getPost('facebook'),
                'twitter'       => $this->request->getPost('twitter'),
                'linkedin'      => $this->request->getPost('linkedin'),
                'instagram'     => $this->request->getPost('instagram'),
                'whatsapp'      => $this->request->getPost('whatsapp'),
                'updated_by'    => $this->userdata->user_id,
                'updated_at'    => date('Y-m-d H:i:s')
            ];

            // Only update photo if new one was uploaded
            if ($logoFileName) {
                $data['photo'] = $logoFileName;
            }

            try {
                $result = false;
                $message = '';
                $notificationMessage = '';

                if ($id) {
                    // Update existing company
                    $result = $this->companysettingsModel->update($id, $data);
                    $message = 'Company details updated successfully';
                    $notificationMessage = 'Company Settings Updated: ' . $this->request->getPost('name');
                } else {
                    // Create new company
                    $data['created_by'] = $this->userdata->user_id;
                    $result = $this->companysettingsModel->insert($data);
                    $message = 'Company details created successfully';
                    $notificationMessage = 'New Company Settings Created: ' . $this->request->getPost('name');
                }

                if ($result) {
                    // Send comprehensive notification for company settings
                    $notificationType = 'settings';
                    $this->sendAdminNotifications($notificationMessage, $notificationType, $this->userdata->user_id);

                    return $this->response->setJSON([
                        'success' => true,
                        'message' => $message,
                        'data' => $data
                    ]);
                } else {
                    return $this->response->setJSON([
                        'success' => false,
                        'message' => 'Failed to save company details'
                    ]);
                }

            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Error saving company details: ' . $e->getMessage()
                ]);
            }
        }

        // GET request handling - for editing
        if ($id = $this->request->getGet('id')) {
            $company = $this->companysettingsModel->getCompanyById($id);
            return $this->response->setJSON([
                'success' => true,
                'data' => $company ? $company : null
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Invalid request method'
        ]);
    }

    // Payment Gateway Settings
    public function updatePaymentGateway()
    {
        if ($this->request->getMethod() === 'POST') {
            $id = $this->request->getPost('id');
            $gatewayName = $this->request->getPost('name');
            
            $data = [
                'is_active' => $this->request->getPost('is_active') ? 1 : 0,
                'test_mode' => $this->request->getPost('test_mode') ? 1 : 0,
                'public_key' => $this->request->getPost('public_key'),
                'private_key' => $this->request->getPost('private_key'),
                'updated_by' => $this->userdata->user_id,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            try {
                if ($this->paymentgatewaysModel->update($id, $data)) {
                    // Send notification for payment gateway update
                    $status = $data['is_active'] ? 'activated' : 'deactivated';
                    $adminmessage = 'Payment Gateway Updated: ' . $gatewayName . ' (' . $status . ')';
                    $notificationType = 'settings';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    return $this->response->setJSON([
                        'success' => true,
                        'message' => 'Payment gateway updated successfully'
                    ]);
                } else {
                    return $this->response->setJSON([
                        'success' => false,
                        'message' => 'Failed to update payment gateway'
                    ]);
                }
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Error updating payment gateway: ' . $e->getMessage()
                ]);
            }
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Invalid request method'
        ]);
    }

    // Policy Settings Management
    public function updatePolicy()
    {
        if ($this->request->getMethod() === 'POST') {
            $id = $this->request->getPost('id');
            $policyName = $this->request->getPost('name');
            
            $data = [
                'content' => $this->request->getPost('content'),
                'is_active' => $this->request->getPost('is_active') ? 1 : 0,
                'updated_by' => $this->userdata->user_id,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            try {
                if ($this->policysettingsModel->update($id, $data)) {
                    // Send notification for policy update
                    $status = $data['is_active'] ? 'activated' : 'deactivated';
                    $adminmessage = 'Policy Updated: ' . $policyName . ' (' . $status . ')';
                    $notificationType = 'settings';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    return $this->response->setJSON([
                        'success' => true,
                        'message' => 'Policy updated successfully'
                    ]);
                } else {
                    return $this->response->setJSON([
                        'success' => false,
                        'message' => 'Failed to update policy'
                    ]);
                }
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Error updating policy: ' . $e->getMessage()
                ]);
            }
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Invalid request method'
        ]);
    }

    // Add new policy
    public function addPolicy()
    {
        if ($this->request->getMethod() === 'POST') {
            $policyName = $this->request->getPost('name');
            
            $data = [
                'name' => $policyName,
                'slug' => preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($policyName))),
                'content' => $this->request->getPost('content'),
                'is_active' => $this->request->getPost('is_active') ? 1 : 0,
                'created_by' => $this->userdata->user_id,
                'created_at' => date('Y-m-d H:i:s')
            ];

            try {
                if ($this->policysettingsModel->insert($data)) {
                    // Send notification for new policy creation
                    $adminmessage = 'New Policy Created: ' . $policyName;
                    $notificationType = 'settings';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    return $this->response->setJSON([
                        'success' => true,
                        'message' => 'Policy created successfully'
                    ]);
                } else {
                    return $this->response->setJSON([
                        'success' => false,
                        'message' => 'Failed to create policy'
                    ]);
                }
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Error creating policy: ' . $e->getMessage()
                ]);
            }
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Invalid request method'
        ]);
    }

    // Delete policy
    public function deletePolicy($id)
    {
        $policy = $this->policysettingsModel->find($id);
        
        if (!$policy) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Policy not found'
            ]);
        }

        $policyName = $policy['name'] ?? 'Unknown Policy';
        
        try {
            if ($this->policysettingsModel->delete($id)) {
                // Send notification for policy deletion
                $adminmessage = 'Policy Deleted: ' . $policyName;
                $notificationType = 'settings';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Policy deleted successfully'
                ]);
            } else {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Failed to delete policy'
                ]);
            }
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error deleting policy: ' . $e->getMessage()
            ]);
        }
    }

    // Bulk settings operations
    public function bulkAction()
    {
        if ($this->request->getMethod() == 'POST') {
            $action = $this->request->getVar('action');
            $itemType = $this->request->getVar('item_type'); // 'policy' or 'payment_gateway'
            $itemIds = $this->request->getVar('item_ids');

            if (empty($itemIds)) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'No items selected'
                ]);
            }

            $successCount = 0;
            $failedCount = 0;

            foreach ($itemIds as $itemId) {
                switch ($itemType) {
                    case 'policy':
                        $model = $this->policysettingsModel;
                        $item = $model->find($itemId);
                        break;
                    
                    case 'payment_gateway':
                        $model = $this->paymentgatewaysModel;
                        $item = $model->find($itemId);
                        break;
                    
                    default:
                        $failedCount++;
                        continue 2; // Continue to next item
                }

                if (!$item) {
                    $failedCount++;
                    continue;
                }

                switch ($action) {
                    case 'activate':
                        if ($model->update($itemId, ['is_active' => 1])) {
                            $successCount++;
                        } else {
                            $failedCount++;
                        }
                        break;
                    
                    case 'deactivate':
                        if ($model->update($itemId, ['is_active' => 0])) {
                            $successCount++;
                        } else {
                            $failedCount++;
                        }
                        break;
                    
                    case 'delete':
                        if ($model->delete($itemId)) {
                            $successCount++;
                        } else {
                            $failedCount++;
                        }
                        break;
                    
                    default:
                        $failedCount++;
                        break;
                }
            }

            if ($successCount > 0) {
                // Send notification for bulk action
                $itemTypeName = $itemType == 'policy' ? 'Policies' : 'Payment Gateways';
                $adminmessage = 'Bulk ' . $itemTypeName . ' Action: ' . $action . ' performed on ' . $successCount . ' items';
                $notificationType = 'settings';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                return $this->response->setJSON([
                    'success' => true,
                    'message' => $successCount . ' items processed successfully'
                ]);
            }

            if ($failedCount > 0) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $failedCount . ' items failed to process'
                ]);
            }
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Invalid request'
        ]);
    }

    // Add method to get states and cities for dropdowns
    public function getStates($countryId)
    {
        $states = $this->countryModel->getStates($countryId);
        return $this->response->setJSON($states);
    }

    public function getCities($stateId)
    {
        $cities = $this->stateModel->getCities($stateId);
        return $this->response->setJSON($cities);
    }
}