<?php

namespace Modules\Posts\Models;

use CodeIgniter\Model;

class ServiceGroupsModel extends Model
{
    // Use the prefixed table name used across the app
    protected $table = 'usc_service_groups';
    protected $primaryKey = 'id';
    
    protected $allowedFields = ['name', 'slug', 'description', 'is_active', 'preview_image', 'created_by', 'created_at', 'updated_at', 'deleted_at'];
    
    protected $returnType = 'array';
    protected $useTimestamps = true;
    protected $createdField = 'created_at';
    
    /**
     * Get all service groups
     */
    public function getAll()
    {
        return $this->findAll();
    }
    
    /**
     * Get service group by slug
     */
    public function getBySlug($slug)
    {
        return $this->where('slug', $slug)->first();
    }

    /**
     * Return active service groups for a given nav context slug
     * joins the pivot `usc_service_group_nav_contexts` and `usc_nav_contexts`
     * Returns array of arrays
     */
    public function getByNavContext($slug)
    {
        $builder = $this->db->table('usc_service_groups as sg');
        $builder->select('sg.*');
        $builder->join('usc_service_group_nav_contexts as sgnc', 'sgnc.service_group_id = sg.id', 'inner');
        $builder->join('usc_nav_contexts as nc', 'nc.id = sgnc.nav_context_id', 'inner');
        $builder->where('nc.slug', $slug);
        $builder->where('sg.is_active', 1);
        $builder->orderBy('sg.name', 'ASC');
        return $builder->get()->getResultArray();
    }

    /**
     * Atomically set nav contexts for a given service group
     * Replaces existing pivot rows with the provided list
     * @param int $serviceGroupId
     * @param array $navContextIds
     * @return bool
     */
    public function setNavContexts(int $serviceGroupId, array $navContextIds)
    {
        $pivot = $this->db->table('usc_service_group_nav_contexts');
        $this->db->transStart();
        // delete existing
        $pivot->where('service_group_id', $serviceGroupId)->delete();

        if (!empty($navContextIds)) {
            $batch = [];
            foreach ($navContextIds as $ncid) {
                $batch[] = ['service_group_id' => $serviceGroupId, 'nav_context_id' => (int)$ncid];
            }
            $pivot->insertBatch($batch);
        }

        $this->db->transComplete();
        return $this->db->transStatus();
    }
}
