<?php

namespace Modules\Posts\Models;

use CodeIgniter\Model;

class PostBlocksModel extends Model
{

  protected $table = 'usc_post_blocks';
  protected $primaryKey = 'id';

    protected $allowedFields = ['post_id', 'block_type', 'block_cat', 'content', 'heading', 'heading_type', 'color', 'photo', 'caption', 'list_type', 'code', 'advert_code', 'ordering', 'status', 'block_definition_id', 'title_emphasized', 'subtitle', 'settings', 'custom_css', 'custom_classes'];

  protected $returnType     = 'array';
  protected $useTimestamps  = true;
  protected $useSoftDeletes = true;
  
  protected $createdField  = 'created_at';
  protected $updatedField  = 'updated_at';
  protected $deletedField  = 'deleted_at';

  public function getSingleBlockPost($post)
  {
      $builder = $this->db->table('usc_post_blocks');
      $builder->select('usc_post_blocks.*, usc_posts.title')
              ->join('usc_posts', 'usc_posts.id=usc_post_blocks.post_id')
              ->where('usc_post_blocks.deleted_at', null)
              ->where('usc_post_blocks.post_id', $post);
      $blocks = $builder->get()->getResult();

      // Attach child items to each block
      foreach ($blocks as &$block) {
          $block->children = $this->db->table('usc_post_block_items')
              ->where('deleted_at', null)
              ->where('block_id', $block->id)
              ->orderBy('ordering', 'ASC')
              ->get()
              ->getResult();
      }

      return $blocks;
  }

    /**
     * Get next ordering number for a given post_id
     */
    public function getNextOrdering($post_id)
    {
        $maxOrder = $this->where('post_id', $post_id)
                         ->selectMax('ordering')
                         ->first();

        return isset($maxOrder['ordering']) ? ($maxOrder['ordering'] + 1) : 1;
    }

    /**
     * Insert a new block with auto-ordering
     */
    public function insertWithOrdering($data)
    {
        if (!isset($data['post_id'])) {
            throw new \Exception("post_id is required.");
        }

        $data['ordering'] = $this->getNextOrdering($data['post_id']);

        return $this->insert($data);
    }

    /**
     * Get blocks for a specific page by alias (for Personal/Business pages)
     */
    public function getPageBlocks($pageAlias)
    {
        $builder = $this->db->table('usc_post_blocks as pb');
        $builder->select('pb.*, p.title as post_title, p.alias as post_alias');
        $builder->join('usc_posts as p', 'p.id = pb.post_id');
        $builder->where('pb.deleted_at', null);
        $builder->where('pb.status', 'active');
        $builder->where('p.alias', $pageAlias);
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->orderBy('pb.ordering', 'ASC');
        
        $blocks = $builder->get()->getResultArray();

        // Attach child items to each block
        foreach ($blocks as &$block) {
            $block['items'] = $this->db->table('usc_post_block_items')
                ->where('deleted_at', null)
                ->where('status', 'active')
                ->where('block_id', $block['id'])
                ->orderBy('ordering', 'ASC')
                ->get()
                ->getResultArray();
        }

        return $blocks;
    }

    /**
     * Get service cards grouped by category for Personal/Business pages
     */
    public function getServiceCardsByCategory($pageAlias, $category = null)
    {
        $builder = $this->db->table('usc_post_block_items as pbi');
        $builder->select('pbi.*, pb.post_id, p.alias as post_alias');
        $builder->join('usc_post_blocks as pb', 'pb.id = pbi.block_id');
        $builder->join('usc_posts as p', 'p.id = pb.post_id');
        $builder->where('pbi.deleted_at', null);
        $builder->where('pbi.status', 'active');
        $builder->where('pbi.item_type', 'cards');
        $builder->where('p.alias', $pageAlias);
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        
        if ($category) {
            $builder->where('pbi.caption', $category);
        }
        
        $builder->orderBy('pbi.ordering', 'ASC');
        return $builder->get()->getResultArray();
    }

    /**
     * Get service counts by category for Personal/Business pages
     */
    public function getServiceCountsByCategory($pageAlias)
    {
        $builder = $this->db->table('usc_post_block_items as pbi');
        $builder->select('pbi.caption, COUNT(*) as count');
        $builder->join('usc_post_blocks as pb', 'pb.id = pbi.block_id');
        $builder->join('usc_posts as p', 'p.id = pb.post_id');
        $builder->where('pbi.deleted_at', null);
        $builder->where('pbi.status', 'active');
        $builder->where('pbi.item_type', 'cards');
        $builder->where('p.alias', $pageAlias);
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->groupBy('pbi.caption');
        
        $results = $builder->get()->getResultArray();
        
        $counts = [];
        foreach ($results as $result) {
            $counts[$result['caption']] = (int)$result['count'];
        }
        
        return $counts;
    }

    /**
     * Get blocks for a specific post by post_id (for individual service pages)
     */
    public function getPostBlocks($postId)
    {
        $builder = $this->db->table('usc_post_blocks as pb');
        $builder->select('pb.*');
        $builder->where('pb.deleted_at', null);
        $builder->where('pb.status', 'active');
        $builder->where('pb.post_id', $postId);
        $builder->orderBy('pb.ordering', 'ASC');

        $blocks = $builder->get()->getResultArray();

        // Attach child items to each block
        foreach ($blocks as &$block) {
            $block['items'] = $this->db->table('usc_post_block_items')
                ->where('deleted_at', null)
                ->where('status', 'active')
                ->where('block_id', $block['id'])
                ->orderBy('ordering', 'ASC')
                ->get()
                ->getResultArray();
        }

        return $blocks;
    }

    protected function shiftOrdering($postId, $newOrder, $blockId)
    {
        // Get all blocks ordered by ordering
        $blocks = $this->where('post_id', $postId)
                    ->where('id !=', $blockId)
                    ->orderBy('ordering', 'ASC')
                    ->findAll();

        $order = 1;
        foreach ($blocks as $b) {
            if ($order == $newOrder) {
                $order++; // leave space for updated block
            }
            $this->update($b['id'], ['ordering' => $order]);
            $order++;
        }
    }

    public function updateWithOrdering($id, $data)
    {
        // Get existing block info
        $block = $this->find($id);
        
        if (!$block) {
            return false; // Block not found
        }

        // Get max order for the post
        $maxOrder = $this->where('post_id', $block['post_id'])
                        ->selectMax('ordering')
                        ->first();
                        
        $maxOrder = $maxOrder['ordering'] ?? 0;
        
        // Validate new order is within bounds
        if (isset($data['ordering']) && ($data['ordering'] < 1 || $data['ordering'] > $maxOrder + 1)) {
            throw new \Exception('Invalid order value. Must be between 1 and ' . ($maxOrder + 1));
        }

        $postIdChanged = isset($data['post_id']) && $data['post_id'] != $block['post_id'];
        $orderingChanged = isset($data['ordering']) && $data['ordering'] != $block['ordering'];

        // If block moved to another post OR ordering changed
        if ($postIdChanged || $orderingChanged) {
            if ($postIdChanged) {
                // If moved to another post, insert at bottom of that post's list
                $maxOrder = $this->where('post_id', $data['post_id'])->selectMax('ordering')->first();
                $data['ordering'] = $maxOrder['ordering'] ? $maxOrder['ordering'] + 1 : 1;

                // Reorder old post list
                $this->reorderBlocks($block['post_id']);
            } else if ($orderingChanged) {
                // If ordering manually changed in same post, shift others
                $this->shiftOrdering($block['post_id'], $data['ordering'], $id);
            }
        }

        // Finally update
        return $this->update($id, $data);
    }

    /**
     * Delete a block and reorder remaining
     */
    public function deleteAndReorder($id)
    {
        // Get post_id of the block
        $block = $this->find($id);
        if (!$block) {
            return false;
        }

        $post_id = $block['post_id'];

        // Delete the block
        $this->delete($id);

        // Reorder remaining
        $this->reorderBlocks($post_id);

        return true;
    }

    /**
     * Reorder blocks for a specific post
     */
    private function reorderBlocks($postId)
    {
        $blocks = $this->where('post_id', $postId)
                    ->orderBy('ordering', 'ASC')
                    ->findAll();

        $order = 1;
        foreach ($blocks as $block) {
            $this->update($block['id'], ['ordering' => $order]);
            $order++;
        }
    }




    /**
 * Get allowed item types for this block based on block definition
 */
public function getAllowedItemTypes($blockId)
{
    // Get the block to find its definition
    $block = $this->find($blockId);
    if (!$block || !$block['block_definition_id']) {
        return [];
    }

    $itemTypeModel = new \Modules\Posts\Models\ItemTypeDefinitionsModel();
    return $itemTypeModel->getByBlockDefinition($block['block_definition_id']);
}

/**
 * Get block with allowed item types
 */
public function getBlockWithItemTypes($blockId)
{
    $block = $this->find($blockId);
    if (!$block) {
        return null;
    }

    $block['allowed_item_types'] = $this->getAllowedItemTypes($blockId);
    $block['items'] = $this->db->table('usc_post_block_items')
        ->where('deleted_at', null)
        ->where('block_id', $blockId)
        ->orderBy('ordering', 'ASC')
        ->get()
        ->getResultArray();

    return $block;
}
public function getBlockItems($blockId)
{
    $builder = $this->db->table('usc_post_block_items');
    return $builder->where('block_id', $blockId)
                  ->where('status', 'active')
                  ->where('deleted_at IS NULL')
                  ->orderBy('ordering', 'ASC')
                  ->get()
                  ->getResultArray();
}

    /**
     * Add block using business rules (checks definition max_instances etc.)
     * Returns array with success/message and blockId when created.
     */
    public function addBlock(array $data)
    {
        $blockDefinitionsModel = new \Modules\Posts\Models\BlockDefinitionsModel();

        $definition = $blockDefinitionsModel->find($data['definition_id'] ?? null);
        if (!$definition) {
            return ['success' => false, 'message' => 'Block definition not found', 'code' => 404];
        }

        // Check max instances
        if (!empty($definition['max_instances']) && $definition['max_instances'] > 0) {
            $currentCount = $this->where('post_id', $data['post_id'])
                                 ->where('block_type', $definition['slug'])
                                 ->countAllResults();

            if ($currentCount >= $definition['max_instances']) {
                return ['success' => false, 'message' => 'Maximum instances (' . $definition['max_instances'] . ') reached for this block type', 'code' => 400];
            }
        }

        // Build block data including optional fields supported by the model
        $blockData = [
            'post_id'            => $data['post_id'],
            'block_type'         => $definition['slug'],
            'block_definition_id'=> $definition['id'],
            'block_cat'          => $data['block_cat'] ?? 'content',
            'heading'            => $data['heading'] ?? null,
            'heading_type'       => $data['heading_type'] ?? null,
            'title_emphasized'   => $data['title_emphasized'] ?? null,
            'subtitle'           => $data['subtitle'] ?? null,
            'color'              => $data['color'] ?? null,
            'content'            => $data['content'] ?? null,
            'photo'              => $data['photo'] ?? null,
            'caption'            => $data['caption'] ?? null,
            'list_type'          => $data['list_type'] ?? null,
            'code'               => $data['code'] ?? null,
            'advert_code'        => $data['advert_code'] ?? null,
            'custom_css'         => $data['custom_css'] ?? null,
            'custom_classes'     => $data['custom_classes'] ?? null,
            'ordering'           => $data['ordering'] ?? 0,
            'status'             => $data['status'] ?? 'active'
        ];

        // Normalize settings: accept array or JSON string
        if (isset($data['settings'])) {
            if (is_array($data['settings'])) {
                $blockData['settings'] = json_encode($data['settings']);
            } else {
                $blockData['settings'] = $data['settings'];
            }
        }

        try {
            // If ordering not provided (or <= 0) use insertWithOrdering to calculate next ordering
            if (empty($blockData['ordering']) || (int)$blockData['ordering'] <= 0) {
                $insertResult = $this->insertWithOrdering($blockData);
                if ($insertResult) {
                    return ['success' => true, 'message' => 'Block added successfully', 'blockId' => $this->getInsertID()];
                }
                return ['success' => false, 'message' => 'Failed to add block', 'code' => 500];
            }

            if ($this->save($blockData)) {
                return ['success' => true, 'message' => 'Block added successfully', 'blockId' => $this->getInsertID()];
            }

            return ['success' => false, 'message' => 'Failed to add block: ' . implode(', ', $this->errors()), 'code' => 500];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'Error adding block: ' . $e->getMessage(), 'code' => 500];
        }
    }

    /**
     * Delete a block along with its items and reorder remaining blocks.
     * Returns boolean true on success, false otherwise.
     */
    public function deleteBlock($id)
    {
        $postBlockItemsModel = new \Modules\Posts\Models\PostBlockItemsModel();
        // delete associated items (soft delete via model delete())
        $postBlockItemsModel->where('block_id', $id)->delete();

        // use existing deleteAndReorder to delete and reorder
        return $this->deleteAndReorder($id);
    }
}