<?php

namespace Modules\Posts\Models;

use CodeIgniter\Model;

class PostBlockItemsModel extends Model
{
    protected $table = 'usc_post_block_items';
    protected $primaryKey = 'id';

    protected $allowedFields = [
        'block_id', 'item_type', 'title', 'icon', 'value', 'description', 
        'width', 'photo', 'caption', 'link', 'ordering', 'status'
    ];

    protected $returnType     = 'array';
    protected $useTimestamps  = true;
    protected $useSoftDeletes = true;
    
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    public function singleBlockItem($id)
    {
      $builder = $this->db->table('usc_post_block_items');
      $builder->select('usc_post_block_items.*, usc_post_blocks.post_id, usc_posts.alias as post_alias ')
              ->join('usc_post_blocks', 'usc_post_blocks.id=usc_post_block_items.block_id', 'left')
              ->join('usc_posts', 'usc_posts.id=usc_post_blocks.post_id', 'left')
              ->where('usc_post_block_items.id', $id);
      return $result = $builder->get()->getRow();
    }

/**
 * Get next ordering number for a given block_id
 */
public function getNextOrdering($block_id)
{
    $maxOrder = $this->where('block_id', $block_id)
                     ->selectMax('ordering')
                     ->first();

    return isset($maxOrder['ordering']) ? ($maxOrder['ordering'] + 1) : 1;
}

/**
 * Insert a new item with auto-ordering
 */
public function insertWithOrdering($data)
{
    if (!isset($data['block_id'])) {
        throw new \Exception("block_id is required.");
    }

    $data['ordering'] = $this->getNextOrdering($data['block_id']);

    return $this->insert($data);
}

/**
 * Get items grouped by item_type for a block
 */
public function getItemsGroupedByType($blockId)
{
    $items = $this->where('block_id', $blockId)
                  ->where('deleted_at', null)
                  ->orderBy('item_type', 'ASC')
                  ->orderBy('ordering', 'ASC')
                  ->findAll();

    $grouped = [];
    foreach ($items as $item) {
        $grouped[$item['item_type']][] = $item;
    }

    return $grouped;
}

    /**
     * Add an item to a block, honoring block definition constraints.
     * Returns array with success/message.
     */
    public function addItem(array $data)
    {
        $postBlocksModel = new \Modules\Posts\Models\PostBlocksModel();
        $blockDefinitionsModel = new \Modules\Posts\Models\BlockDefinitionsModel();

        // Ensure block exists
        $block = $postBlocksModel->find($data['block_id'] ?? null);
        if (!$block) {
            return ['success' => false, 'message' => 'Block not found', 'code' => 404];
        }

        // Get block definition to ensure block supports items (repeatable)
        $definition = $blockDefinitionsModel->where('slug', $block['block_type'])->first();
        if (!$definition || empty($definition['is_repeatable'])) {
            return ['success' => false, 'message' => 'This block type does not support items', 'code' => 400];
        }

        // Set defaults
        $data['item_type'] = $data['item_type'] ?? 'text';
        $data['ordering'] = $data['ordering'] ?? $this->getNextOrdering($data['block_id']);
        $data['status'] = $data['status'] ?? 'active';

        if ($this->save($data)) {
            return ['success' => true, 'message' => 'Item added successfully'];
        }

        return ['success' => false, 'message' => 'Failed to add item', 'code' => 500];
    }

    /**
     * Return HTML table for block items (used by AJAX in controller).
     */
    public function getHtmlForBlock($blockId)
    {
        $items = $this->where('block_id', $blockId)
                      ->where('deleted_at', null)
                      ->orderBy('ordering', 'ASC')
                      ->findAll();

        if (!$items) {
            return '';
        }

        $html = '<div class="table-responsive"><table class="table table-sm">';
        $html .= '<thead><tr><th>Title</th><th>Type</th><th>Order</th><th>Status</th><th>Actions</th></tr></thead>';
        $html .= '<tbody>';

        foreach ($items as $item) {
            $title = isset($item['title']) ? esc($item['title']) : 'N/A';
            $type = isset($item['item_type']) ? $item['item_type'] : '';
            $order = isset($item['ordering']) ? $item['ordering'] : 0;
            $statusHtml = (isset($item['status']) && $item['status'] == 'active') ? '<span class="badge bg-success">Active</span>' : '<span class="badge bg-secondary">Inactive</span>';

            $html .= '<tr>';
            $html .= '<td>' . $title . '</td>';
            $html .= '<td><code>' . $type . '</code></td>';
            $html .= '<td><span class="badge bg-primary">' . $order . '</span></td>';
            $html .= '<td>' . $statusHtml . '</td>';
            $html .= '<td>';
            $html .= '<button class="btn btn-sm btn-outline-primary edit-item" data-id="' . $item['id'] . '"><i class="bi bi-pencil"></i></button> ';
            $html .= '<button class="btn btn-sm btn-outline-danger delete-item" data-id="' . $item['id'] . '"><i class="bi bi-trash"></i></button>';
            $html .= '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody></table></div>';

        return $html;
    }
}