<?php

namespace Modules\Posts\Models;

use CodeIgniter\Model;

class ItemTypeDefinitionsModel extends Model
{
    protected $table = 'usc_item_type_definitions';
    protected $primaryKey = 'id';

    protected $allowedFields = [
        'name', 'slug', 'description', 'icon', 'category', 
        'fields_config', 'settings_config', 
        'is_repeatable', 'max_instances', 'preview_image', 
        'is_active', 'created_by'
    ];

    protected $returnType = 'array';
    protected $useTimestamps = true;
    protected $useSoftDeletes = true;
    
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    /**
     * Get active item type definitions
     */
    public function getActiveItemTypes()
    {
        return $this->where('is_active', 1)
                    ->where('deleted_at', null)
                    ->orderBy('name', 'ASC')
                    ->findAll();
    }

    /**
     * Get item types by category
     */
    public function getByCategory($category)
    {
        return $this->where('category', $category)
                    ->where('is_active', 1)
                    ->where('deleted_at', null)
                    ->orderBy('name', 'ASC')
                    ->findAll();
    }

    /**
     * Get item type by slug
     */
    public function getBySlug($slug)
    {
        return $this->where('slug', $slug)
                    ->where('deleted_at', null)
                    ->first();
    }

    /**
     * Get item types allowed for a block definition
     */
    public function getByBlockDefinition($blockDefinitionId)
    {
        return $this->select('itd.*, rel.is_default, rel.max_items, rel.ordering')
                    ->from('usc_item_type_definitions itd')
                    ->join('usc_block_item_type_relationships rel', 'rel.item_type_definition_id = itd.id')
                    ->where('rel.block_definition_id', $blockDefinitionId)
                    ->where('itd.is_active', 1)
                    ->where('itd.deleted_at', null)
                    ->orderBy('rel.ordering', 'ASC')
                    ->findAll();
    }

    /**
     * Delete item type safely after ensuring no items use this type.
     * Returns an array with success/message and optional code.
     */
    public function deleteSafely($id)
    {
        $itemType = $this->find($id);
        if (!$itemType) {
            return ['success' => false, 'message' => 'Item type not found', 'code' => 404];
        }

        $postBlockItemsModel = new \Modules\Posts\Models\PostBlockItemsModel();
        $itemsUsingType = $postBlockItemsModel->where('item_type', $itemType['slug'])->countAllResults();

        if ($itemsUsingType > 0) {
            return [
                'success' => false,
                'message' => 'Cannot delete item type. There are ' . $itemsUsingType . ' items using this type.',
                'code' => 400,
                'count' => $itemsUsingType
            ];
        }

        if ($this->delete($id)) {
            return ['success' => true, 'message' => 'Item type deleted successfully'];
        }

        return ['success' => false, 'message' => 'Failed to delete item type', 'code' => 500];
    }
}