<?php

namespace Modules\Posts\Models;

use CodeIgniter\Model;

class BlockDefinitionsModel extends Model
{
    protected $table = 'block_definitions';
    protected $primaryKey = 'id';

    protected $allowedFields = [
        'name', 'slug', 'description', 'icon', 'category', 
        'fields_config', 'settings_config', 'template_file', 
        'is_repeatable', 'max_instances', 'preview_image', 
        'is_active', 'created_by'
    ];

    protected $returnType     = 'array';
    protected $useTimestamps  = true;
    protected $useSoftDeletes = true;
    
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    /**
     * Get active block definitions
     */
    public function getActiveBlocks()
    {
        return $this->where('is_active', 1)
                    ->where('deleted_at', null)
                    ->orderBy('name', 'ASC')
                    ->findAll();
    }

    /**
     * Get blocks by category
     */
    public function getByCategory($category)
    {
        return $this->where('category', $category)
                    ->where('is_active', 1)
                    ->where('deleted_at', null)
                    ->orderBy('name', 'ASC')
                    ->findAll();
    }

    /**
     * Get block definition by slug
     */
    public function getBySlug($slug)
    {
        return $this->where('slug', $slug)
                    ->where('deleted_at', null)
                    ->first();
    }

    /**
     * Get a definition along with associated item types
     */
    public function getWithItemTypes($id)
    {
        $definition = $this->find($id);
        if (!$definition) {
            return null;
        }

        $relModel = new \Modules\Posts\Models\BlockItemTypeRelationshipsModel();
        $definition['item_types'] = $relModel->where('block_definition_id', $id)->findAll();

        return $definition;
    }

    /**
     * Delete definition safely after checking for blocks using it.
     * Returns an array with success/message and optional code.
     */
    public function deleteSafely($id)
    {
        $definition = $this->find($id);
        if (!$definition) {
            return ['success' => false, 'message' => 'Definition not found', 'code' => 404];
        }

        $postBlocksModel = new \Modules\Posts\Models\PostBlocksModel();
        $blocksUsingDefinition = $postBlocksModel->where('block_type', $definition['slug'])->countAllResults();

        if ($blocksUsingDefinition > 0) {
            return [
                'success' => false,
                'message' => 'Cannot delete definition. There are ' . $blocksUsingDefinition . ' blocks using this definition.',
                'code' => 400,
                'count' => $blocksUsingDefinition
            ];
        }

        if ($this->delete($id)) {
            return ['success' => true, 'message' => 'Block definition deleted'];
        }

        return ['success' => false, 'message' => 'Failed to delete block definition', 'code' => 500];
    }
}
