<?php

namespace Modules\Posts\Controllers;

use App\Controllers\Auth\AuthController;

class Tags extends AuthController
{
    public function initController($request, $response, $logger)
    {
        parent::initController($request, $response, $logger);
        $this->data['tags'] = $this->tagsModel->getAll();
    }

    public function index()
    {
        // Display header information
        $this->data['title'] = "Tags Manager";
        $this->data['bcrumb'] = "Tags";

        return view('Modules\Posts\Views\tags\tags', $this->data);
    }

    public function getTagsList()
    {
        try {
            $tags = $this->tagsModel->getAllTags();
            $response = [
                'tags' => $tags,
                'success' => true
            ];
            return $this->response->setJSON($response);
        } catch (\Exception $e) {
            $response = [
                'success' => false,
                'msg' => 'Failed to retrieve tags'
            ];
            return $this->response->setStatusCode(500)->setJSON($response);
        }
    }

    // Check if Tag Exists in DB
    public function checkTag()
    {
        $tagName = $this->request->getPost('name');
        $tag = $this->tagsModel->where('name', $tagName)->first();

        $response = [
            'exists' => ($tag !== null),
            'tagId' => ($tag !== null) ? $tag['id'] : null,
        ];
        return $this->response->setJSON($response);
    }

    public function addTag()
    { 
        $this->data['title'] = "Add Tag";
        $this->data['bcrumb'] = "Tags";

        if ($this->request->getMethod() == 'POST') {
            $rules = [
                'name' => [
                    'label' => 'Tag Name',
                    'rules' => 'required|min_length[2]|max_length[100]',
                    'errors' => [
                        'required' => 'Tag name is required',
                        'min_length' => 'Tag name must be at least 2 characters',
                        'max_length' => 'Tag name cannot exceed 100 characters'
                    ]
                ],
            ];

            if (!$this->validate($rules)) {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => 'Validation failed',
                        'errors' => $this->validator->getErrors()
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'Please check the form for errors', 3);
                    return redirect()->back()->withInput()->with('validation', $this->validator);
                }
            }

            $tagName = trim($this->request->getVar('name'));
            
            // Check if tag already exists
            $existingTag = $this->tagsModel
                ->where('name', $tagName)
                ->first();
            
            if ($existingTag) {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'error' => true,
                        'msg' => 'Tag already exists'
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'Tag already exists', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            }

            $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($tagName)));
            
            // Check if slug exists (in case different name creates same slug)
            $existingSlug = $this->tagsModel
                ->where('alias', $slug)
                ->first();
            
            if ($existingSlug) {
                // Append timestamp to make slug unique
                $slug = $slug . '-' . time();
            }

            $tagData = [
                'name' => $tagName,
                'alias' => $slug,
                'description' => $this->request->getVar('description') ?? '',
                'created_by' => $this->userdata->user_id,
            ];
            
            if ($this->tagsModel->save($tagData)) {
                // Send notification for tag creation
                $adminmessage = 'New Tag Created: ' . $tagName;
                $notificationType = 'content';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => true,
                        'msg' => 'Tag added successfully',
                        'tag' => [
                            'id' => $this->tagsModel->getInsertID(),
                            'name' => $tagName,
                            'alias' => $slug
                        ]
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('success', 'Tag added successfully', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            } else {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => 'Failed to add tag. Please try again.'
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'Failed to add tag', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            }
        }
        
        return view('Modules\Posts\Views\tags\tags', $this->data);
    }

    public function editTag($id = null)
    {
        if (empty($id)) {
            session()->setTempdata('error', 'Invalid tag ID', 3);
            return redirect()->to('administrator/posts/tags');
        }

        $this->data['title'] = "Edit Tag";
        $this->data['bcrumb'] = "Edit Tag";

        $tag = $this->tagsModel->find($id);
        
        if (!$tag) {
            session()->setTempdata('error', 'Tag not found', 3);
            return redirect()->to('administrator/posts/tags');
        }

        $this->data['singleTag'] = $tag;
        return view('Modules\Posts\Views\tags\tags', $this->data);
    }

    public function updateTag()
    {
        $this->data['title'] = "Edit Tag";
        $this->data['bcrumb'] = "Edit Tag";
    
        if ($this->request->getMethod() == 'POST') {
            $id = $this->request->getVar('id');
            
            if (empty($id)) {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => 'Invalid tag ID'
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'Invalid tag ID', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            }

            $existingTag = $this->tagsModel->find($id);
            
            if (!$existingTag) {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => 'Tag not found'
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'Tag not found', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            }

            $rules = [
                'name' => [
                    'label' => 'Tag Name',
                    'rules' => 'required|min_length[2]|max_length[100]',
                    'errors' => [
                        'required' => 'Tag name is required',
                        'min_length' => 'Tag name must be at least 2 characters',
                        'max_length' => 'Tag name cannot exceed 100 characters'
                    ]
                ],
            ];

            if (!$this->validate($rules)) {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => 'Validation failed',
                        'errors' => $this->validator->getErrors()
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'Please check the form for errors', 3);
                    return redirect()->back()->withInput()->with('validation', $this->validator);
                }
            }

            $tagName = trim($this->request->getVar('name'));
            
            // Check if new name conflicts with another tag
            $conflict = $this->tagsModel
                ->where('name', $tagName)
                ->where('id !=', $id)
                ->first();
            
            if ($conflict) {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => 'A tag with this name already exists'
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'A tag with this name already exists', 3);
                    return redirect()->back();
                }
            }

            $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($tagName)));
            
            // Check if new slug conflicts with another tag
            $slugConflict = $this->tagsModel
                ->where('alias', $slug)
                ->where('id !=', $id)
                ->first();
            
            if ($slugConflict) {
                // Append timestamp to make slug unique
                $slug = $slug . '-' . time();
            }

            $tagData = [
                'name' => $tagName,
                'alias' => $slug,
                'description' => $this->request->getVar('description') ?? '',
                'updated_at' => date("Y-m-d H:i:s"),
                'updated_by' => $this->userdata->user_id,
            ];
            
            if ($this->tagsModel->update($id, $tagData)) {
                // Send notification for tag update
                $adminmessage = 'Tag Updated: ' . $tagName;
                $notificationType = 'content';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => true,
                        'msg' => 'Tag updated successfully',
                        'tag' => [
                            'id' => $id,
                            'name' => $tagName,
                            'alias' => $slug
                        ]
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('success', 'Tag updated successfully', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            } else {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => 'Failed to update tag. Please try again.'
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'Failed to update tag', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            }
        }
        
        return view('Modules\Posts\Views\tags\tags', $this->data);
    }

    public function delTag($id)
    {
        if (empty($id)) {
            if ($this->request->isAJAX()) {
                $response = [
                    'success' => false,
                    'msg' => 'Invalid tag ID'
                ];
                return $this->response->setJSON($response);
            } else {
                session()->setTempdata('error', 'Invalid tag ID', 3);
                return redirect()->to('administrator/posts/tags');
            }
        }

        $tag = $this->tagsModel->find($id);
        
        if (!$tag) {
            if ($this->request->isAJAX()) {
                $response = [
                    'success' => false,
                    'msg' => 'Tag not found'
                ];
                return $this->response->setJSON($response);
            } else {
                session()->setTempdata('error', 'Tag not found', 3);
                return redirect()->to('administrator/posts/tags');
            }
        }

        $tagName = $tag['name'] ?? 'Unknown Tag';
        
        // Check if tag is being used by posts
        $postTagsModel = new \Modules\Posts\Models\PostTagsModel();
        $usageCount = $postTagsModel->where('tag_id', $id)->countAllResults();
        
        if ($usageCount > 0) {
            $message = 'Cannot delete this tag. It is being used by ' . $usageCount . ' post(s). Please remove the tag from all posts first.';
            
            if ($this->request->isAJAX()) {
                $response = [
                    'success' => false,
                    'msg' => $message
                ];
                return $this->response->setJSON($response);
            } else {
                session()->setTempdata('error', $message, 3);
                return redirect()->to('administrator/posts/tags');
            }
        }
        
        if ($this->tagsModel->delete($id)) {
            // Send notification for tag deletion
            $adminmessage = 'Tag Deleted: ' . $tagName;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            if ($this->request->isAJAX()) {
                $response = [
                    'success' => true,
                    'msg' => 'Tag deleted successfully'
                ];
                return $this->response->setJSON($response);
            } else {
                session()->setTempdata('success', 'Tag deleted successfully', 3);
                return redirect()->to('administrator/posts/tags');
            }
        } else {
            if ($this->request->isAJAX()) {
                $response = [
                    'success' => false,
                    'msg' => 'Failed to delete tag. Please try again.'
                ];
                return $this->response->setJSON($response);
            } else {
                session()->setTempdata('error', 'Failed to delete tag', 3);
                return redirect()->to('administrator/posts/tags');
            }
        }
    }

    public function searchTags()
    {
        try {
            $searchTerm = $this->request->getGet('search');
            $tags = $this->tagsModel->searchTags($searchTerm);
            $response = [
                'tags' => $tags,
                'success' => true
            ];
            return $this->response->setJSON($response);
        } catch (\Exception $e) {
            $response = [
                'success' => false,
                'msg' => 'Search failed'
            ];
            return $this->response->setStatusCode(500)->setJSON($response);
        }
    }

    // Bulk tag operations
    public function bulkAction()
    {
        if ($this->request->getMethod() == 'POST') {
            $action = $this->request->getVar('action');
            $tagIds = $this->request->getVar('tag_ids');

            if (empty($tagIds) || !is_array($tagIds)) {
                if ($this->request->isAJAX()) {
                    $response = ['success' => false, 'msg' => 'No tags selected'];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', 'No tags selected', 3);
                    return redirect()->to('administrator/posts/tags');
                }
            }

            $successCount = 0;
            $failedCount = 0;
            $postTagsModel = new \Modules\Posts\Models\PostTagsModel();

            foreach ($tagIds as $tagId) {
                $tag = $this->tagsModel->find($tagId);
                if (!$tag) {
                    $failedCount++;
                    continue;
                }

                switch ($action) {
                    case 'delete':
                        // Check if tag is in use
                        $usageCount = $postTagsModel->where('tag_id', $tagId)->countAllResults();
                        
                        if ($usageCount > 0) {
                            $failedCount++;
                        } else {
                            if ($this->tagsModel->delete($tagId)) {
                                $successCount++;
                            } else {
                                $failedCount++;
                            }
                        }
                        break;
                    
                    default:
                        $failedCount++;
                        break;
                }
            }

            if ($successCount > 0) {
                // Send notification for bulk action
                $adminmessage = 'Bulk Tag Action: ' . $action . ' performed on ' . $successCount . ' tags';
                $notificationType = 'content';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => true,
                        'msg' => $successCount . ' tags processed successfully' . 
                                 ($failedCount > 0 ? ', ' . $failedCount . ' failed' : ''),
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('success', $successCount . ' tags processed successfully', 3);
                }
            }

            if ($failedCount > 0 && $successCount == 0) {
                if ($this->request->isAJAX()) {
                    $response = [
                        'success' => false,
                        'msg' => $failedCount . ' tags failed to process',
                    ];
                    return $this->response->setJSON($response);
                } else {
                    session()->setTempdata('error', $failedCount . ' tags failed to process', 3);
                }
            }
        }

        return redirect()->to('administrator/posts/tags');
    }
}