<?php

namespace Modules\Posts\Controllers;

use App\Controllers\Auth\AuthController;
use Modules\Posts\Models\NavContextsModel;
use Modules\Posts\Models\ServiceGroupsModel;

class ContextManagerController extends AuthController
{
    public function index()
    {
        $navContextsModel = new NavContextsModel();
        $serviceGroupsModel = new ServiceGroupsModel();
        
        // Use AuthController's $this->data for notifications and user info
        $this->data['nav_contexts'] = $navContextsModel->getAllRaw();
        $this->data['service_groups'] = $serviceGroupsModel->getAll();
        $this->data['title'] = 'Context Manager';
        $this->data['bcrumb'] = 'Context Manager';
        
        return view('Modules\\Posts\\Views\\context_manager\\index', $this->data);
    }

    /** Nav Contexts CRUD */
    public function createNavContext()
    {
        if ($this->request->getMethod() != 'POST') {
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $name = trim($this->request->getVar('name'));
        
        if (empty($name)) {
            session()->setTempdata('error', 'Name is required', 3);
            return redirect()->back();
        }
        
        $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($name)));
        $navContextsModel = new NavContextsModel();
        
        // Check if slug already exists
        $existing = $navContextsModel->where('slug', $slug)->first();
        if ($existing) {
            session()->setTempdata('error', 'A nav context with similar name already exists', 3);
            return redirect()->back();
        }
        
        $data = [
            'name' => $name,
            'slug' => $slug,
            'description' => $this->request->getVar('description'),
            'is_active' => 1,
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $this->userdata->user_id
        ];
        
        if ($navContextsModel->save($data)) {
            // Send notification for nav context creation
            $adminmessage = 'New Navigation Context Created: ' . $name;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Navigation Context added successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to add Navigation Context', 3);
        }
        
        return redirect()->to('administrator/posts/context-manager');
    }

    public function editNavContext($id = null)
    {
        if (empty($id)) {
            session()->setTempdata('error', 'Invalid request', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $navContextsModel = new NavContextsModel();
        $navContext = $navContextsModel->where('id', $id)->first();
        
        if (!$navContext) {
            session()->setTempdata('error', 'Navigation Context not found', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $this->data['singleNavContext'] = $navContext;
        
        // Load index with singleNavContext to show edit form
        return $this->index();
    }

    public function updateNavContext()
    {
        if ($this->request->getMethod() != 'POST') {
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $id = $this->request->getVar('id');
        $name = trim($this->request->getVar('name'));
        
        if (empty($name) || empty($id)) {
            session()->setTempdata('error', 'Invalid data provided', 3);
            return redirect()->back();
        }
        
        $navContextsModel = new NavContextsModel();
        $existing = $navContextsModel->find($id);
        
        if (!$existing) {
            session()->setTempdata('error', 'Navigation Context not found', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($name)));
        
        // Check if new slug conflicts with another record
        $conflict = $navContextsModel->where('slug', $slug)
                                     ->where('id !=', $id)
                                     ->first();
        if ($conflict) {
            session()->setTempdata('error', 'A nav context with similar name already exists', 3);
            return redirect()->back();
        }
        
        $data = [
            'name' => $name,
            'slug' => $slug,
            'description' => $this->request->getVar('description'),
            'updated_at' => date('Y-m-d H:i:s'),
            'updated_by' => $this->userdata->user_id
        ];
        
        if ($navContextsModel->update($id, $data)) {
            // Send notification for nav context update
            $adminmessage = 'Navigation Context Updated: ' . $name;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Navigation Context updated successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to update Navigation Context', 3);
        }
        
        return redirect()->to('administrator/posts/context-manager');
    }

    public function deleteNavContext($id)
    {
        if (empty($id)) {
            session()->setTempdata('error', 'Invalid request', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $navContextsModel = new NavContextsModel();
        $navContext = $navContextsModel->find($id);
        
        if (!$navContext) {
            session()->setTempdata('error', 'Navigation Context not found', 3);
            return redirect()->to('administrator/posts/context-manager');
        }

        $contextName = $navContext['name'] ?? 'Unknown Context';
        
        // Check if nav context is being used by service groups
        $db = \Config\Database::connect();
        $usageCount = $db->table('usc_service_group_nav_contexts')
                        ->where('nav_context_id', $id)
                        ->countAllResults();
        
        if ($usageCount > 0) {
            session()->setTempdata('error', 'Cannot delete. This nav context is being used by ' . $usageCount . ' service group(s)', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        if ($navContextsModel->delete($id)) {
            // Send notification for nav context deletion
            $adminmessage = 'Navigation Context Deleted: ' . $contextName;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Navigation Context deleted successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to delete Navigation Context', 3);
        }
        
        return redirect()->to('administrator/posts/context-manager');
    }

    /** Service Groups CRUD */
    public function createServiceGroup()
    {
        if ($this->request->getMethod() != 'POST') {
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $name = trim($this->request->getVar('name'));
        
        if (empty($name)) {
            session()->setTempdata('error', 'Name is required', 3);
            return redirect()->back();
        }
        
        $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($name)));
        $serviceGroupsModel = new ServiceGroupsModel();
        
        // Check if slug already exists
        $existing = $serviceGroupsModel->where('slug', $slug)->first();
        if ($existing) {
            session()->setTempdata('error', 'A service group with similar name already exists', 3);
            return redirect()->back();
        }
        
        $data = [
            'name' => $name,
            'slug' => $slug,
            'description' => $this->request->getVar('description'),
            'is_active' => 1,
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $this->userdata->user_id,
        ];
        
        // Handle file upload
        $file = $this->request->getFile('preview_image');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            $newName = $file->getRandomName();
            $uploadPath = FCPATH . 'media/service-groups/';
            
            // Create directory if it doesn't exist
            if (!is_dir($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }
            
            if ($file->move($uploadPath, $newName)) {
                $data['preview_image'] = $newName;
            }
        }
        
        if ($serviceGroupsModel->insert($data)) {
            $newId = $serviceGroupsModel->getInsertID();
            $selected = $this->request->getVar('nav_contexts');
            $serviceGroupsModel->setNavContexts($newId, is_array($selected) ? $selected : []);
            
            // Send notification for service group creation
            $adminmessage = 'New Service Group Created: ' . $name;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Service Group added successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to add Service Group', 3);
        }
        
        return redirect()->to('administrator/posts/context-manager');
    }

    public function editServiceGroup($id = null)
    {
        if (empty($id)) {
            session()->setTempdata('error', 'Invalid request', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $serviceGroupsModel = new ServiceGroupsModel();
        $serviceGroup = $serviceGroupsModel->where('id', $id)->first();
        
        if (!$serviceGroup) {
            session()->setTempdata('error', 'Service Group not found', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $this->data['singleServiceGroup'] = $serviceGroup;
        
        // Load nav context selections
        $db = \Config\Database::connect();
        $rows = $db->table('usc_service_group_nav_contexts')
                   ->where('service_group_id', $id)
                   ->get()
                   ->getResultArray();
        $this->data['singleServiceGroupNavs'] = array_column($rows, 'nav_context_id');
        
        return $this->index();
    }

    public function updateServiceGroup()
    {
        if ($this->request->getMethod() != 'POST') {
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $id = $this->request->getVar('id');
        $name = trim($this->request->getVar('name'));
        
        if (empty($name) || empty($id)) {
            session()->setTempdata('error', 'Invalid data provided', 3);
            return redirect()->back();
        }
        
        $serviceGroupsModel = new ServiceGroupsModel();
        $existing = $serviceGroupsModel->find($id);
        
        if (!$existing) {
            session()->setTempdata('error', 'Service Group not found', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($name)));
        
        // Check if new slug conflicts with another record
        $conflict = $serviceGroupsModel->where('slug', $slug)
                                       ->where('id !=', $id)
                                       ->first();
        if ($conflict) {
            session()->setTempdata('error', 'A service group with similar name already exists', 3);
            return redirect()->back();
        }
        
        $data = [
            'name' => $name,
            'slug' => $slug,
            'description' => $this->request->getVar('description'),
            'updated_at' => date('Y-m-d H:i:s'),
            'updated_by' => $this->userdata->user_id
        ];
        
        // Handle file upload
        $file = $this->request->getFile('preview_image');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            $newName = $file->getRandomName();
            $uploadPath = FCPATH . 'media/service-groups/';
            
            // Create directory if it doesn't exist
            if (!is_dir($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }
            
            if ($file->move($uploadPath, $newName)) {
                // Delete old image if exists
                if (!empty($existing['preview_image'])) {
                    $oldFile = $uploadPath . $existing['preview_image'];
                    if (file_exists($oldFile)) {
                        @unlink($oldFile);
                    }
                }
                $data['preview_image'] = $newName;
            }
        }
        
        if ($serviceGroupsModel->update($id, $data)) {
            $selected = $this->request->getVar('nav_contexts');
            $serviceGroupsModel->setNavContexts($id, is_array($selected) ? $selected : []);
            
            // Send notification for service group update
            $adminmessage = 'Service Group Updated: ' . $name;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Service Group updated successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to update Service Group', 3);
        }
        
        return redirect()->to('administrator/posts/context-manager');
    }

    public function deleteServiceGroup($id)
    {
        if (empty($id)) {
            session()->setTempdata('error', 'Invalid request', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        $serviceGroupsModel = new ServiceGroupsModel();
        $serviceGroup = $serviceGroupsModel->find($id);
        
        if (!$serviceGroup) {
            session()->setTempdata('error', 'Service Group not found', 3);
            return redirect()->to('administrator/posts/context-manager');
        }

        $groupName = $serviceGroup['name'] ?? 'Unknown Service Group';
        
        // Check if service group is being used by posts
        $db = \Config\Database::connect();
        $usageCount = $db->table('usc_posts')
                        ->where('service_group', $id)
                        ->orWhere('service_group', $serviceGroup['slug'])
                        ->countAllResults();
        
        if ($usageCount > 0) {
            session()->setTempdata('error', 'Cannot delete. This service group is being used by ' . $usageCount . ' post(s)', 3);
            return redirect()->to('administrator/posts/context-manager');
        }
        
        if ($serviceGroupsModel->delete($id)) {
            // Delete associated image if exists
            if (!empty($serviceGroup['preview_image'])) {
                $imagePath = FCPATH . 'media/service-groups/' . $serviceGroup['preview_image'];
                if (file_exists($imagePath)) {
                    @unlink($imagePath);
                }
            }
            
            // Send notification for service group deletion
            $adminmessage = 'Service Group Deleted: ' . $groupName;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Service Group deleted successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to delete Service Group', 3);
        }
        
        return redirect()->to('administrator/posts/context-manager');
    }

    /** AJAX: groups for nav context */
    public function groupsByNavContext($slug = null)
    {
        $out = [];
        if (empty($slug)) {
            return $this->response->setJSON($out);
        }
        try {
            $serviceGroupsModel = new ServiceGroupsModel();
            $groups = $serviceGroupsModel->getByNavContext($slug);
            return $this->response->setJSON($groups);
        } catch (\Exception $e) {
            return $this->response->setStatusCode(500)->setJSON(['error' => $e->getMessage()]);
        }
    }
}