<?php

namespace Modules\Posts\Controllers;

use App\Controllers\Auth\AuthController;
use Modules\Posts\Models\BlockDefinitionsModel;
use Modules\Posts\Models\PostBlocksModel;
use Modules\Posts\Models\PostBlockItemsModel;
use Modules\Posts\Models\PostsModel;
use Modules\Posts\Models\ServiceGroupsModel;
use Modules\Posts\Models\ItemTypeDefinitionsModel;
use Modules\Posts\Models\BlockItemTypeRelationshipsModel;


class BlocksController extends AuthController
{
      public function initController($request, $response, $logger)
  {
    parent::initController($request, $response, $logger);
    $this->db = \Config\Database::connect();
    $this->blockdefinitionsModel = new BlockDefinitionsModel();
    $this->postBlocksModel = new PostBlocksModel();
    $this->postBlockItemsModel = new PostBlockItemsModel();
    $this->postsModel = new PostsModel();
    $this->servicegroupsModel = new ServiceGroupsModel();
    $this->itemtypedefinitionsModel = new ItemTypeDefinitionsModel();
    $this->blockitemtyperelationshipsModel = new BlockItemTypeRelationshipsModel();
   
  }

    /**
     * Display blocks management page
     */
    public function index()
    {
        $this->data['title'] = "Blocks Management";
        $this->data['bcrumb'] = "Blocks";
        
        // Handle active tab from query parameter
        $activeTab = $this->request->getGet('tab') ?? 'definitions';
        $this->data['activeTab'] = $activeTab;

        // Get all block definitions
        $this->data['blockDefinitions'] = $this->blockdefinitionsModel->findAll();
        
        // Get all post blocks with post info and items count
        $builder = $this->db->table('usc_post_blocks as pb');
        $builder->select('pb.*, p.title as post_title, COUNT(pbi.id) as items_count');
        $builder->join('usc_posts as p', 'p.id = pb.post_id', 'left');
        $builder->join('usc_post_block_items as pbi', 'pbi.block_id = pb.id', 'left');
        $builder->where('pb.deleted_at', null);
        $builder->groupBy('pb.id');
        $builder->orderBy('pb.post_id', 'DESC');
        $this->data['postBlocks'] = $builder->get()->getResult();
        
        // Get all block items
        $this->data['blockItems'] = $this->postBlockItemsModel
                                         ->where('deleted_at', null)
                                         ->orderBy('block_id', 'DESC')
                                         ->findAll();
        
        // Get all pages and services for filter
        $this->data['allPosts'] = $this->postsModel
                                      ->where('is_block', 'Yes')
                                      ->orWhere('content_type_id', 1) // Include pages
                                      ->orWhere('content_type_id', 3) // Include services
                                      ->findAll();

        // Get service groups for block definition categories
        $this->data['serviceGroups'] = $this->servicegroupsModel
                                           ->where('deleted_at', null)
                                           ->findAll();

        // Get all available item types
        $this->data['availableItemTypes'] = $this->itemtypedefinitionsModel->getActiveItemTypes();
        
        // Get all item type definitions for the table
        $this->data['itemTypeDefinitions'] = $this->itemtypedefinitionsModel->findAll();

        return view('Modules\Posts\Views\blocks\index', $this->data);
    }

    /**
     * Move an UploadedFile instance to media/files and return stored filename
     */
    private function moveFileToMedia($file)
    {
        // Normalize settings field for storage
        if (isset($data['settings'])) {
            if (is_array($data['settings'])) {
                $data['settings'] = json_encode($data['settings']);
            }
        }

        try {
            $mediaPath = ROOTPATH . 'media' . DIRECTORY_SEPARATOR . 'files';
            if (!is_dir($mediaPath)) {
                mkdir($mediaPath, 0755, true);
            }

            $newName = time() . '_' . $file->getRandomName();
            $file->move($mediaPath, $newName);
            return $newName;
        } catch (\Exception $e) {
            // log error and return null
            log_message('error', 'File move error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Convenience to handle a single file upload field name and return filename
     */
    private function handleUpload($fieldName)
    {
        try {
            $file = $this->request->getFile($fieldName);
            if ($file && $file->isValid() && !$file->hasMoved()) {
                return $this->moveFileToMedia($file);
            }
        } catch (\Exception $e) {
            log_message('error', 'Upload handling error for ' . $fieldName . ': ' . $e->getMessage());
        }
        return null;
    }

    /**
     * Validate an UploadedFile against size and mime rules
     * Returns array ['ok' => bool, 'message' => '...']
     */
    private function validateFileUpload($file)
    {
        try {
            // Max size 2MB
            $maxBytes = 2 * 1024 * 1024;
            $allowed = [
                'image/jpeg', 'image/png', 'image/gif', 'image/webp'
            ];

            if (!$file->isValid()) {
                return ['ok' => false, 'message' => 'Invalid file upload'];
            }

            if ($file->getSize() > $maxBytes) {
                return ['ok' => false, 'message' => 'File too large. Max 2MB allowed'];
            }

            $mime = $file->getMimeType();
            if ($mime && !in_array($mime, $allowed)) {
                return ['ok' => false, 'message' => 'Unsupported file type'];
            }

            return ['ok' => true, 'message' => 'OK'];
        } catch (\Exception $e) {
            log_message('error', 'File validation error: ' . $e->getMessage());
            return ['ok' => false, 'message' => 'File validation error'];
        }
    }

    /**
     * Get active block definitions
     */
    public function getActiveDefinitions()
    {
        $definitions = $this->blockdefinitionsModel->getActiveBlocks();
        return $this->respond(['success' => true, 'data' => $definitions]);
    }

    /**
     * Get single block definition
     */
    public function getBlockDefinition($id)
    {
        $definition = $this->blockdefinitionsModel->getWithItemTypes($id);
        if (!$definition) {
            return $this->respond(['success' => false, 'message' => 'Definition not found'], 404);
        }

        return $this->respond(['success' => true, 'data' => $definition]);
    }

    /**
     * Add new block definition
     */
    public function addBlockDefinition()
    {
        $rules = [
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[block_definitions.slug]'],
        ];

        if (!$this->validate($rules)) {
            if ($this->request->isAJAX()) {
                return $this->respond([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $this->validator->getErrors()
                ], 400);
            } else {
                session()->setTempdata('error', 'Validation failed', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        $data = $this->request->getPost();
        $itemTypes = $data['item_types'] ?? [];
        unset($data['item_types']);
        $data['created_by'] = $this->userdata->user_id;

        try {
            if ($this->blockdefinitionsModel->save($data)) {
                $blockDefinitionId = $this->blockdefinitionsModel->getInsertID();
                foreach ($itemTypes as $itemTypeId) {
                    $this->blockitemtyperelationshipsModel->insert([
                        'block_definition_id' => $blockDefinitionId,
                        'item_type_definition_id' => $itemTypeId,
                        'is_default' => 0,
                        'max_items' => 0,
                        'ordering' => 0
                    ]);
                }

                // Send admin notification for new block definition
                $adminmessage = 'Block definition added: ' . ($data['name'] ?? 'Unnamed');
                $notificationType = 'content';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                if ($this->request->isAJAX()) {
                    return $this->respond(['success' => true, 'message' => 'Block definition added successfully']);
                } else {
                    session()->setTempdata('success', 'Block definition added successfully', 3);
                    return redirect()->to('administrator/posts/blocks');
                }
            }

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => false, 'message' => 'Failed to add block definition'], 500);
            } else {
                session()->setTempdata('error', 'Failed to add block definition', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        } catch (\Exception $e) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
            } else {
                session()->setTempdata('error', 'Error adding block definition', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }
    }

    /**
     * Update block definition
     */
    public function updateBlockDefinition()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required'],
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[block_definitions.slug,id,{id}]'],
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        $blockDefinitionId = $data['id'];
        $itemTypes = $data['item_types'] ?? [];
        unset($data['item_types']);
        
        if ($this->blockdefinitionsModel->save($data)) {
            // Delete existing relationships
            $this->blockitemtyperelationshipsModel->where('block_definition_id', $blockDefinitionId)->delete();

            // Insert new relationships
            foreach ($itemTypes as $itemTypeId) {
                $this->blockitemtyperelationshipsModel->insert([
                    'block_definition_id' => $blockDefinitionId,
                    'item_type_definition_id' => $itemTypeId,
                    'is_default' => 0, // Default value, adjust as needed
                    'max_items' => 0, // Default value, adjust as needed
                    'ordering' => 0 // Default value, adjust as needed
                ]);
            }
            // Send admin notification for update
            $adminmessage = 'Block definition updated: ' . ($data['name'] ?? ('ID ' . $blockDefinitionId));
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            return $this->respond(['success' => true, 'message' => 'Block definition updated successfully']);
        }

        return $this->respond(['success' => false, 'message' => 'Failed to update block definition'], 500);
    }

    /**
     * Delete block definition
     */
    public function deleteBlockDefinition($id)
    {
        if ($this->userdata->group_id != 1) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => false, 'message' => 'Access denied'], 403);
            } else {
                session()->setFlashdata('warning', 'Access denied');
                return redirect()->to(previous_url());
            }
        }

        // Check if any blocks are using this definition
        $definition = $this->blockdefinitionsModel->find($id);
        if ($definition) {
            $blocksUsingDefinition = $this->postBlocksModel
                                        ->where('block_type', $definition['slug'])
                                        ->countAllResults();
            
            if ($blocksUsingDefinition > 0) {
                if ($this->request->isAJAX()) {
                    return $this->respond([
                        'success' => false, 
                        'message' => 'Cannot delete definition. There are ' . $blocksUsingDefinition . ' blocks using this definition.'
                    ], 400);
                } else {
                    session()->setTempdata('error', 'Cannot delete definition. There are blocks using this definition.', 3);
                    return redirect()->to('administrator/posts/blocks');
                }
            }
        }

        if ($this->blockdefinitionsModel->delete($id)) {
            // send admin notification
            $defName = $definition['name'] ?? ('ID ' . $id);
            $adminmessage = 'Block definition deleted: ' . $defName;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Block definition deleted']);
            } else {
                session()->setTempdata('success', 'Block definition deleted', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to delete block definition'], 500);
        } else {
            session()->setTempdata('error', 'Failed to delete block definition', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Add new block
     */
    public function addBlock()
    {
        $rules = [
            'post_id' => ['label' => 'Post', 'rules' => 'required|is_not_unique[usc_posts.id]'],
            'definition_id' => ['label' => 'Definition', 'rules' => 'required|is_not_unique[block_definitions.id]'],
            'heading' => ['label' => 'Heading', 'rules' => 'permit_empty|max_length[191]'],
            'content' => ['label' => 'Content', 'rules' => 'permit_empty'],
            'heading_type' => ['label' => 'Heading Type', 'rules' => 'permit_empty|in_list[h1,h2,h3,h4,h5,h6]'],
            'list_type' => ['label' => 'List Type', 'rules' => 'permit_empty|in_list[none,ordered,unordered]'],
            'status' => ['label' => 'Status', 'rules' => 'required|in_list[active,inactive]'],
            'title_emphasized' => ['label' => 'Title Emphasized', 'rules' => 'permit_empty'],
            'subtitle' => ['label' => 'Subtitle', 'rules' => 'permit_empty|max_length[255]'],
            'settings' => ['label' => 'Settings', 'rules' => 'permit_empty'],
            'custom_css' => ['label' => 'Custom CSS', 'rules' => 'permit_empty'],
            'custom_classes' => ['label' => 'Custom Classes', 'rules' => 'permit_empty|max_length[255]']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();

        // Validate file upload if present
        $file = $this->request->getFile('photo');
        if ($file && $file->getError() !== 4) { // 4 = no file uploaded
            $fileValidation = $this->validateFileUpload($file);
            if (!$fileValidation['ok']) {
                return $this->respond(['success' => false, 'message' => $fileValidation['message']], 400);
            }
        }

        // Handle optional uploaded image for block (field name: 'photo')
        $uploaded = $this->handleUpload('photo');
        if ($uploaded) {
            $data['photo'] = $uploaded;
        }

        $result = $this->postBlocksModel->addBlock($data);

        if (!empty($result['success'])) {
            // send admin notification about new block
            if (!empty($result['blockId'])) {
                $block = $this->postBlocksModel->find($result['blockId']);
                $postTitle = null;
                if (!empty($block['post_id'])) {
                    $post = $this->postsModel->find($block['post_id']);
                    $postTitle = $post['title'] ?? null;
                }
                $adminmessage = 'Block added: ' . ($block['block_type'] ?? 'unknown') . ' on ' . ($postTitle ?? ('post ID ' . ($block['post_id'] ?? 'N/A')));
                $notificationType = 'content';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);
            }

            return $this->respond(['success' => true, 'message' => $result['message'], 'blockId' => $result['blockId'] ?? null]);
        }

        $code = $result['code'] ?? 500;
        return $this->respond(['success' => false, 'message' => $result['message']], $code);
    }

    /**
     * Get single block
     */
    public function getBlock($id)
    {
        $block = $this->postBlocksModel->find($id);
        if (!$block) {
            return $this->respond(['success' => false, 'message' => 'Block not found'], 404);
        }
        return $this->respond(['success' => true, 'data' => $block]);
    }

    /**
     * View block details
     */
    public function viewBlock($id)
    {
        // Get block with extended info
        $builder = $this->db->table('usc_post_blocks as pb');
        $builder->select('pb.*, p.title as post_title, bd.name as definition_name, bd.description as definition_description');
        $builder->join('usc_posts as p', 'p.id = pb.post_id', 'left');
        $builder->join('block_definitions as bd', 'bd.slug = pb.block_type', 'left');
        $builder->where('pb.id', $id);
        $block = $builder->get()->getRow();

        if (!$block) {
            return $this->respond(['success' => false, 'message' => 'Block not found'], 404);
        }

        // Get block items
        $items = $this->postBlockItemsModel
            ->where('block_id', $id)
            ->where('deleted_at', null)
            ->orderBy('ordering', 'ASC')
            ->findAll();

        return $this->respond([
            'success' => true,
            'data' => [
                'block' => $block,
                'items' => $items
            ]
        ]);
    }

    
    /**
     * Update block
     */
    public function updateBlock()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_blocks.id]']
        ];

        // Accept and validate additional fields
        $rules['title_emphasized'] = ['label' => 'Title Emphasized', 'rules' => 'permit_empty'];
        $rules['subtitle'] = ['label' => 'Subtitle', 'rules' => 'permit_empty|max_length[255]'];
        $rules['settings'] = ['label' => 'Settings', 'rules' => 'permit_empty'];
        $rules['custom_css'] = ['label' => 'Custom CSS', 'rules' => 'permit_empty'];
        $rules['custom_classes'] = ['label' => 'Custom Classes', 'rules' => 'permit_empty|max_length[255]'];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();

        // First check if block exists (needed for cleanup of old files)
        $existingBlock = $this->postBlocksModel->find($data['id']);
        if (!$existingBlock) {
            return $this->respond([
                'success' => false,
                'message' => 'Block not found'
            ], 404);
        }

        // Validate file upload if present
        $file = $this->request->getFile('photo');
        if ($file && $file->getError() !== 4) {
            $fileValidation = $this->validateFileUpload($file);
            if (!$fileValidation['ok']) {
                return $this->respond(['success' => false, 'message' => $fileValidation['message']], 400);
            }
        }

        // Handle optional uploaded image for block
        $uploaded = $this->handleUpload('photo');
        if ($uploaded) {
            // Remove previous photo file if exists
            if (!empty($existingBlock['photo'])) {
                $oldPath = ROOTPATH . 'media' . DIRECTORY_SEPARATOR . 'files' . DIRECTORY_SEPARATOR . $existingBlock['photo'];
                if (is_file($oldPath)) {
                    @unlink($oldPath);
                }
            }
            $data['photo'] = $uploaded;
        }

        try {
            if ($this->postBlocksModel->save($data)) {
                // send admin notification
                $block = $this->postBlocksModel->find($data['id']);
                $postTitle = null;
                if (!empty($block['post_id'])) {
                    $post = $this->postsModel->find($block['post_id']);
                    $postTitle = $post['title'] ?? null;
                }
                $adminmessage = 'Block updated: ' . ($block['block_type'] ?? 'ID ' . $data['id']) . ' on ' . ($postTitle ?? ('post ID ' . ($block['post_id'] ?? 'N/A')));
                $notificationType = 'content';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                return $this->respond([
                    'success' => true, 
                    'message' => 'Block updated successfully'
                ]);
            }

            return $this->respond([
                'success' => false,
                'message' => 'Failed to update block: ' . implode(', ', $this->postBlocksModel->errors())
            ], 500);
        } catch (\Exception $e) {
            return $this->respond([
                'success' => false,
                'message' => 'Error updating block: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update block order
     */
    public function updateBlockOrder()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_blocks.id]'],
            'ordering' => ['label' => 'Order', 'rules' => 'required|numeric']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        try {
            $this->db->transBegin();
            
            if ($this->postBlocksModel->updateWithOrdering($data['id'], $data)) {
                $this->db->transCommit();

                // send admin notification for ordering change
                $block = $this->postBlocksModel->find($data['id']);
                $adminmessage = 'Block order updated: ' . ($block['block_type'] ?? 'ID ' . $data['id']) . ' new order ' . ($data['ordering'] ?? 'N/A');
                $notificationType = 'content';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                return $this->respond(['success' => true, 'message' => 'Block order updated successfully']);
            }

            $this->db->transRollback();
            return $this->respond(['success' => false, 'message' => 'Failed to update block order']);
        } catch (\Exception $e) {
            $this->db->transRollback();
            return $this->respond(['success' => false, 'message' => 'Error updating block order']);
        }
    }

    /**
     * Delete block
     */
    public function deleteBlock($id)
    {
    // fetch block info before delete for notification
    $block = $this->postBlocksModel->find($id);
    $deleted = $this->postBlocksModel->deleteBlock($id);

        if ($deleted) {
            // send admin notification
            $postTitle = null;
            if (!empty($block['post_id'])) {
                $post = $this->postsModel->find($block['post_id']);
                $postTitle = $post['title'] ?? null;
            }
            $adminmessage = 'Block deleted: ' . ($block['block_type'] ?? 'ID ' . $id) . ' on ' . ($postTitle ?? ('post ID ' . ($block['post_id'] ?? 'N/A')));
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Block deleted']);
            } else {
                session()->setTempdata('success', 'Block deleted', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to delete block'], 500);
        } else {
            session()->setTempdata('error', 'Failed to delete block', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Get block items for a specific block
     */
    public function getBlockItems($blockId)
    {
        $html = $this->postBlockItemsModel->getHtmlForBlock($blockId);

        if (empty($html)) {
            return $this->respond(['success' => false, 'message' => 'No items found for this block'], 404);
        }

        return $this->respond(['success' => true, 'html' => $html]);
    }

    /**
     * Get single block item
     */
    public function getBlockItem($id)
    {
        $item = $this->postBlockItemsModel->find($id);
        if (!$item) {
            return $this->respond(['success' => false, 'message' => 'Item not found'], 404);
        }
        return $this->respond(['success' => true, 'data' => $item]);
    }

    /**
     * Add new block item
     */
    public function addBlockItem()
    {
        $rules = [
            'block_id' => ['label' => 'Block', 'rules' => 'required|is_not_unique[usc_post_blocks.id]'],
            'title' => ['label' => 'Title', 'rules' => 'required']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();

        // Validate file upload for item if present
        $file = $this->request->getFile('photo');
        if ($file && $file->getError() !== 4) {
            $fileValidation = $this->validateFileUpload($file);
            if (!$fileValidation['ok']) {
                return $this->respond(['success' => false, 'message' => $fileValidation['message']], 400);
            }
        }

        // Handle optional uploaded image for item (field name: 'photo')
        $uploaded = $this->handleUpload('photo');
        if ($uploaded) {
            $data['photo'] = $uploaded;
        }

        // If dynamic file fields were used, move them and replace in fields
        if ($this->request->getFiles()) {
            $files = $this->request->getFiles();
            foreach ($files as $key => $file) {
                // handle files named fields_files[...] by storing filename in $data['value'] as JSON
                if (strpos($key, 'fields_files') !== false) {
                    // $file may be an array or UploadedFile; handle accordingly
                    if (is_array($file)) {
                        foreach ($file as $f) {
                            if ($f->isValid() && !$f->hasMoved()) {
                                $fname = $this->moveFileToMedia($f);
                                // store under fields in posted data
                                $fieldName = str_replace(['fields_files[', ']'], '', $key);
                                $postedFields = $this->request->getPost('fields') ?? [];
                                $postedFields[$fieldName] = $fname;
                                $data['fields'] = $postedFields;
                            }
                        }
                    } else {
                        if ($file->isValid() && !$file->hasMoved()) {
                            $fname = $this->moveFileToMedia($file);
                            $fieldName = str_replace(['fields_files[', ']'], '', $key);
                            $postedFields = $this->request->getPost('fields') ?? [];
                            $postedFields[$fieldName] = $fname;
                            $data['fields'] = $postedFields;
                        }
                    }
                }
            }
        }

        $result = $this->postBlockItemsModel->addItem($data);

        if (!empty($result['success'])) {
            // send admin notification about new item
            $block = $this->postBlocksModel->find($data['block_id'] ?? null);
            $postTitle = null;
            if (!empty($block['post_id'])) {
                $post = $this->postsModel->find($block['post_id']);
                $postTitle = $post['title'] ?? null;
            }
            $adminmessage = 'Block item added: ' . ($data['title'] ?? 'Untitled') . ' in block ' . ($block['block_type'] ?? ('ID ' . ($block['id'] ?? 'N/A'))) . ' on ' . ($postTitle ?? ('post ID ' . ($block['post_id'] ?? 'N/A')));
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            return $this->respond(['success' => true, 'message' => $result['message']]);
        }

        $code = $result['code'] ?? 500;
        return $this->respond(['success' => false, 'message' => $result['message']], $code);
    }

    /**
     * Update block item
     */
    public function updateBlockItem()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_block_items.id]'],
            'block_id' => ['label' => 'Block', 'rules' => 'required|is_not_unique[usc_post_blocks.id]'],
            'title' => ['label' => 'Title', 'rules' => 'required']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();

        // Validate file upload for item if present
        $file = $this->request->getFile('photo');
        if ($file && $file->getError() !== 4) {
            $fileValidation = $this->validateFileUpload($file);
            if (!$fileValidation['ok']) {
                return $this->respond(['success' => false, 'message' => $fileValidation['message']], 400);
            }
        }

        // Handle optional uploaded image for item and cleanup old file
        $uploaded = $this->handleUpload('photo');
        if ($uploaded) {
            // remove previous photo if exists
            $existingItem = $this->postBlockItemsModel->find($data['id']);
            if (!empty($existingItem['photo'])) {
                $oldPath = ROOTPATH . 'media' . DIRECTORY_SEPARATOR . 'files' . DIRECTORY_SEPARATOR . $existingItem['photo'];
                if (is_file($oldPath)) {
                    @unlink($oldPath);
                }
            }
            $data['photo'] = $uploaded;
        }

        if ($this->postBlockItemsModel->save($data)) {
            // send admin notification
            $item = $this->postBlockItemsModel->find($data['id']);
            $block = $this->postBlocksModel->find($item['block_id'] ?? null);
            $postTitle = null;
            if (!empty($block['post_id'])) {
                $post = $this->postsModel->find($block['post_id']);
                $postTitle = $post['title'] ?? null;
            }
            $adminmessage = 'Block item updated: ' . ($data['title'] ?? ($item['title'] ?? 'Untitled')) . ' in block ' . ($block['block_type'] ?? ('ID ' . ($block['id'] ?? 'N/A'))) . ' on ' . ($postTitle ?? ('post ID ' . ($block['post_id'] ?? 'N/A')));
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            return $this->respond(['success' => true, 'message' => 'Item updated successfully']);
        }

        return $this->respond(['success' => false, 'message' => 'Failed to update item'], 500);
    }

    /**
     * Update block item order
     */
    public function updateItemOrder()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_block_items.id]'],
            'ordering' => ['label' => 'Order', 'rules' => 'required|numeric']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        if ($this->postBlockItemsModel->save($data)) {
            // send admin notification
            $item = $this->postBlockItemsModel->find($data['id']);
            $block = $this->postBlocksModel->find($item['block_id'] ?? null);
            $adminmessage = 'Block item order updated: ' . ($item['title'] ?? 'ID ' . $data['id']) . ' new order ' . ($data['ordering'] ?? 'N/A');
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            return $this->respond(['success' => true, 'message' => 'Item order updated successfully']);
        }

        return $this->respond(['success' => false, 'message' => 'Failed to update item order'], 500);
    }

    /**
     * Delete block item
     */
    public function deleteBlockItem($id)
    {
        $item = $this->postBlockItemsModel->find($id);
        if ($this->postBlockItemsModel->delete($id)) {
            // send admin notification
            $block = $this->postBlocksModel->find($item['block_id'] ?? null);
            $postTitle = null;
            if (!empty($block['post_id'])) {
                $post = $this->postsModel->find($block['post_id']);
                $postTitle = $post['title'] ?? null;
            }
            $adminmessage = 'Block item deleted: ' . ($item['title'] ?? 'ID ' . $id) . ' from block ' . ($block['block_type'] ?? ('ID ' . ($block['id'] ?? 'N/A'))) . ' on ' . ($postTitle ?? ('post ID ' . ($block['post_id'] ?? 'N/A')));
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Item deleted']);
            } else {
                session()->setTempdata('success', 'Block item deleted', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to delete item'], 500);
        } else {
            session()->setTempdata('error', 'Failed to delete block item', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Get single item type definition
     */
    public function getItemType($id)
    {
        $itemType = $this->itemtypedefinitionsModel->find($id);
        if (!$itemType) {
            return $this->respond(['success' => false, 'message' => 'Item type not found'], 404);
        }
        return $this->respond(['success' => true, 'data' => $itemType]);
    }

    /**
     * View item type definition details
     */
    public function viewItemType($id)
    {
        $itemType = $this->itemtypedefinitionsModel->find($id);
        if (!$itemType) {
            return $this->response->setJSON(['success' => false, 'message' => 'Item type not found'])->setStatusCode(404);
        }

        // Get usage statistics
        $usageCount = $this->postBlockItemsModel
                          ->where('item_type', $itemType['slug'])
                          ->countAllResults();

        // Get associated block definitions
        $associatedBlocks = $this->blockitemtyperelationshipsModel
                                ->select('bd.name, bd.slug')
                                ->join('block_definitions as bd', 'bd.id = block_item_type_relationships.block_definition_id')
                                ->where('item_type_definition_id', $id)
                                ->findAll();

        $data = [
            'itemType' => $itemType,
            'usageCount' => $usageCount,
            'associatedBlocks' => $associatedBlocks
        ];

        return $this->response->setJSON(['success' => true, 'data' => $data]);
    }

    /**
     * Add new item type
     */
    public function addItemType()
    {
        $rules = [
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[usc_item_type_definitions.slug]']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        $data['created_by'] = $this->userdata->user_id;
        
        if ($this->itemtypedefinitionsModel->save($data)) {
            // send admin notification
            $adminmessage = 'Item type added: ' . ($data['name'] ?? 'Unnamed');
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Item type added successfully']);
            } else {
                session()->setTempdata('success', 'Item type added successfully', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to add item type'], 500);
        } else {
            session()->setTempdata('error', 'Failed to add item type', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Update item type
     */
    public function updateItemType()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required'],
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[usc_item_type_definitions.slug,id,{id}]']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        if ($this->itemtypedefinitionsModel->save($data)) {
            // send admin notification
            $adminmessage = 'Item type updated: ' . ($data['name'] ?? ('ID ' . ($data['id'] ?? 'N/A')));
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Item type updated successfully']);
            } else {
                session()->setTempdata('success', 'Item type updated successfully', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to update item type'], 500);
        } else {
            session()->setTempdata('error', 'Failed to update item type', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Delete item type
     */
    public function deleteItemType($id)
    {
        // First get the item type to get its name (for notification) and then attempt safe delete
        $itemType = $this->itemtypedefinitionsModel->find($id);
        $result = $this->itemtypedefinitionsModel->deleteSafely($id);

        if (!empty($result['success'])) {
            // send admin notification
            $adminName = $itemType['name'] ?? ($itemType['slug'] ?? ('ID ' . $id));
            $adminmessage = 'Item type deleted: ' . $adminName;
            $notificationType = 'content';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => $result['message']]);
            } else {
                session()->setTempdata('success', $result['message'], 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        // failure
        $code = $result['code'] ?? 500;
        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => $result['message']], $code);
        } else {
            session()->setTempdata('error', $result['message'], 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }
}