<?php

namespace Modules\Menu\Controllers;

use App\Controllers\Auth\AuthController;
use Modules\Posts\Models\PostsCategoryModel;
use Modules\Posts\Models\PostsModel;
use Modules\Posts\Models\ContentTypesModel;
use Modules\Menu\Models\MenusCategoryModel;
use Modules\Menu\Models\MenusModel;

class Menus extends AuthController
{
    public $contentTypesModel;
    public $menusModel;
    public $menuscategoryModel;
    public $postsModel;
    public $postscategoryModel;

    public function initController($request, $response, $logger)
    {
        parent::initController($request, $response, $logger);
        $this->contentTypesModel = new ContentTypesModel();
        $this->menusModel = new MenusModel();
        $this->menuscategoryModel = new MenusCategoryModel();
        $this->postsModel = new PostsModel();
        $this->postscategoryModel = new PostsCategoryModel();
    }

    public function index()
    {
        // You might want to add a dashboard view here
        return redirect()->to('administrator/menu/groups');
    }

    /**========================================================================
     * MENU TYPES /ADD/EDIT/DELETE
     ==========================================================================*/
    public function menuTypes()
    {
        // display header information
        $this->data['title'] = "Menu Groups";
        $this->data['bcrumb'] = "Menus";
        $this->data['menu_types'] = $this->menuscategoryModel->getAll();

        return view('Modules\Menu\Views\category', $this->data);
    }

    public function addType()
    {
        $this->data['menu_types'] = $this->menuscategoryModel->getAll();
        
        if($this->request->getMethod() == 'POST'){
            $rules = [
                'name' => ['label'=>'Group Name', 'rules'=>'required'],
            ];

            if($this->validate($rules))
            {
                $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($this->request->getVar('name'))));
                $menuData = [
                    'name'          => $this->request->getVar("name"),
                    'alias'         => $slug,
                    'description'   => $this->request->getVar("description"),
                    'created_by'    => $this->userdata->user_id,
                ];
                
                if($this->menuscategoryModel->save($menuData))
                {
                    // Send notification for menu group creation
                    $adminmessage = 'New Menu Group Created: ' . $this->request->getVar("name");
                    $notificationType = 'menu';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setTempdata('success', 'Menu Group added successfully', 3);
                    return redirect()->to('administrator/menu/groups');
                }else{
                    session()->setTempdata('error', 'Menu Group NOT added', 3);
                    return redirect()->to('administrator/menu/groups');
                }
            }else{
                session()->setTempdata('error', 'Group Name can not be empty', 3);
                return redirect()->to('Menu Group');
            }
        }
        return view('Modules\Menu\Views\category', $this->data);
    }

    public function editType($id = null)
    {
        $this->data['title'] = "Edit Group";
        $this->data['bcrumb'] = "Edit Group Menu";
        $this->data['menu_types'] = $this->menuscategoryModel->getAll();
        $this->data['singleGroup'] = $this->menuscategoryModel->where('id', $id)->first($id);

        return view('Modules\Menu\Views\category', $this->data);
    }

    public function updateType()
    {
        $this->data['menu_types'] = $this->menuscategoryModel->getAll();
        
        if($this->request->getMethod() == 'POST'){
            $rules = [
                'name' => ['label'=>'Group Name', 'rules'=>'required'],
            ];

            if($this->validate($rules))
            {
                $id = $this->request->getVar("id");
                $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($this->request->getVar('name'))));
                $menuData = [
                    'name'          => $this->request->getVar("name"),
                    'alias'         => $slug,
                    'description'   => $this->request->getVar("description"),
                    'created_by'    => $this->userdata->user_id,
                ];
                
                if($this->menuscategoryModel->update($id, $menuData))
                {
                    // Send notification for menu group update
                    $adminmessage = 'Menu Group Updated: ' . $this->request->getVar("name");
                    $notificationType = 'menu';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setTempdata('success', 'Menu Group updated successfully', 3);
                    return redirect()->to('administrator/menu/groups');
                }else{
                    session()->setTempdata('error', 'Menu Group NOT updated', 3);
                    return redirect()->to('administrator/menu/groups');
                }
            }else{
                session()->setTempdata('error', 'Group Name can not be empty', 3);
                return redirect()->to('Menu Group');
            }
        }
        return view('Modules\Menu\Views\category', $this->data);
    }

    public function delType($id)
    {
        $menuGroup = $this->menuscategoryModel->find($id);
        
        if (!$menuGroup) {
            if($this->request->isAJAX()) {
                $response = [
                    'error' => false,
                    'msg' => 'Menu Group not found',	   
                ];
                return $this->response->setJSON($response);
            } else {
                session()->setFlashdata('error', 'Menu Group not found');
                return redirect()->to('administrator/menu/groups');
            }
        }

        $this->menuscategoryModel->where('id', $id)->delete($id);
        
        if($this->request->isAJAX())
        {
            // Send notification for menu group deletion
            $adminmessage = 'Menu Group Deleted: ' . ($menuGroup['name'] ?? 'Unknown Group');
            $notificationType = 'menu';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            $response = [
                'error' => true,
                'msg' =>'Menu Group deleted successfully',	   
            ];
            return $this->response->setJSON($response);
        } else {
            // Send notification for menu group deletion
            $adminmessage = 'Menu Group Deleted: ' . ($menuGroup['name'] ?? 'Unknown Group');
            $notificationType = 'menu';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setFlashdata('success', 'Menu Group deleted successfully');
            return redirect()->to('administrator/menu/groups');
        }
    }

    /**========================================================================
     * MENU ITEMS /ADD/EDIT/DELETE
     ==========================================================================*/
    public function viewMenu($id)
    {
        $menuGroup = $this->menuscategoryModel->getSingleGroup($id);
        $this->data['title'] = $menuGroup->name." Items";
        $this->data['bcrumb'] = $menuGroup->name." Items";
        $this->data['post_categories'] = $this->postscategoryModel->getAll();
        $this->data['singleGroup'] = $menuGroup;
        $this->data['menu_items'] = $this->menusModel->getGroupMenu($id);

        return view('Modules\Menu\Views\all_menus', $this->data);
    }

    public function addMenu()
    {
        $this->data['title'] = "Add Menu Item";
        $this->data['bcrumb'] = "Add Menu Item";
        $this->data['menuGroups'] = $this->menuscategoryModel->getAll();
        $this->data['menus'] = $this->menusModel->getAll();
        
        // Get content types from database
        $this->data['contentTypes'] = $this->contentTypesModel->getActiveTypes();
        
        if($this->request->getMethod() == 'POST') {
            return $this->processMenuSave();
        }
        
        return view('Modules\Menu\Views\add_menu', $this->data);
    }

    public function editMenu($item = null)
    {
        $menuItem = $this->menusModel->singleMenuItem($item);
        $this->data['title'] = "Edit ".$menuItem->title." Item";
        $this->data['bcrumb'] = "Edit ".$menuItem->title;
        $this->data['menuGroups'] = $this->menuscategoryModel->getAll();
        $this->data['menus'] = $this->menusModel->getAll();
        $this->data['menuItem'] = $menuItem;
        
        // Get content types from database
        $this->data['contentTypes'] = $this->contentTypesModel->getActiveTypes();

        if($this->request->getMethod() == 'POST') {
            return $this->processMenuSave($item);
        }
        
        return view('Modules\Menu\Views\edit_menu', $this->data);
    }

    /**
     * Process menu saving with dynamic link generation
     */
    private function processMenuSave($id = null)
    {
        $rules = [
            'title' => ['label'=>'Menu Title', 'rules'=>'required'],
            'category_id' => ['label'=>'Menu Category', 'rules'=>'required'],
            'menu_type' => ['label'=>'Menu Type', 'rules'=>'required'],
        ];

        if($this->validate($rules)) {
            $link = $this->generateMenuLink();
            $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($this->request->getVar('title'))));
            
            // Get parent level to calculate current level
            $parentId = $this->request->getVar("parent_id") ?: null;
            $level = 1;
            if ($parentId) {
                $parentMenu = $this->menusModel->find($parentId);
                $level = $parentMenu ? ($parentMenu->level + 1) : 1;
            }

            $menuData = [
                'title' => $this->request->getVar("title"),
                'alias' => $slug,
                'category_id' => $this->request->getVar("category_id"),
                'menu_type' => $this->request->getVar("menu_type"),
                'parent_id' => $parentId,
                'ordering' => $this->request->getVar("ordering") ?: 0,
                'level' => $level,
                'access' => $this->request->getVar("access") ?: 'Public',
                'status' => (int)($this->request->getVar("status") ?: 1),
                'link' => $link,
                'post_id' => $this->request->getVar("post_id") ?: null,
                'post_cat' => $this->request->getVar("post_category") ?: null,
                'created_by' => $this->userdata->user_id,
            ];
            
            if ($id) {
                // Update existing menu
                if($this->menusModel->update($id, $menuData)) {
                    // Send notification for menu item update
                    $adminmessage = 'Menu Item Updated: ' . $this->request->getVar("title") . ' (' . $this->request->getVar("menu_type") . ')';
                    $notificationType = 'menu';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setTempdata('success', 'Menu Item updated successfully', 3);
                    return redirect()->to('administrator/menu/item/edit/'.$id.'-'.$slug);
                }
            } else {
                // Create new menu
                if($this->menusModel->save($menuData)) {
                    $menuId = $this->menusModel->getInsertID();
                    
                    // Send notification for menu item creation
                    $adminmessage = 'New Menu Item Created: ' . $this->request->getVar("title") . ' (' . $this->request->getVar("menu_type") . ')';
                    $notificationType = 'menu';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setTempdata('success', 'Menu Item added successfully', 3);
                    return redirect()->to('administrator/menu/item/edit/'.$menuId.'-'.$slug);
                }
            }
            
            session()->setTempdata('error', 'Menu Item NOT saved', 3);
        } else {
            session()->setTempdata('error', 'Please check required fields', 3);
        }
        
        return $id ? redirect()->to('administrator/menu/item/edit/'.$id) : redirect()->to('administrator/menu/item/add');
    }

    /**
     * Generate menu link based on content type selection
     */
    private function generateMenuLink()
    {
        $menuType = $this->request->getVar('menu_type');
        $customLink = $this->request->getVar('custom');

        // Handle custom links
        if ($menuType === 'custom' || !empty($customLink)) {
            return $customLink ?: '#';
        }

        // Map legacy slugs to database slugs for consistency
        $slugMapping = [
            'post' => 'blog',
            'page' => 'page',
            'service' => 'service'
        ];

        $mappedType = isset($slugMapping[$menuType]) ? $slugMapping[$menuType] : $menuType;

        // Handle different content types
        switch ($mappedType) {
            case 'page':
                $pageId = $this->request->getVar('page_id');
                if ($pageId) {
                    $page = $this->postsModel->where('id', $pageId)->first();
                    return $page ? ($page['menu_link'] ?? $page['alias'] ?? '#') : '#';
                }
                break;

            case 'blog':
                $postId = $this->request->getVar('post_id');
                if ($postId) {
                    $post = $this->postsModel->where('id', $postId)->first();
                    return $post ? ($post['menu_link'] ?? 'blog/' . ($post['alias'] ?? '#')) : '#';
                }
                break;

            case 'service':
                $serviceId = $this->request->getVar('service_id');
                if ($serviceId) {
                    $service = $this->postsModel->where('id', $serviceId)->first();
                    return $service ? ($service['menu_link'] ?? 'services/' . ($service['alias'] ?? '#')) : '#';
                }
                break;

            case 'category':
                $categoryId = $this->request->getVar('post_category');
                if ($categoryId) {
                    $category = $this->postscategoryModel->where('id', $categoryId)->first();
                    return $category ? ('category/' . ($category['alias'] ?? '#')) : '#';
                }
                break;
        }

        return '#';
    }

    /**
     * AJAX method to get content by type
     */
    public function getContentByType()
    {
        $contentTypeSlug = $this->request->getVar('content_type');
        $content = [];

        if (!$contentTypeSlug) {
            return $this->response->setJSON([]);
        }

        // Use a switch statement to handle content fetching based on the slug.
        // This approach is consistent with other working parts of the application
        // and avoids issues with the ContentTypesModel or slug resolution.
        switch ($contentTypeSlug) {
            case 'page':
                // Fetch all posts with content_type_id = 1 (Pages)
                $content = $this->postsModel->getByContentType(1, false);
                break;

            case 'post':
                // Fetch all posts with content_type_id = 2 (Blog Posts)
                $content = $this->postsModel->getByContentType(2, false);
                break;

            case 'service':
                // Fetch all posts with content_type_id = 3 (Services)
                $content = $this->postsModel->getByContentType(3, false);
                break;

            case 'category':
                // Handle 'category' as a special case, as it's a taxonomy.
                $categories = $this->postscategoryModel->getAll();
                $formattedContent = [];
                foreach ($categories as $cat) {
                    $formattedContent[] = [
                        'id' => $cat->id,
                        'title' => $cat->name,
                        'alias' => $cat->alias,
                        'menu_link' => 'category/' . $cat->alias
                    ];
                }
                $content = $formattedContent;
                break;

            default:
                // If the content type is unknown, return an empty array.
                $content = [];
                break;
        }

        return $this->response->setJSON($content);
    }

    /**
     * AJAX method to get menus by category
     */
    public function getMenusByCategory()
    {
        $categoryId = $this->request->getVar('category_id');
        
        if (!$categoryId) {
            return $this->response->setJSON([]);
        }
        
        $menus = $this->menusModel->getMenusByCategory($categoryId);
        return $this->response->setJSON($menus);
    }

    public function delMenu($id)
    {
        $menuItem = $this->menusModel->singleMenuItem($id);
        
        if (!$menuItem) {
            if($this->request->isAJAX()) {
                $response = ['error' => false, 'msg' =>'Menu Item not found'];
                return $this->response->setJSON($response);
            } else {
                session()->setTempdata('error', 'Menu Item not found', 3);
                return redirect()->to('administrator/menu/groups');
            }
        }

        $this->menusModel->where('id', $id)->delete($id);
        
        if($this->request->isAJAX()) {
            // Send notification for menu item deletion
            $adminmessage = 'Menu Item Deleted: ' . $menuItem->title . ' (' . ($menuItem->menu_type ?? 'Unknown Type') . ')';
            $notificationType = 'menu';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            $response = ['error' => true, 'msg' =>'Menu Item deleted successfully'];
            return $this->response->setJSON($response);
        } else {
            // Send notification for menu item deletion
            $adminmessage = 'Menu Item Deleted: ' . $menuItem->title . ' (' . ($menuItem->menu_type ?? 'Unknown Type') . ')';
            $notificationType = 'menu';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Menu Item deleted successfully', 3);
            return redirect()->to('administrator/menu/group/'.$menuItem->category_alias);
        }
    }

    // Bulk menu operations (you might want to add this)
    public function bulkAction()
    {
        if ($this->request->getMethod() == 'POST') {
            $action = $this->request->getVar('action');
            $menuIds = $this->request->getVar('menu_ids');

            if (empty($menuIds)) {
                session()->setTempdata('error', 'No menu items selected', 3);
                return redirect()->to('administrator/menu/groups');
            }

            $successCount = 0;
            $failedCount = 0;

            foreach ($menuIds as $menuId) {
                $menu = $this->menusModel->find($menuId);
                if (!$menu) {
                    $failedCount++;
                    continue;
                }

                switch ($action) {
                    case 'activate':
                        if ($this->menusModel->update($menuId, ['status' => 1])) {
                            $successCount++;
                        } else {
                            $failedCount++;
                        }
                        break;
                    
                    case 'deactivate':
                        if ($this->menusModel->update($menuId, ['status' => 0])) {
                            $successCount++;
                        } else {
                            $failedCount++;
                        }
                        break;
                    
                    case 'delete':
                        if ($this->menusModel->delete($menuId)) {
                            $successCount++;
                        } else {
                            $failedCount++;
                        }
                        break;
                    
                    default:
                        $failedCount++;
                        break;
                }
            }

            if ($successCount > 0) {
                // Send notification for bulk action
                $adminmessage = 'Bulk Menu Action: ' . $action . ' performed on ' . $successCount . ' menu items';
                $notificationType = 'menu';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                session()->setTempdata('success', $successCount . ' menu items processed successfully', 3);
            }

            if ($failedCount > 0) {
                session()->setTempdata('error', $failedCount . ' menu items failed to process', 3);
            }
        }

        return redirect()->to('administrator/menu/groups');
    }
}