<?php

namespace Modules\ForexExchange\Models;

use CodeIgniter\Model;

class ForexExchangeModel extends Model
{
    protected $table = 'usc_forex_rates';
    protected $primaryKey = 'id';

    protected $allowedFields = ['currency_id', 'buying', 'selling', 'tt_buying', 'tt_selling','status',  'created_by', 'updated_at', 'updated_by'];

    protected $returnType     = 'object';
    protected $useTimestamps  = true;
    protected $useSoftDeletes = true;

    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    /**
     * Get all active forex rates with currency information
     * @return array
     */
    public function getActiveRates()
    {
        $builder = $this->db->table('usc_forex_rates as f');
        $builder->select([
            'f.*',
            'c.sortname as currency_code',
            'c.name as currency_name',
            'c.currency_symbol',
            'c.currency_code as currency_code_iso',
            'c.currency_name as currency_full_name',
            'f.buying as buying_rate',
            'f.selling as selling_rate',
            'f.tt_buying',
            'f.tt_selling',
            'f.updated_at as last_updated'
        ]);
        $builder->join('usc_country_currency as c', 'c.id=f.currency_id');
        $builder->where('f.deleted_at', null);
        $builder->where('f.status', 1); // Only active rates
        $builder->orderBy('f.id', 'ASC');
        $rates = $builder->get()->getResult();

        // Format rates for frontend compatibility
        foreach ($rates as $rate) {
            // Ensure numeric values are properly formatted
            $rate->buying_rate = (float)$rate->buying_rate;
            $rate->selling_rate = (float)$rate->selling_rate;
            $rate->tt_buying = (float)$rate->tt_buying;
            $rate->tt_selling = (float)$rate->tt_selling;
            
            // Generate flag icon URL from currency code
            $rate->flag_icon = "https://flagcdn.com/w160/" . strtolower($rate->currency_code) . ".png";
            
            // Ensure currency formatting is consistent
            $rate->currency_symbol = $rate->currency_symbol ?? '';
            $rate->currency_code = strtoupper($rate->currency_code_iso);
            $rate->currency_name = $rate->currency_full_name;
        }

        return $rates;
    }

    /**
     * Get all rates for admin display
     */
    public function getAll()
    {
        $builder = $this->db->table('usc_forex_rates as f');
        $builder->select('f.*, c.sortname, c.currency_symbol, c.name, c.currency_code, c.currency_name');
        $builder->join('usc_country_currency as c', 'c.id=f.currency_id');
        $builder->where('f.deleted_at', null);
        return $builder->get()->getResult();
    }
    
    /**
     * Get single forex rate with currency information
     */
    public function getSingleForex()
    {
        $builder = $this->db->table('usc_forex_rates as f');
        $builder->select('f.*, c.sortname, c.currency_symbol, c.name, c.currency_code, c.currency_name');
        $builder->join('usc_country_currency as c', 'c.id=f.currency_id');
        $builder->where('f.deleted_at', null);
        $builder->orderBy('f.updated_at', 'DESC');
        $builder->limit(1);
        return $builder->get()->getRow();
    }

    /**
     * Get a specific rate by currency code
     */
    public function getRateByCode($currencyCode)
    {
        $builder = $this->db->table('usc_forex_rates as f');
        $builder->select('f.*, c.sortname as currency_code, c.name as currency_name, c.currency_symbol, c.currency_code as currency_code_iso, c.currency_name as currency_full_name');
        $builder->join('usc_country_currency as c', 'c.id=f.currency_id');
        $builder->where('c.currency_code', $currencyCode);
        $builder->where('f.deleted_at', null);
        $rate = $builder->get()->getRow();

        if ($rate) {
            $rate->buying_rate = $rate->buying;
            $rate->selling_rate = $rate->selling;
            $rate->last_updated = $rate->updated_at;
            $rate->flag_icon = "https://flagcdn.com/w160/" . strtolower($rate->currency_code) . ".png";
            $rate->currency_code = strtoupper($rate->currency_code_iso);
            $rate->currency_name = $rate->currency_full_name;
        }

        return $rate;
    }

    /**
     * Get a specific rate by ID
     */
    public function getRate($id)
    {
        $builder = $this->db->table('usc_forex_rates as f');
        $builder->select('f.*, c.sortname as currency_code, c.name as currency_name, c.currency_symbol, c.currency_code as currency_code_iso, c.currency_name as currency_full_name');
        $builder->join('usc_country_currency as c', 'c.id=f.currency_id');
        $builder->where('f.id', $id);
        $builder->where('f.deleted_at', null);
        $rate = $builder->get()->getRow();

        if ($rate) {
            $rate->buying_rate = $rate->buying;
            $rate->selling_rate = $rate->selling;
            $rate->last_updated = $rate->updated_at;
            $rate->flag_icon = "https://flagcdn.com/w160/" . strtolower($rate->currency_code) . ".png";
            $rate->currency_code = strtoupper($rate->currency_code_iso);
            $rate->currency_name = $rate->currency_full_name;
        }

        return $rate;
    }

    /**
     * Update rates in bulk (for admin use)
     */
    public function updateRates($data)
    {
        $this->db->transStart();

        foreach ($data as $rate) {
            $builder = $this->db->table('usc_forex_rates as f');
            $builder->join('usc_country_currency as c', 'c.id=f.currency_id');
            $builder->where('c.currency_code', $rate['currency_code'])
                   ->set([
                       'buying' => $rate['buying_rate'],
                       'selling' => $rate['selling_rate'],
                       'updated_at' => date('Y-m-d H:i:s'),
                       'updated_by' => session()->get('user_id')
                   ])
                   ->update();
        }

        $this->db->transComplete();
        return $this->db->transStatus();
    }
}