<?php

namespace Modules\ForexExchange\Controllers;

use App\Controllers\Auth\AuthController;
use Modules\ForexExchange\Models\ForexExchangeModel;

class ForexExchange extends AuthController
{
    public $forexexchangeModel;

     public function initController($request, $response, $logger)
    
    {
        parent::initController($request, $response, $logger);
        $this->forexexchangeModel = new ForexExchangeModel();
    }

    public function index()
    {		
        // display header information
        $this->data['title'] = "Forex Exchange Rates";
        $this->data['bcrumb'] = "Forex Rates";

        // Get all countries with currencies for the selector
        $this->data['currencies'] = $this->countryCurrencyModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        // Get forex rates with country details (both active and inactive)
        $this->data['forex_rates'] = $this->forexexchangeModel->getAll();
        
        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    // AJAX endpoint for currency search
    public function searchCurrencies()
    {
        try {
            $search = $this->request->getGet('search');
            $id = $this->request->getGet('id');
            $builder = $this->countryCurrencyModel->builder();

            // If specific ID is requested, return that currency only
            if (!empty($id) && is_numeric($id)) {
                $builder->where('id', $id);
            } elseif (!empty($search)) {
                $builder->groupStart()
                       ->like('name', $search)
                       ->orLike('currency_code', $search)
                       ->orLike('currency_symbol', $search)
                       ->orLike('sortname', $search)
                       ->orLike('currency_name', $search)
                       ->groupEnd();
            }

            $builder->where('status', 1)
                   ->where('currency_code IS NOT NULL')
                   ->where('currency_code !=', '')
                   ->orderBy('name', 'ASC');

            // Only limit if not searching by specific ID
            if (empty($id)) {
                $builder->limit(50); // Limit results for better performance
            }

            $countries = $builder->get()->getResult();
            return $this->response->setJSON(['success' => true, 'data' => $countries]);
        } catch (\Exception $e) {
            log_message('error', 'Currency search error: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error fetching currencies'
            ])->setStatusCode(500);
        }
    }

    public function getCurrency($id)
    {
        try {
            if (!is_numeric($id)) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Invalid currency ID'
                ])->setStatusCode(400);
            }

            $country = $this->countryCurrencyModel->where('id', $id)
                                        ->where('status', 1)
                                        ->where('currency_code IS NOT NULL')
                                        ->where('currency_code !=', '')
                                        ->first();

            if (!$country) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Currency not found'
                ])->setStatusCode(404);
            }

            return $this->response->setJSON([
                'success' => true,
                'data' => $country
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Get currency error: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error fetching currency details'
            ])->setStatusCode(500);
        }
    }

    public function addForex()
    {
        $this->data['currencies'] = $this->countryCurrencyModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        // Ensure forex rates are available for the admin listing in the view
        $this->data['forex_rates'] = $this->forexexchangeModel->getAll();

        if($this->request->getMethod() == 'POST'){
            $rules = [
                'currency_id' => ['label'=>'Currency', 'rules'=>'required'],
                'buying' => ['label'=>'Buying', 'rules'=>'required|numeric'],
                'selling' => [
                    'label' => 'Selling',
                    'rules' => 'required|numeric',
                ],
                // T/T fields are optional, but if provided enforce tt_selling > tt_buying
                'tt_buying' => ['label' => 'T/T Buying', 'rules' => 'permit_empty|numeric'],
                'tt_selling' => [
                    'label' => 'T/T Selling',
                    'rules' => 'permit_empty|numeric',
                ],
            ];

            if($this->validate($rules))
            {
                // Additional numeric comparisons: selling must be > buying
                $buyingVal = (float) $this->request->getVar('buying');
                $sellingVal = (float) $this->request->getVar('selling');
                    if ($sellingVal <= $buyingVal) {
                        session()->setTempdata('error', 'Selling rate must be greater than Buying rate', 3);
                        session()->setTempdata('validation', $this->validator->listErrors(), 3);
                        return redirect()->back()->withInput();
                    }
                // If T/T provided, enforce tt_selling > tt_buying
                $ttBuyingVal = $this->request->getVar('tt_buying');
                $ttSellingVal = $this->request->getVar('tt_selling');
                if ($ttBuyingVal !== null && $ttBuyingVal !== '' && $ttSellingVal !== null && $ttSellingVal !== '') {
                    if ((float)$ttSellingVal <= (float)$ttBuyingVal) {
                            session()->setTempdata('error', 'T/T Selling must be greater than T/T Buying', 3);
                            session()->setTempdata('validation', $this->validator->listErrors(), 3);
                            return redirect()->back()->withInput();
                    }
                }
                $forexData = [
                    'currency_id'   => $this->request->getVar("currency_id"),
                    'buying'        => $this->request->getVar("buying"),
                    'selling'       => $this->request->getVar("selling"),
                    'tt_buying'     => $this->request->getVar("tt_buying"),
                    'tt_selling'    => $this->request->getVar("tt_selling"),
                    'status'        => $this->request->getVar("status") ? 1 : 0,
                    'created_by'    => $this->userdata->user_id,
                ];

                if($this->forexexchangeModel->save($forexData))
                {
                    // Get currency details for notification
                    $currencyId = $this->request->getVar("currency_id");
                    $currency = $this->countryCurrencyModel->where('id', $currencyId)->first();
                    
                    // Send standardized notification
                    $adminmessage = 'Forex Rate added for ' . $currency['name'] . ' (' . $currency['currency_code'] . ') - Buying: ' . $this->request->getVar("buying") . ', Selling: ' . $this->request->getVar("selling");
                    $notificationType = 'finance';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setTempdata('success', 'Forex Rate added successfully', 3);
                    return redirect()->to('administrator/forex/rates');
                } else {
                    session()->setTempdata('error', 'Forex Rate NOT added', 3);
                    return redirect()->to('administrator/forex/rates');
                }
            } else {
                // Use tempdata and redirect so the browser does a GET (prevents CSRF token reuse on refresh)
                session()->setTempdata('validation', $this->validator->listErrors(), 3);
                return redirect()->back()->withInput();
            }
        }
        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    public function editForex($id = null)
    {
        $this->data['title'] = "Update Forex Exchange Rates";
        $this->data['bcrumb'] = "Update Forex Rates";

        $this->data['currencies'] = $this->countryCurrencyModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        // Get forex rates with country details
        $this->data['forex_rates'] = $this->forexexchangeModel->getAll();

        // Get single rate with country details
        $this->data['singlerate'] = $this->forexexchangeModel->getRate($id);

        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    public function updateForex()
    {
        $this->data['title'] = "Update Forex Exchange Rates";
        $this->data['bcrumb'] = "Update Forex Rates";

        $this->data['currencies'] = $this->countryCurrencyModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        // Provide forex rates for the view so the listing table doesn't error when validation fails
        $this->data['forex_rates'] = $this->forexexchangeModel->getAll();

        if($this->request->getMethod() == 'POST'){
            $rules = [
                'currency_id' => ['label'=>'Currency', 'rules'=>'required'],
                'buying' => ['label'=>'Buying', 'rules'=>'required|numeric'],
                'selling' => [
                    'label' => 'Selling',
                    'rules' => 'required|numeric',
                ],
                // T/T fields optional but validate relative values when present
                'tt_buying' => ['label' => 'T/T Buying', 'rules' => 'permit_empty|numeric'],
                'tt_selling' => [
                    'label' => 'T/T Selling',
                    'rules' => 'permit_empty|numeric',
                ],
            ];

            if($this->validate($rules))
            {
                // Additional numeric comparisons: selling must be > buying
                $buyingVal = (float) $this->request->getVar('buying');
                $sellingVal = (float) $this->request->getVar('selling');
                if ($sellingVal <= $buyingVal) {
                    session()->setTempdata('error', 'Selling rate must be greater than Buying rate', 3);
                    session()->setTempdata('validation', $this->validator->listErrors(), 3);
                    return redirect()->back()->withInput();
                }
            

                $id = $this->request->getVar("id");
                $forexData = [
                    'currency_id'   => $this->request->getVar("currency_id"),
                    'buying'        => $this->request->getVar("buying"),
                    'selling'       => $this->request->getVar("selling"),
                    'tt_buying'     => $this->request->getVar("tt_buying"),
                    'tt_selling'    => $this->request->getVar("tt_selling"),
                    'status'        => $this->request->getVar("status") ? 1 : 0,
                    'updated_by'    => $this->userdata->user_id,
                    'updated_at'    => date('Y-m-d h:i:s'),
                ];

                if($this->forexexchangeModel->update($id, $forexData))
                {
                    // Get currency details for notification
                    $currencyId = $this->request->getVar("currency_id");
                    $currency = $this->countryCurrencyModel->where('id', $currencyId)->first();
                    
                    // Send standardized notification
                    $adminmessage = 'Forex Rate updated for ' . $currency['name'] . ' (' . $currency['currency_code'] . ') - Buying: ' . $this->request->getVar("buying") . ', Selling: ' . $this->request->getVar("selling");
                    $notificationType = 'finance';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setTempdata('success', 'Forex Rate updated successfully', 3);
                    return redirect()->to('administrator/forex/rates');
                } else {
                    session()->setTempdata('error', 'Forex Rate NOT updated', 3);
                    return redirect()->to('administrator/forex/rates');
                }
            } else {
                // Redirect back with validation errors to avoid CSRF token reuse on refresh
                session()->setTempdata('validation', $this->validator->listErrors(), 3);
                return redirect()->back()->withInput();
            }
        }
        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    public function delForex($id = null)
    {
        // Get forex rate details before deletion for notification
        $forexRate = $this->forexexchangeModel->getRate($id);
        
        if ($forexRate && $this->forexexchangeModel->delete($id)) 
        {
            // Send notification for forex rate deletion
            // Model returns currency fields as currency_name / currency_full_name and codes as currency_code_iso or currency_code
            $currencyName = $forexRate->currency_name ?? $forexRate->currency_full_name ?? ($forexRate->name ?? null);
            $currencyCode = $forexRate->currency_code_iso ?? $forexRate->currency_code ?? $forexRate->sortname ?? 'N/A';
            $adminmessage = 'Forex Rate deleted for ' . ($currencyName ?? 'Unknown Currency') . ' (' . $currencyCode . ')';
            $notificationType = 'finance';
            $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

            session()->setTempdata('success', 'Forex Rate deleted successfully', 3);
            return redirect()->to('administrator/forex/rates');
        } else {
            session()->setTempdata('error', 'Forex Rate NOT deleted or not found', 3);
            return redirect()->to('administrator/forex/rates');
        }
    }

    // Bulk update forex rates (useful for daily updates)
    public function bulkUpdate()
    {
        if($this->request->getMethod() == 'POST') {
            $rates = $this->request->getVar('rates');
            $updatedCount = 0;

            if(is_array($rates)) {
                foreach($rates as $id => $rateData) {
                    if(isset($rateData['buying']) && isset($rateData['selling'])) {
                        $updateData = [
                            'buying' => $rateData['buying'],
                            'selling' => $rateData['selling'],
                            'tt_buying' => $rateData['tt_buying'] ?? null,
                            'tt_selling' => $rateData['tt_selling'] ?? null,
                            'updated_by' => $this->userdata->user_id,
                            'updated_at' => date('Y-m-d H:i:s')
                        ];

                        if($this->forexexchangeModel->update($id, $updateData)) {
                            $updatedCount++;
                        }
                    }
                }

                if($updatedCount > 0) {
                    // Send notification for bulk update
                    $adminmessage = 'Bulk Forex Rates updated: ' . $updatedCount . ' currencies modified';
                    $notificationType = 'finance';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setTempdata('success', $updatedCount . ' forex rates updated successfully', 3);
                } else {
                    session()->setTempdata('error', 'No forex rates were updated', 3);
                }
            } else {
                session()->setTempdata('error', 'Invalid data format for bulk update', 3);
            }
            
            return redirect()->to('administrator/forex/rates');
        }
    }
}