<?php

namespace Modules\Admin\Controllers;

use App\Controllers\Auth\AuthController;

class AdminUsers extends AuthController
{
    public function initController($request, $response, $logger)
    {
        parent::initController($request, $response, $logger);
    }

    public function index()
    {
        $this->data['title'] = "User List";
        $this->data['bcrumb'] = 'All Users';

        $this->data['groups'] = $this->usergroupsModel->getUserGroups();
        $this->data['users'] = $this->usersModel->getAllUsers();

        return view('Modules\Admin\Views\users\all_users', $this->data);
    }

    public function profile()
    {
        $this->data['title'] = 'My Profile';
        $this->data['bcrumb'] = 'My Profile';

        $id = $this->userdata->user_id;
        $this->data['userdata'] = $this->usersModel->singleUser($id);
        $this->data['user_roles'] = $this->usergroupsModel->getAll();
        $this->data['departments']  = $this->departmentModel->getAll();
        $this->data['countries'] = $this->countryModel->getAll();
        $this->data['groups'] = $this->usergroupsModel->getAll();

        // Set user_states and user_cities based on user's location
        $user = $this->data['userdata'];
        if ($user && $user->country_id) {
            $this->data['user_states'] = $this->stateModel->where('country_id', $user->country_id)->findAll();
        } else {
            $this->data['user_states'] = [];
        }
        if ($user && $user->state_id) {
            $this->data['user_cities'] = $this->cityModel->where('state_id', $user->state_id)->findAll();
        } else {
            $this->data['user_cities'] = [];
        }

        if($this->request->getMethod() == 'POST')
        {
            //Rules set
            $rules = [
                'first_name'    => ["label"=>"First Name", "rules"=>"required"],
                'last_name'     => ["label"=>"Other Name", "rules"=>"required"],
                // use the actual table name "usc_users" for unique checks
                'email'         => ['label' => 'Email', 'rules' => "required|min_length[5]|max_length[100]|valid_email|is_unique[usc_users.email,id,{$id}]"],
            ];
                
            if ($this->validate($rules))
            {
                $regData = [
                    'first_name' 			=> $this->request->getVar('first_name'),
                    'last_name' 			=> $this->request->getVar('last_name'),
                    'email' 				=> $this->request->getVar('email'),
                    'mobile'				=> $this->request->getVar('mobile'),
                    'address'				=> $this->request->getVar('address'),
                    'position'				=> $this->request->getVar('position'),
                    'department_id'			=> $this->request->getVar('department_id'),
                    'twitter'				=> $this->request->getVar('twitter'),
                    'facebook'				=> $this->request->getVar('facebook'),
                    'instagram'				=> $this->request->getVar('instagram'),
                    'linkedin'				=> $this->request->getVar('linkedin'),
                    'country_id'			=> $this->request->getVar('country_id'),
                    'state_id'				=> $this->request->getVar('state_id'),
                    'city_id'				=> $this->request->getVar('city_id'),
                    'description'		    => $this->request->getVar('description'),
                ];
                
                // Handle photo upload if provided
                $file = $this->request->getFile('photo');
                if ($file && $file->isValid() && !$file->hasMoved()) {
                    $destPath = FCPATH . 'media/files/';
                    if (!is_dir($destPath)) {
                        mkdir($destPath, 0755, true);
                    }
                    $newName = $file->getRandomName();
                    $file->move($destPath, $newName);
                    $regData['photo'] = $newName;
                }
                
                $results = $this->usersModel->update($id, $regData);
                if($results)
                {
                    // Send notification for profile update
                    $adminmessage = 'User: ' . $this->userdata->first_name . ' ' . $this->userdata->last_name . ' updated their profile';
                    $notificationType = 'user';
                    $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                    session()->setFlashdata('success', 'Profile updated successfully', 3);
                    return redirect()->to('administrator/profile');

                }else
                {
                    session()->setTempdata('error', 'Sorry! Profile not updated', 3);
                    return redirect()->to('administrator/profile');
                }
            }else{
                $this->data['validation'] = $this->validator;
            }
        }

    return view('Modules\Admin\Views\dashboard\users_profile', $this->data);
    }

    public function changePassword()
    {
        if($this->request->getMethod() == 'POST')
        {
            $rules = [
                'password'	=> ['label' => 'Password', 'rules' => 'required'],
                'new_password'	=> ['label' => 'New Password', 'rules' => 'required|min_length[8]|max_length[20]|alpha_numeric_punct'],
		        'renew_password'	    => ['label' => 'Confirm Password', 'rules' => 'required|matches[new_password]'],
            ];

            if(!$this->validate($rules)){
                session()->setTempdata('error', $this->validator->listErrors(), 3);
                return redirect()->to('administrator/profile');
            }

            $user = $this->usersModel->find($this->userdata->user_id);

            if (!$user) {
                session()->setTempdata('error', 'User not found', 3);
                return redirect()->to('administrator/profile');
            }

            if(!password_verify($this->request->getVar('password'), $user->password ?? $user['password'])) {
                session()->setTempdata('error', 'Incorrect current password', 3);
                return redirect()->to('administrator/profile');
            }

            $password = password_hash($this->request->getVar('new_password'), PASSWORD_DEFAULT);
            $newPwd = [
                'password' => $password,
            ];
            if($this->usersModel->update($this->userdata->user_id, $newPwd))
            {
                // Send notification for password change
                $adminmessage = 'User: ' . $this->userdata->first_name . ' ' . $this->userdata->last_name . ' changed their password';
                $notificationType = 'security';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                session()->setTempdata('success', 'Password updated successfully', 3);
                return redirect()->to('administrator/profile');
            } else {
                session()->setTempdata('error', 'Sorry! Password not updated', 3);
                return redirect()->to('administrator/profile');
            }

        }
    }

    public function addUser()
{
    $this->data['title'] = 'Add User';
    $this->data['bcrumb'] = 'Add User';
    $this->data['countries']    = $this->countryModel->getAll();
    $this->data['states']       = $this->stateModel->getAll();
    $this->data['cities']       = $this->cityModel->getAll();
    $this->data['groups']       = $this->usergroupsModel->getAll();
    $this->data['user_roles']   = $this->usergroupsModel->getAll();
    $this->data['departments']  = $this->departmentModel->getAll();

    helper('security');

    if($this->request->getMethod() == 'POST')
    {
        $hash = bin2hex(random_bytes(24));
        $validation = \Config\Services::validation();

        // Custom email validation with better error message
        $email = $this->request->getVar('email');
        $existingUser = $this->usersModel->where('email', $email)->first();
        
        if ($existingUser) {
            session()->setFlashdata('error', 'This email address is already registered. Please use a different email address.');
            return redirect()->back()->withInput();
        }

        //Rules Set with required role
        $rules = [
            'first_name'    => ["label"=>"First Name", "rules"=>"required"],
            'last_name'     => ["label"=>"Other Name", "rules"=>"required"],
            'email'         => ['label' => 'Email', 'rules' => 'required|min_length[5]|max_length[100]|valid_email'],
            'password'      => ['label' => 'Password', 'rules' => 'required|min_length[8]|max_length[20]|alpha_numeric_punct'],
            'cpass'         => ['label' => 'Confirm Password', 'rules' => 'required|matches[password]'],
            'group_id'      => ['label' => 'User Role', 'rules' => 'required'], // Added required role
        ];
            
        if ($this->validate($rules))
        {
            $regData = [
                'first_name'            => $this->request->getVar('first_name'),
                'last_name'             => $this->request->getVar('last_name'),
                'email'                 => $this->request->getVar('email'),
                'status'                => 'active',
                'password'              => password_hash($this->request->getVar('password'), PASSWORD_DEFAULT),
                'mobile'                => $this->request->getVar('mobile'),
                'address'               => $this->request->getVar('address'),
                'position'              => $this->request->getVar('position'),
                'department_id'         => $this->request->getVar('department_id'),
                'twitter'               => $this->request->getVar('twitter'),
                'facebook'              => $this->request->getVar('facebook'),
                'instagram'             => $this->request->getVar('instagram'),
                'linkedin'              => $this->request->getVar('linkedin'),
                'country_id'            => $this->request->getVar('country_id'),
                'state_id'              => $this->request->getVar('state_id'),
                'city_id'               => $this->request->getVar('city_id'),
                'description'           => $this->request->getVar('description'),
            ];
            
            $results = $this->usersModel->save($regData);
            if($results)
            {
                $users = $this->usersModel->getAdminUsers();
                $firstName = $this->request->getVar('first_name');
                $lastName = $this->request->getVar('last_name');
                $id = $this->usersModel->getInsertID();

                //Save user in Auth Group with user_type
                $groupData = [
                    'user_id' => $id,
                    'group_id'=> $this->request->getVar('group_id'),
                    'user_type' => 'admin',
                    'status' => '1',
                ];
                $this->userauthgroupModel->save($groupData);

                //Notification
                $adminmessage = 'New User: '.$firstName.' '.$lastName.' account has been <strong>CREATED</strong>';
                $notificationType = 'security';
                $this->sendAdminNotifications($adminmessage, $notificationType, $id);

                if($this->request->isAJAX()){
                    $response = [
                        'success' => true,
                        'msg'     => 'User created successfully',
                    ];
                    return $this->response->setJSON($response);

                }else{
                    session()->setFlashdata('success', 'User created successfully', 3);
                    return redirect()->to('administrator/users');
                }
            }else
            {
                if($this->request->isAJAX()){
                    $response = [
                        'error'     => true,
                        'msg'     => 'Sorry! Account not created',
                    ];
                    return $this->response->setJSON($response);

                }else{
                session()->setTempdata('error', 'Sorry! Account not created', 3);
                return redirect()->to('administrator/users/add');
                }
            }
        }else{
            $this->data['validation'] = $this->validator;
        }
    }
    return View('Modules\Admin\Views\users\add_user', $this->data);
}

    public function editUser($id)
{
    $user = $this->usersModel->where('id', $id)->first();
    if (!$user) {
        session()->setTempdata('error', 'User not found', 3);
        return redirect()->to('administrator/users');
    }
    
    $this->data['title'] = 'Edit '.$user->first_name.' '.$user->last_name;
    $this->data['bcrumb'] = $user->first_name.' '.$user->last_name .' Details';

    $this->data['user'] = $this->usersModel->singleUser($id);
    $this->data['user_roles'] = $this->usergroupsModel->getAll();
    $this->data['departments'] = $this->departmentModel->getAll();
    $this->data['countries'] = $this->countryModel->getAll();
    $this->data['states'] = $this->stateModel->getAll();
    $this->data['cities'] = $this->cityModel->getAll();
    $this->data['groups'] = $this->usergroupsModel->getAll();

    if($this->request->getMethod() == 'POST')
    {
        $email = $this->request->getVar('email');
        
        // Custom email validation for editing
        if ($email !== $user->email) {
            $existingUser = $this->usersModel->where('email', $email)->first();
            if ($existingUser) {
                session()->setFlashdata('error', 'This email address is already registered. Please use a different email address.');
                return redirect()->back()->withInput();
            }
        }

        //Rules set with required role
        $rules = [
            'first_name'    => ["label"=>"First Name", "rules"=>"required"],
            'last_name'     => ["label"=>"Other Name", "rules"=>"required"],
            'email'         => ['label' => 'Email', 'rules' => "required|min_length[5]|max_length[100]|valid_email"],
            'group_id'      => ['label' => 'User Role', 'rules' => 'required'], // Added required role
        ];
            
        if ($this->validate($rules))
        {
            $regData = [
                'first_name'            => $this->request->getVar('first_name'),
                'last_name'             => $this->request->getVar('last_name'),
                'email'                 => $this->request->getVar('email'),
                'mobile'                => $this->request->getVar('mobile'),
                'address'               => $this->request->getVar('address'),
                'position'              => $this->request->getVar('position'),
                'department_id'         => $this->request->getVar('department_id'),
                'twitter'               => $this->request->getVar('twitter'),
                'facebook'              => $this->request->getVar('facebook'),
                'instagram'             => $this->request->getVar('instagram'),
                'linkedin'              => $this->request->getVar('linkedin'),
                'country_id'            => $this->request->getVar('country_id'),
                'state_id'              => $this->request->getVar('state_id'),
                'city_id'               => $this->request->getVar('city_id'),
                'description'           => $this->request->getVar('description'),
                'status'                => $this->request->getVar('status'),
            ];
            
            // Handle photo upload if provided (edit user)
            $file = $this->request->getFile('photo');
            if ($file && $file->isValid() && !$file->hasMoved()) {
                $destPath = FCPATH . 'media/files/';
                if (!is_dir($destPath)) {
                    mkdir($destPath, 0755, true);
                }
                $newName = $file->getRandomName();
                $file->move($destPath, $newName);
                $regData['photo'] = $newName;
            }

            $results = $this->usersModel->update($id, $regData);
            if($results)
            {
                // Update user group
                if($this->request->getVar('group_id')) {
                    $authGroupRecord = $this->userauthgroupModel->where('user_id', $id)
                                                                ->where('user_type', 'admin')
                                                                ->first();
                    if($authGroupRecord) {
                        $grpData = [
                            'group_id' => $this->request->getVar('group_id'),
                        ];
                        $this->userauthgroupModel->update($authGroupRecord->id ?? $authGroupRecord['id'], $grpData);
                    }
                }

                $firstName = $this->request->getVar('first_name');
                $lastName = $this->request->getVar('last_name');
                $userName = $firstName.' '.$lastName;
                $adminmessage = 'User '.$userName.' account details updated successfully';
                $notificationType = 'security';
                $this->sendAdminNotifications($adminmessage, $notificationType, $id);

                session()->setFlashdata('success', 'User updated successfully', 3);
                return redirect()->to('administrator/users');

            }else
            {
                session()->setTempdata('error', 'Sorry! User not updated', 3);
                return redirect()->to('administrator/users/edit/'.$id);
            }
        }else{
            $this->data['validation'] = $this->validator;
        }
    }
    return View('Modules\Admin\Views\users\edit_user', $this->data);
}

    public function deleteUser($id)
    {
        $admins = $this->usersModel->getAdminUsers();
        $user = $this->usersModel->singleUser($id);
         
        $this->usersModel->where('id', $id)->delete($id);
        if($this->request->isAJAX())
        {
            $response = [
            'success' => true,
            'msg' =>'User deleted successfully',    
            ];
            return $this->response->setJSON($response);
        }else{
            //Save & Send Notification
            $adminmessage = $user->first_name .' '.$user->last_name.'\'s account has been <strong>DELETED</strong>';
            $notificationType = 'security';
            $this->sendAdminNotifications($adminmessage, $notificationType, $id);

            session()->setFlashdata('error', 'User deleted successfully', 3);
            return redirect()->to('administrator/users');
        }
    }

     //Update user password
    public function changePwd()
    {
        if($this->request->getMethod() == 'POST')
        {
            $rules = [
                'password'	=> ['label' => 'Password', 'rules' => 'required|min_length[8]|max_length[20]|alpha_numeric_punct'],
		        'cpass'	    => ['label' => 'Confirm Password', 'rules' => 'required|matches[password]'],
            ];

            if(!$this->validate($rules)){
                $validation = \Config\Services::validation();
                $errors = $validation->getErrors();
                
                $errorMessage = implode('<br>', $errors);
                session()->setFlashdata('error', $errorMessage, 5);
                return redirect()->to('administrator/users');
            }
                

            $id = $this->request->getVar('id');
            $u = $this->usersModel->where('id', $id)->first($id);
            $password = password_hash($this->request->getVar('password'), PASSWORD_DEFAULT);
            $newPwd = [
                'password' => $password,
            ];
            if($this->usersModel->update($id, $newPwd))
            {
                $adminmessage = 'User '.($u->first_name ?? $u['first_name']).' '.($u->last_name ?? $u['last_name']).' password updated by Admin';
                $notificationType = 'security';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                if ($this->request->isAJAX()) {
                    return $this->response->setJSON(['success' => true, 'msg' => 'User password updated successfully']);
                }

                session()->setFlashdata('success', 'User password updated successfully', 3);
                return redirect()->to('administrator/users');
            }

        }
    }

    //Change User Group
    public function changeGroup()
    {
        $userId = $this->request->getVar('id');
        $u = $this->usersModel->singleUser($userId);

        if($this->request->getMethod() == 'POST')
        {
            // Find the user_auth_group record ID for this user (admin type)
            $authGroupRecord = $this->userauthgroupModel->where('user_id', $userId)
                                                        ->where('user_type', 'admin')
                                                        ->first();

            if(!$authGroupRecord){
                session()->setFlashdata('error', 'User auth group not found', 3);
                return redirect()->to('administrator/users');
            }

            $grpData = [
                'group_id' => $this->request->getVar('group_id'),
            ];
            // Use the auth_group record ID, not user_id
            $results = $this->userauthgroupModel->update($authGroupRecord->id ?? $authGroupRecord['id'], $grpData);

            if($results)
            {
                //Notification save
                $adminmessage = 'User - '.$u->first_name.' '.$u->last_name.' GROUP updated';
                $notificationType = 'security';
                $this->sendAdminNotifications($adminmessage, $notificationType, $this->userdata->user_id);

                session()->setFlashdata('success', 'User Group updated successfully', 3);
                return redirect()->to('administrator/users');
            }
        }
    }

    // Get states by country ID for cascading dropdown
    public function getStates($countryId)
    {
        $states = $this->countryModel->getStates($countryId);
        return $this->response->setJSON($states);
    }

    // Get cities by state ID for cascading dropdown
    public function getCities($stateId)
    {
        $cities = $this->stateModel->getCities($stateId);
        return $this->response->setJSON($cities);
    }

}