<?php

namespace App\Models;

use CodeIgniter\Model;

class NotificationsModel extends Model
{
    protected $table = 'notifications';
    protected $primaryKey = 'id';

    protected $allowedFields = ['message', 'user_id', 'status', 'notification_type', 'created_by', 'updated_by', 'updated_at'];

    protected $returnType     = 'array';
    protected $useTimestamps  = true;
    protected $useSoftDeletes = true;

    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';


    //Latest Activities
    public function getLatestActivities($user)
    {
        $builder = $this->db->table('notifications as n')
            ->select('n.*, first_name, last_name')
            ->join('users as u', 'u.id=n.created_by', 'left')
            ->where('n.user_id', $user)
            ->where('n.status', '0')
            ->orderBy('n.id', 'DESC');
        return $builder->get()->getResultArray();
    }


    public function getTotalUnreadAlerts(int $userId): int
{
    return $this->where('user_id', $userId)
                 ->where('status', '0') 
                 ->countAllResults();
}
    /**
     * Get latest notifications for a user (non-breaking helper).
     *
     * @param int $user
     * @param int $limit
     * @return array
     */
    public function getLatest(int $user, int $limit = 10)
    {
        $builder = $this->db->table('notifications as n')
                ->select('n.*, u.first_name, u.last_name')
                ->join('users as u', 'u.id=n.created_by', 'left')
                ->where('n.deleted_at', null)
                ->where('n.user_id', $user)
                ->orderBy('n.created_at', 'DESC')
                ->limit($limit);
        return $builder->get()->getResultArray();
    }

    // Get total unread notifications for a user
    public function getTotalUserAlerts($userId)
    {
        return $this->where('user_id', $userId)
                    ->countAllResults();
    }

    // Get all alerts/notifications for a user
    public function getAlertsByUserId($userId)
    {
        return $this->where('user_id', $userId)
                    ->where('deleted_at IS NULL')
                    ->orderBy('created_at', 'DESC')
                    ->findAll();
    }

    /**
     * Get notifications for a user with optional filters.
     * $filters = [ 'type' => 'security', 'status' => 0, 'from' => '2025-01-01', 'to' => '2025-01-31' ]
     */
    public function getByFilters(int $userId, array $filters = [], ?int $limit = null, ?int $offset = null)
    {
        $builder = $this->builder();
        $builder->where('user_id', $userId);
        $builder->where('deleted_at', null);

        // Support special token for unspecified types
        if (isset($filters['type']) && $filters['type'] !== '') {
            if ($filters['type'] === '__unspecified__') {
                $builder->groupStart();
                $builder->where('notification_type', '');
                $builder->orWhere('notification_type IS NULL', null, false);
                $builder->groupEnd();
            } else {
                $builder->where('notification_type', $filters['type']);
            }
        }

        if (isset($filters['status']) && $filters['status'] !== '') {
            $builder->where('status', intval($filters['status']));
        }

        if (!empty($filters['from'])) {
            $builder->where('created_at >=', $filters['from']);
        }

        if (!empty($filters['to'])) {
            $builder->where('created_at <=', $filters['to']);
        }

        $builder->orderBy('created_at', 'DESC');

        if ($limit !== null) {
            if ($offset !== null) {
                $builder->limit($limit, $offset);
            } else {
                $builder->limit($limit);
            }
        }

        return $builder->get()->getResultArray();
    }

    /**
     * Count notifications with filters for a user
     */
    public function countByFilters(int $userId, array $filters = [])
    {
        $builder = $this->builder();
        $builder->where('user_id', $userId);

        if (isset($filters['type']) && $filters['type'] !== '') {
            if ($filters['type'] === '__unspecified__') {
                $builder->groupStart();
                $builder->where('notification_type', '');
                $builder->orWhere('notification_type IS NULL', null, false);
                $builder->groupEnd();
            } else {
                $builder->where('notification_type', $filters['type']);
            }
        }

        if (isset($filters['status']) && $filters['status'] !== '') {
            $builder->where('status', intval($filters['status']));
        }

        if (!empty($filters['from'])) {
            $builder->where('created_at >=', $filters['from']);
        }

        if (!empty($filters['to'])) {
            $builder->where('created_at <=', $filters['to']);
        }

        return $builder->countAllResults();
    }

    /**
     * Return distinct notification types for a user with counts
     */
    public function getTypesWithCounts(int $userId)
    {
        // Return empty string for unspecified types so view can treat them specially
        $builder = $this->db->table('notifications')
                    ->select("CASE WHEN notification_type IS NULL OR notification_type = '' THEN '' ELSE notification_type END as notification_type, COUNT(*) as cnt", false)
                    ->where('user_id', $userId)
                    ->groupBy('notification_type')
                    ->orderBy('cnt', 'DESC');

        return $builder->get()->getResultArray();
    }

    /**
     * Get all notifications for user without limit
     */
    public function getAllForUser(int $userId)
    {
        return $this->where('user_id', $userId)
                    ->orderBy('created_at', 'DESC')
                    ->findAll();
    }


}