// Unified, organized module for Settings page interactions
(function ($) {
  'use strict';

  // Utility: safe DataTables init
  function initTables() {
    if ($.fn && $.fn.DataTable) {
      if ($('#permissionTable').length) $('#permissionTable').DataTable();
      if ($('#statusTable').length) $('#statusTable').DataTable();
    }
  }

  // Utility: optional CSRF header for AJAX
  function setupAjaxCsrf() {
    if (window.csrfTokenValue) {
      $.ajaxSetup({
        headers: { 'X-CSRF-TOKEN': window.csrfTokenValue },
      });
    }
  }

  // Company modal section
  function initCompany() {
    if (typeof bootstrap === 'undefined') return;
    const coForm = $('#companyForm');
    if (!coForm.length) return;

    const coModal = new bootstrap.Modal('#companyModal');

    $(document).on('click', '.edit-company', function (event) {
      event.preventDefault();
      const companyId = $(this).data('id');
      $.get(base_url + 'administrator/settings/company', { id: companyId })
        .done(function (response) {
          if (!response || !response.data) return;
          const company = response.data;
          coForm.find('input[name="id"]').val(company.id);
          coForm.find('input[name="name"]').val(company.name);
          coForm.find('input[name="email"]').val(company.email);
          coForm.find('input[name="tel"]').val(company.tel);
          coForm.find('input[name="tax_id"]').val(company.tax_id);
          coForm.find('input[name="website"]').val(company.website);
          coForm.find('select[name="default_currency_id"]').val(company.default_currency_id);
          coForm.find('input[name="address_1"]').val(company.address_1);
          coForm.find('input[name="address_2"]').val(company.address_2);
          coForm.find('textarea[name="description"]').val(company.description);
          coForm.find('input[name="area_code"]').val(company.area_code);
          coForm.find('textarea[name="vision"]').val(company.vision);
          coForm.find('textarea[name="mission"]').val(company.mission);
          coForm.find('select[name="country_id"]').val(company.country_id);

          const displayImage = $('#display-image').empty();
          if (company.photo) {
            const img = $('<img>', { class: 'img-fluid', alt: 'Company Photo' });
            img.attr('src', company.photo);
            displayImage.append(img);
          }

          $('#companyModalLabel').text('Edit Company');
          $('.sbtn').text('Update');
          coModal.show();
        })
        .fail(function () {
          toastr.error('Failed to load company data');
        });
    });

    coForm.on('submit', function (event) {
      event.preventDefault();
      const formData = new FormData(this);
      if (window.csrfTokenName && window.csrfTokenValue) {
        formData.append(window.csrfTokenName, window.csrfTokenValue);
      }

      $.ajax({
        url: base_url + 'administrator/settings/company',
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        dataType: 'json',
        success: function (response) {
          if (response && response.success) {
            toastr.success(response.message || 'Saved');
            coModal.hide();
            setTimeout(() => location.reload(), 1000);
            return;
          }

          // Validation errors
          $('.invalid-feedback').text('');
          $('.is-invalid').removeClass('is-invalid');
          if (response && response.errors) {
            $.each(response.errors, function (key, value) {
              const input = coForm.find('[name="' + key + '"]');
              input.addClass('is-invalid');
              input.next('.invalid-feedback').text(value);
            });
          } else {
            toastr.error((response && response.message) || 'An error occurred');
          }
        },
        error: function (xhr) {
          toastr.error('Server error occurred. Please try again.');
          if (xhr && xhr.responseText) console.error(xhr.responseText);
        },
      });
    });
  }

  // Permissions section
  function initPermissions() {
    if (typeof bootstrap === 'undefined') return;
    const permiForm = $('#permissionForm');
    if (!permiForm.length) return;

    const permiModal = new bootstrap.Modal('#permissionModal');

    $('[data-bs-target="#permissionModal"]').on('click', function () {
      if (permiForm[0]) permiForm[0].reset();
      permiForm.find('input[name="id"]').val('');
      $('#permissionModalLabel').text('Add New Permission');
      $('.is-invalid').removeClass('is-invalid');
      $('.invalid-feedback').text('');
    });

    $(document).on('click', '.edit-permission', function () {
      const permissionId = $(this).data('id');
      $.get(base_url + 'administrator/settings/permissions', { id: permissionId }, function (response) {
        if (!response || !response.data) return;
        const permission = response.data;
        permiForm.find('input[name="id"]').val(permission.id);
        permiForm.find('input[name="name"]').val(permission.name);
        permiForm.find('select[name="parent_id"]').val(permission.parent_id || '');
        permiForm.find('select[name="module_section_id"]').val(permission.module_section_id);
        permiForm.find('input[name="module"]').val(permission.module);
        permiForm.find('textarea[name="description"]').val(permission.description || '');
        $('#permissionModalLabel').text('Edit Permission');
        permiModal.show();
      });
    });

    permiForm.on('submit', function (e) {
      e.preventDefault();
      const payload = permiForm.serialize();
      $.ajax({
        url: base_url + 'administrator/settings/permissions',
        method: 'POST',
        data: payload,
        success: function (response) {
          if (response && response.success) {
            toastr.success(response.message || 'Saved');
            permiModal.hide();
            setTimeout(() => location.reload(), 1000);
            return;
          }
          toastr.error('An error occurred');
        },
        error: function (xhr) {
          const errors = (xhr.responseJSON && xhr.responseJSON.errors) || null;
          if (errors) {
            $.each(errors, function (key, value) {
              const input = permiForm.find('[name="' + key + '"]');
              input.addClass('is-invalid');
              input.next('.invalid-feedback').text(value);
            });
          }
        },
      });
    });
  }

  // Policies section
  function initPolicies() {
    if (typeof bootstrap === 'undefined') return;
    const policyForm = $('#policyModal form');
    if (!policyForm.length) return;

    const policyModal = new bootstrap.Modal('#policyModal');

    $('[data-bs-target="#policyModal"]').on('click', function () {
      if (policyForm[0]) policyForm[0].reset();
      policyForm.find('input[name="id"]').val('');
      $('#staticBackdropLabel').text('Add New Policy');
      $('.sbtn').text('Save Policy');
      $('.is-invalid').removeClass('is-invalid');
      $('.invalid-feedback').text('');
      $('#currentPhoto').hide().empty();
    });

    $(document).on('click', '.edit-policy', function (e) {
      e.preventDefault();
      const policyId = $(this).data('id');
      $.get(base_url + 'administrator/settings/policies', { id: policyId })
        .done(function (response) {
          if (!response || !response.data) {
            toastr.error('Policy data not found');
            return;
          }
          const policy = response.data;
          policyForm.find('input[name="id"]').val(policy.id);
          policyForm.find('input[name="title"]').val(policy.title);
          policyForm.find('select[name="policy_type"]').val(policy.policy_type);
          policyForm.find('textarea[name="description"]').val(policy.description);

          const cur = $('#currentPhoto').empty();
          if (policy.photo) {
            const wrap = $('<div>', { class: 'mb-3' });
            wrap.append($('<label>', { class: 'form-label', text: 'Current Photo' }));
            wrap.append($('<img>', { class: 'img-thumbnail', css: { maxHeight: '50px' } }).attr('src', policy.photo));
            wrap.append($('<div>', { class: 'form-text', text: 'Leave blank to keep existing', css: { fontSize: '10px', color: '#f00' } }));
            cur.append(wrap).show();
          } else {
            cur.hide();
          }

          $('#staticBackdropLabel').text('Edit Policy');
          $('.sbtn').text('Update Policy');
          policyModal.show();
        })
        .fail(function () {
          toastr.error('Failed to load policy data');
        });
    });

    policyForm.on('submit', function (e) {
      e.preventDefault();
      const formData = new FormData(this);
      if (window.csrfTokenName && window.csrfTokenValue) {
        formData.append(window.csrfTokenName, window.csrfTokenValue);
      }

      $.ajax({
        url: policyForm.attr('action'),
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        dataType: 'json',
        success: function (response) {
          if (response && response.success) {
            toastr.success(response.message || 'Saved');
            policyModal.hide();
            setTimeout(() => location.reload(), 1000);
            return;
          }
          handlePolicyErrors(response);
        },
        error: function (xhr) {
          toastr.error('Server error occurred. Please try again.');
          if (xhr && xhr.responseText) console.error(xhr.responseText);
        },
      });
    });

    function handlePolicyErrors(response) {
      if (response && response.errors) {
        $.each(response.errors, function (key, value) {
          const input = policyForm.find('[name="' + key + '"]');
          input.addClass('is-invalid');
          input.next('.invalid-feedback').text(value);
        });
      } else {
        toastr.error((response && response.message) || 'An error occurred');
      }
    }
  }

  // Expose legacy global functions used by markup
  function exposeGlobals() {
    window.delete_permission = function (id, name) {
      const ok = confirm('Are you sure want to delete this permission? ' + name);
      if (ok) {
        window.location.href = base_url + 'administrator/settings/delete-permission/' + id;
      }
    };

    window.delete_policy = function (id, name) {
      const ok = confirm('Are you sure want to delete this policy? ' + name);
      if (ok) {
        window.location.href = base_url + 'administrator/settings/delete-policy/' + id;
      }
    };

    window.add_module_section = function () {
      $('#modSectionModal').modal('show');
      if (typeof start_load === 'function') start_load();
      $('.modal-title').text('Create Module Section');
      $('.sbtn').text('Save');
      setTimeout(function () {
        if (typeof end_load === 'function') end_load();
      }, 500);
    };

    window.edit_module_section = function (id) {
      $('#modSectionModal').modal('show');
      $.ajax({
        url: base_url + 'administrator/settings/module-sections/edit/' + id,
        type: 'GET',
        dataType: 'JSON',
        success: function (data) {
          $('[name="id"]').val(data.id);
          $('[name="name"]').val(data.name);
          $('[name="sort_order"]').val(data.sort_order);
          $('[name="icon"]').val(data.icon);
          $('[name="is_active"]').val(data.is_active);
          $('[name="description"]').val(data.description);
          $('form').attr('action', base_url + 'administrator/settings/module-sections/update');

          if (typeof start_load === 'function') start_load();
          $('#modSectionModal').modal('show');
          $('.modal-title').text('Edit Module Section!');
          $('.sbtn').text('Update');
          setTimeout(function () {
            if (typeof end_load === 'function') end_load();
          }, 500);
        },
      });
    };

    window.delete_module_section = function (id, name) {
      const ok = confirm('Are you sure you want to delete ' + name + ' Module?');
      if (ok) {
        window.location.href = base_url + 'administrator/settings/module-sections/delete/' + id;
      }
    };
  }

  // Bootstrap
  $(function () {
    setupAjaxCsrf();
    initTables();
    initCompany();
    initPermissions();
    initPolicies();
    exposeGlobals();
  });
})(jQuery);

