// Banner form enhancements

// Store page URLs metadata for easy access
let pageUrlsData = {};

/**
 * Initializes the banner form with all the dynamic functionality.
 * @param {Object} pageUrls - The page URLs metadata.
 */
function initBannerForm(pageUrls) {
    pageUrlsData = pageUrls.reduce((acc, item) => {
        acc[item.url] = item;
        return acc;
    }, {});

    setupImagePreview();
    setupUrlLinkage();
    setupClearButtons();
}

/**
 * Sets up the image preview functionality.
 */
function setupImagePreview() {
    $('input[name="photo"]').on('change', function(e) {
        handleImageUpload(e, '#bannerImgPreview');
    });
}

/**
 * Handles the image upload, validates it, and creates a preview.
 * @param {Event} event - The file input change event.
 * @param {string} previewSelector - The CSS selector for the preview container.
 */
function handleImageUpload(event, previewSelector) {
    const file = event.target.files[0];
    const previewContainer = $(previewSelector);

    // Clear previous preview and hide clear button
    previewContainer.empty();
    $('#clearImageBtn').hide();

    if (!file) return;

    // Validate file type
    const validTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/tiff', 'image/webp'];
    if (!validTypes.includes(file.type)) {
        toastr.error('Invalid file type. Please upload an image.');
        resetFileInput(event.target);
        return;
    }

    // Validate file size (2MB max)
    const maxSize = 2 * 1024 * 1024; // 2MB
    if (file.size > maxSize) {
        toastr.error(`Image too large (${formatFileSize(file.size)}). Maximum 2MB allowed.`);
        resetFileInput(event.target);
        return;
    }

    // Create and display preview
    const reader = new FileReader();
    reader.onload = function(e) {
        const img = document.createElement('img');
        img.src = e.target.result;
        img.className = 'img-thumbnail thumb';
        img.style.height = '120px';
        img.alt = 'Image preview';
        previewContainer.append(img);
        $('#clearImageBtn').show(); // Show clear button
    };
    reader.readAsDataURL(file);
}

/**
 * Sets up the dynamic field updates when a URL is selected.
 */
function setupUrlLinkage() {
    $('#urlLinkSelector').on('change', function() {
        const selectedUrl = $(this).val();
        const urlInput = $('input[name="url"]');
        
        if (selectedUrl && pageUrlsData[selectedUrl]) {
            const data = pageUrlsData[selectedUrl];
            
            // Update the main URL input
            urlInput.val(data.url);

            // Update and disable related fields
            updateAndDisable('select[name="nav_context"]', data.nav_context);
            updateAndDisable('select[name="service_group_id"]', data.service_group_id);
            
            // Display service group name and make it readonly
            const serviceGroupName = data.service_group_name || (data.service_group_id ? 'N/A' : '');
            const sgInput = $('#service_group_name_display');
            sgInput.val(serviceGroupName).show();
            $('select[name="service_group_id"]').hide();

        } else {
            // If '-- Manual URL --' is selected, clear and enable fields
            urlInput.val('');
            enableAndClear('select[name="nav_context"]');
            enableAndClear('select[name="service_group_id"]');
            
            $('#service_group_name_display').hide();
            $('select[name="service_group_id"]').show();
        }
    });
}

/**
 * Sets up the clear buttons for the form and the image.
 */
function setupClearButtons() {
    // Clear Image Button
    $('#clearImageBtn').on('click', function() {
        const fileInput = $('input[name="photo"]');
        resetFileInput(fileInput[0]);
        $('#bannerImgPreview').empty();
        $(this).hide();
    });

    // Clear Form Button
    $('button[type="reset"]').on('click', function() {
        // Also clear the image preview and file input
        const fileInput = $('input[name="photo"]');
        resetFileInput(fileInput[0]);
        $('#bannerImgPreview').empty();
        $('#clearImageBtn').hide();
        
        // Reset URL-linked fields
        enableAndClear('select[name="nav_context"]');
        enableAndClear('select[name="service_group_id"]');
        $('#service_group_name_display').hide();
        $('select[name="service_group_id"]').show();
    });
}

// Helper functions
function updateAndDisable(selector, value) {
    const field = $(selector);
    if (field.length) {
        field.val(value).prop('disabled', true);
    }
}

function enableAndClear(selector) {
    const field = $(selector);
    if (field.length) {
        field.val('').prop('disabled', false);
    }
}

function resetFileInput(input) {
    $(input).val('');
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}
