<?php

namespace Modules\Posts\Models;

use CodeIgniter\Model;

class PostsModel extends Model
{

  protected $table = 'usc_posts';
    protected $primaryKey = 'id';

    protected $allowedFields = [
        'title', 'alias', 'photo', 'file', 'content', 'excerpt',
        'category_id', 'content_type_id', 'template_id', 'layout_settings',
        'custom_fields', 'author_id', 'menu_order', 'menu_link', 'page_type',
        'is_block', 'status', 'access', 'hits', 'views', 'urls',
        'metadesc', 'metakey', 'published_date', 'created_by', 'updated_at',
        'updated_by', 'nav_context', 'banner_config',
        'template_group', 'slug', 'banner_id', 'service_group_id'
    ];

    protected $returnType     = 'array';
    protected $useTimestamps  = true;
    protected $useSoftDeletes = true;

    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    public function searchPosts($searchTerm)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->where('p.content_type_id', 2);
        if (!empty($searchTerm)) {
          $builder->like('p.title', $searchTerm)
                  ->orWhere('p.alias', $searchTerm)
                  ->orWhere('p.photo', $searchTerm)
                  ->orWhere('p.published_date', $searchTerm)
                  ->orWhere('p.excerpt', $searchTerm)
                  ->orWhere('pc.name', $searchTerm)
                  ->orWhere('pmc.name', $searchTerm)
                  ->groupBy('p.id');
        }
        $query = $builder->get();
        return $query->getResultArray();
    }

    //Get all Blog Posts (content_type_id = 2)
    public function getAll()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->orderBy('p.id', 'DESC');
		  $result = $builder->get()->getResult();
		  return $result;
    }
    public function getSinglePost($id)
    {
      $builder = $this->db->table('usc_posts as p');
    $builder->select('p.*, pc.name as category_name, u.first_name, u.last_name, u.email,
                ct.name as content_type_name,
                pt.name as template_name,
                sg.name as service_group_name,
                b.name as banner_title');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->join('usc_content_types as ct', 'ct.id=p.content_type_id', 'left');
      $builder->join('usc_page_templates as pt', 'pt.id=p.template_id', 'left');
      $builder->join('usc_service_groups as sg', 'sg.id=p.service_group_id', 'left');
      $builder->join('usc_banners as b', 'b.id=p.banner_id', 'left');
      $builder->where('p.id', $id);
      $builder->orWhere('p.alias', $id);
	  $result = $builder->get()->getRow();
	  return $result;
    }
    //Published Cat Posts
    public function getPublishedPosts($id)
    {
      $builder = $this->db->table('usc_posts');
      $builder->where('id', $id);
      $builder->where('content_type_id', 2);
      $builder->where('status', 'Published');
      echo $builder->countAll();
    }



    //Get all POSTS (Blog Posts - content_type_id = 2) for menu
    public function getAllPosts()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.is_block', 'No');
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'published'); // Only show published posts
      $builder->orderBy('p.published_date', 'DESC');
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get all EDITOR's POSTS
    public function getAllEditorPosts()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->join('usc_user_auth_group as uag', 'uag.user_id = u.id', 'left');
      $builder->join('usc_user_groups as ug', 'ug.id = uag.group_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.is_block', 'No');
      $builder->where('p.deleted_at', null);
      $builder->whereIn('uag.group_id', [3, 4]);
      $builder->orderBy('p.published_date', 'DESC');
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get all Block Posts
    public function getAllBlockPosts()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, pc.name as category_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.is_block', 'Yes');
      $builder->where('p.deleted_at', null);
      $builder->orderBy('p.id', 'DESC');
		  $result = $builder->get()->getResult();
		  return $result;
    }

    public function getAllBlockPages()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, pt.name as template_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.created_by', 'left');
      $builder->where('p.content_type_id', 1);
      $builder->where('p.is_block', 'Yes');
      $builder->where('p.deleted_at', null);
      $builder->orderBy('p.id', 'DESC');
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get all Block Posts
    public function getAllEditorBlockPosts()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->join('usc_user_auth_group as uag', 'uag.user_id = u.id', 'left');
      $builder->join('usc_user_groups as ug', 'ug.id = uag.group_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.is_block', 'Yes');
      $builder->where('p.deleted_at', null);
      $builder->whereIn('uag.group_id', [3, 4]);
      $builder->orderBy('p.id', 'DESC');
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get all site posts (blog posts + block posts)
    public function getAllSitePosts()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->orderBy('p.published_date', 'DESC');
		  $result = $builder->get()->getResult();
		  return $result;
    }

    public function getHomeServices()
    {
      $pages = [2, 3, 4, 5, 6, 7];
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, sg.name as service_group_name, sg.slug as service_group_slug, pt.name as template_name, u.first_name, u.last_name');
      $builder->join('usc_service_groups as sg', 'sg.id = p.service_group_id', 'left');
      $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.created_by', 'left');
      $builder->where('p.content_type_id', 3);
      $builder->where('p.deleted_at', null);
      $builder->whereIn('p.id', $pages);
      $builder->orderBy('RAND()');
		  $result = $builder->get()->getResult();
		  return $result;
    }


    //Get all PAGES (content_type_id = 1) and SERVICES (content_type_id = 3) for menu
    public function getAllPages()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pt.name as template_name, u.first_name, u.last_name, u.email');
      $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.created_by', 'left');
      $builder->whereIn('p.content_type_id', [1, 3]); // Include both Pages and Services
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'published'); // Only show published pages
      $builder->orderBy('p.content_type_id', 'ASC'); // Pages first, then Services
      $builder->orderBy('p.title', 'ASC'); // Then alphabetically
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get Latest Posts (Max 5)
    public function getLatestPosts()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->orderBy('p.published_date', 'DESC');
      $builder->limit(5);
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get Popular Posts (Max 5)
    public function getPopularPosts()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->orderBy('p.hits', 'DESC');
      $builder->limit(5);
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get latest Blog Posts for Home
    public function getAllPostsForHome()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->orderBy('p.published_date', 'DESC');
      $builder->limit(3);
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get Latest Blog Posts for News & Blog
	public function getAllPostsForNewsBlog()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->orderBy('p.published_date', 'DESC');
      $builder->limit(3);
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get all published blog posts for homepage/blog listing
    public function forHomePage()
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category_name, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'published');
      $builder->orderBy('p.published_date', 'DESC');
		  $result = $builder->get()->getResultArray();
		  return $result;
    }

    //Get all categories with post counts for filtering
    public function getCategoriesWithPostCounts()
    {
      $builder = $this->db->table('usc_post_category as pc');
      $builder->select('pc.id, pc.name, pc.alias, COUNT(p.id) as post_count');
      $builder->join('usc_posts as p', 'p.category_id = pc.id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'published');
      $builder->groupBy('pc.id');
      $builder->orderBy('pc.name', 'ASC');
      $result = $builder->get()->getResultArray();
      return $result;
    }

    //Get posts by category for filtering
    public function getPostsByCategory($categoryId = null)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category_name, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'published');

      if ($categoryId) {
        $builder->where('p.category_id', $categoryId);
      }

      $builder->orderBy('p.published_date', 'DESC');
      $result = $builder->get()->getResultArray();
      return $result;
    }

    //Get all posts with pagination
    public function getAllPostsPaginated($perPage = 9, $page = 1)
    {
        return $this->select('usc_posts.*, pc.name as category_name, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias')
            ->join('usc_post_category as pc', 'pc.id=usc_posts.category_id', 'left')
            ->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left')
            ->join('usc_users as u', 'u.id=usc_posts.author_id', 'left')
            ->where('usc_posts.content_type_id', 2)
            ->where('usc_posts.deleted_at', null)
            ->where('usc_posts.status', 'published')
            ->orderBy('usc_posts.published_date', 'DESC')
            ->paginate($perPage, 'default', $page);
    }

    //Get posts by category with pagination
    public function getPostsByCategoryPaginated($categoryId, $perPage = 9, $page = 1)
    {
        return $this->select('usc_posts.*, pc.name as category_name, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias')
            ->join('usc_post_category as pc', 'pc.id=usc_posts.category_id', 'left')
            ->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left')
            ->join('usc_users as u', 'u.id=usc_posts.author_id', 'left')
            ->where('usc_posts.content_type_id', 2)
            ->where('usc_posts.deleted_at', null)
            ->where('usc_posts.status', 'published')
            ->where('usc_posts.category_id', $categoryId)
            ->orderBy('usc_posts.published_date', 'DESC')
            ->paginate($perPage, 'default', $page);
    }

    //Frontside of Website Post page
    public function showSinglePost($id)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.id', $id);
      $builder->orWhere('p.alias', $id);
		  $result = $builder->get()->getRow();
		  return $result;
    }

    //Get Related Blog Posts
    public function getRelatedPostsForSinglePage($post)
	{
		$builder = $this->db->table('usc_posts as p');
		$builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
		$builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
		$builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
		$builder->join('usc_users as u', 'u.id=p.author_id', 'left');
		$builder->where('p.content_type_id', 2);
		$builder->where('p.deleted_at', null);
		$builder->where('p.status', 'Published');

		// Group the exclusion conditions properly
		$builder->groupStart()
				->where('p.id !=', $post)
				->where('p.alias !=', $post)
				->groupEnd();

		$builder->orderBy('RAND()');
		$builder->limit(5);
		$result = $builder->get()->getResult();
		return $result;
	}

    //Get single Main Category Posts
    public function getSingleMainCategoryPosts($category, $start, $limit)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->where('pmc.id', $category);
      $builder->orWhere('pmc.alias', $category);
      $builder->orderBy('p.id', 'DESC');
      $builder->limit($limit, $start);
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get single Main Category
    public function getSingleMainCategory($category)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->where('pmc.id', $category);
      $builder->orWhere('pmc.alias', $category);
		  $result = $builder->get()->getRow();
		  return $result;
    }

    //Get Sub Categories with Posts under Main Category
    public function listSubCategoriesInMainCat($category)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->where('pmc.id', $category);
      $builder->orWhere('pmc.alias', $category);
      $builder->groupBy('p.category_id');
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get single Sub Category Posts
    public function getSingleSubCategoryPosts($category, $start, $limit)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->where('pc.id', $category);
      $builder->orWhere('pc.alias', $category);
      $builder->orderBy('p.id', 'DESC');
      $builder->limit($limit, $start);
		  $result = $builder->get()->getResult();
		  return $result;
    }

    //Get single Sub Category
    public function getSingleSubCategory($category)
    {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, pc.name as category, pc.alias as category_alias, u.first_name, u.last_name, pmc.master_category_id, pmc.name as master_category, pmc.alias as master_category_alias');
      $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
      $builder->join('usc_post_category as pmc', 'pmc.id=pc.master_category_id', 'left');
      $builder->join('usc_users as u', 'u.id=p.author_id', 'left');
      $builder->where('p.content_type_id', 2);
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'Published');
      $builder->where('pc.id', $category);
      $builder->orWhere('pc.alias', $category);
		  $result = $builder->get()->getRow();
		  return $result;
    }

    /**
     * Get service cards for home page (Personal Banking & Business Banking)
     */
    public function getServiceCards()
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.id, p.title, p.alias, p.excerpt, p.slug, p.custom_fields, p.nav_context');
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->where('p.access', 'public');
        $builder->whereIn('p.alias', ['personal-banking', 'business-banking']);
        $builder->orderBy('p.id', 'ASC');
        return $builder->get()->getResultArray();
    }

    /**
     * Get featured services for home page (Transfers & Loans)
     */
    public function getFeaturedServices()
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.id, p.title, p.alias, p.excerpt, p.content, p.slug, p.nav_context');
        $builder->join('usc_post_category as pc', 'pc.id=p.category_id', 'left');
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->where('p.access', 'public');
        $builder->where('pc.alias', 'featured-services');
        $builder->orderBy('p.id', 'ASC');
        return $builder->get()->getResultArray();
    }

    /**
     * Get posts by service group and context
     */
    public function getByServiceGroupAndContext($serviceGroup, $context = 'personal')
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.*');
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->where('p.access', 'public');
        $builder->where('p.service_group', $serviceGroup);
        $builder->where('p.nav_context', $context);
        $builder->orderBy('p.id', 'ASC');
        return $builder->get()->getResultArray();
    }

    /**
     * Get posts by navigation context
     */
    public function getByContext($context = 'personal')
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.*');
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->where('p.access', 'public');
        $builder->where('p.nav_context', $context);
        $builder->orderBy('p.service_group', 'ASC');
        $builder->orderBy('p.id', 'ASC');
        return $builder->get()->getResultArray();
    }

    /**
     * Get service counts by context from posts table
     * This provides an alternative counting method using posts instead of banners
     */
    public function getServiceCountsFromPosts($context = 'personal')
    {
        $counts = [
            'current' => 0,
            'saving' => 0,
            'loans' => 0,
            'investing' => 0,
            'insurance' => 0,
            'forex' => 0
        ];

        // Map business service groups to standard names
        $serviceGroupMap = [
            'sme-corporate' => 'current',
            'loans-business' => 'loans',
            'investing-business' => 'investing',
            'insurance-business' => 'insurance'
        ];

        $results = $this->select('service_group, COUNT(*) as count')
                       ->where('deleted_at', null)
                       ->where('status', 'published')
                       ->where('access', 'public')
                       ->where('nav_context', $context)
                       ->whereIn('service_group', ['current', 'saving', 'loans', 'investing', 'insurance', 'forex', 'sme-corporate', 'loans-business', 'investing-business', 'insurance-business'])
                       ->groupBy('service_group')
                       ->findAll();

        foreach ($results as $result) {
            $group = $result['service_group'];

            // Map business groups to standard names
            if (isset($serviceGroupMap[$group])) {
                $group = $serviceGroupMap[$group];
            }

            if (isset($counts[$group])) {
                $counts[$group] += (int)$result['count'];
            }
        }

        return $counts;
    }

    /**
     * Get posts by service group and context with banner data joined
     * This combines posts with their corresponding banners for display
     */
    public function getPostsWithBanners($serviceGroup, $context = 'personal')
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.*, b.name as banner_name, b.title_emphasized, b.photo as banner_photo, b.description as banner_description, b.url as banner_url');
        $builder->join('usc_banners as b', 'b.id = p.banner_id', 'left');
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->where('p.access', 'public');
        $builder->where('p.service_group', $serviceGroup);
        $builder->where('p.nav_context', $context);
        $builder->orderBy('p.id', 'ASC');
        return $builder->get()->getResultArray();
    }

    /**
     * Get floating action buttons for home page
     * Uses existing service_group data to create dynamic buttons
     */
    public function getFloatingButtons()
    {
        // Return empty for now - will be populated from service groups dynamically in view
        // Or use hardcoded structure that references existing pages
        return [];
    }

    /**
     * Get ways to bank cards for home page
     * Returns empty - will use hardcoded banking methods in view
     */
    public function getWaysToBankCards()
    {
        // Return empty - banking methods are standard and don't need database
        return [];
    }

    /**
     * Get posts by service group for dynamic sections
     */
    public function getPostsByServiceGroup($serviceGroup, $limit = null)
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.id, p.title, p.alias, p.excerpt, p.slug, p.custom_fields, p.service_group');
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->where('p.access', 'public');
        $builder->where('p.service_group', $serviceGroup);
        $builder->orderBy('p.id', 'ASC');
        if ($limit) {
            $builder->limit($limit);
        }
        return $builder->get()->getResultArray();
    }

    /**
     * Get posts by content type
     */
    public function getByContentType($contentTypeId, $includeBlocks = false)
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.*, ct.name as content_type_name, ct.slug as content_type_slug, ct.icon as content_type_icon, pc.name as category_name, u.first_name, u.last_name, u.email');
        $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
        $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
        $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
        $builder->where('p.deleted_at', null);
        $builder->where('p.content_type_id', $contentTypeId);

        if (!$includeBlocks) {
            $builder->where('p.is_block !=', 'Yes');
        }

        $builder->orderBy('p.id', 'DESC');
        return $builder->get()->getResult();
    }

    /**
     * Get post with full details including content type and blocks
     */
   public function getPostWithDetails($id)
     {
         $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, ct.slug as content_type_slug, 

                          pt.name as template_name, pt.slug as template_slug,
                          sg.name as service_group_name, pc.name as category_name');
     $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
     $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
     $builder->join('usc_service_groups as sg', 'sg.id = p.service_group_id', 'left');
  $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->where('p.id', $id);
      $builder->where('p.deleted_at', null);
      return $builder->get()->getRow();
  }

  /**
   * Get all pages (content_type_id = 1)
   */
  public function getAllPagesList()
  {
      return $this->getByContentType(1, false);
  }

  /**
   * Get all blog posts (content_type_id = 2)
   */
  public function getAllBlogPosts()
  {
      return $this->getByContentType(2, false);
  }
  /**
   * Get all services (content_type_id = 3)
   */
  public function getAllServices()
  {
      return $this->getByContentType(3, true); // Include blocks for services
  }

  /**
   * Get posts grouped by content type
   */
  public function getGroupedByContentType()
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('ct.id as content_type_id, ct.name as content_type_name, ct.slug as content_type_slug, ct.icon, COUNT(p.id) as post_count');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->where('p.deleted_at', null);
      $builder->groupBy('ct.id');
      $builder->orderBy('ct.menu_position', 'ASC');
      return $builder->get()->getResult();
  }

  /**
   * Search posts with content type filter
   */
  public function searchWithContentType($searchTerm, $contentTypeId = null)
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, ct.slug as content_type_slug, pc.name as category_name, u.first_name, u.last_name');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->where('p.deleted_at', null);
      
      if ($contentTypeId) {
          $builder->where('p.content_type_id', $contentTypeId);
      }
      
      if (!empty($searchTerm)) {
          $builder->groupStart()
                  ->like('p.title', $searchTerm)
                  ->orLike('p.alias', $searchTerm)
                  ->orLike('p.excerpt', $searchTerm)
                  ->orLike('p.content', $searchTerm)
                  ->groupEnd();
      }
      
      $builder->orderBy('p.id', 'DESC');
      return $builder->get()->getResult();
  }



  /**
   * Get all Blog Posts (content_type_id = 2) with full details
   * @return object
   */
  public function getBlogPostsWithDetails()
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, pc.name as category_name, 
                       u.first_name, u.last_name, u.email');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->where('p.content_type_id', 2); // Blog posts only
      $builder->where('p.deleted_at', null);
      $builder->orderBy('p.id', 'DESC');
      return $builder->get()->getResult();
  }

  /**
   * Get all Pages & Services (content_type_id = 1 or 3) with full details
   * @return object
   */
  public function getPagesAndServicesWithDetails()
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, ct.slug as content_type_slug, 
                       pc.name as category_name, u.first_name, u.last_name, u.email,
                       sg.name as service_group_name, pt.name as template_name');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->join('usc_service_groups as sg', 'sg.id = p.service_group_id', 'left');
      $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
      $builder->whereIn('p.content_type_id', [1, 3]); // Pages OR Services
      $builder->where('p.deleted_at', null);
      $builder->orderBy('p.content_type_id', 'ASC');
      $builder->orderBy('p.id', 'DESC');
      return $builder->get()->getResult();
  }

  /**
   * Get single post with all related data
   * @param mixed $id Post ID or alias
   * @return object
   */
  public function getPostWithFullDetails($id)
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, ct.slug as content_type_slug,
                       pc.name as category_name, pc.alias as category_alias,
                       u.first_name, u.last_name, u.email,
                       sg.name as service_group_name, sg.slug as service_group_slug,
                       pt.name as template_name, pt.slug as template_slug,
                       pt.template_file, pt.service_group as template_service_group');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->join('usc_service_groups as sg', 'sg.id = p.service_group_id', 'left');
      $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
      $builder->groupStart()
              ->where('p.id', $id)
              ->orWhere('p.alias', $id)
              ->orWhere('p.slug', $id)
              ->groupEnd();
      $builder->where('p.deleted_at', null);
      return $builder->get()->getRow();
  }

  /**
   * Get block-based posts with blocks count
   * @return object
   */
  public function getBlockBasedPosts()
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, pc.name as category_name,
                       u.first_name, u.last_name,
                       COUNT(pb.id) as blocks_count');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->join('usc_post_blocks as pb', 'pb.post_id = p.id', 'left');
      $builder->where('p.is_block', 'Yes');
      $builder->where('p.deleted_at', null);
      $builder->groupBy('p.id');
      $builder->orderBy('p.id', 'DESC');
      return $builder->get()->getResult();
  }

  /**
   * Find published post by slug
   * @param string $slug
   * @return array|null
   */
  public function findPublishedBySlug($slug)
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*');
      $builder->where('p.deleted_at', null);
      $builder->where('p.status', 'published');
      $builder->where('p.access', 'public');
      $builder->groupStart()
              ->where('p.alias', $slug)
              ->orWhere('p.slug', $slug)
              ->groupEnd();
      return $builder->get()->getRowArray();
  }

  /**
   * Find post with relations (full details)
   * @param int $id
   * @return array|null
   */
  public function findWithRelations($id)
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, ct.name as content_type_name, ct.slug as content_type_slug,
                       pc.name as category_name, pc.alias as category_alias,
                       u.first_name, u.last_name, u.email,
                       sg.name as service_group_name, sg.slug as service_group_slug,
                       pt.name as template_name, pt.slug as template_slug,
                       pt.template_file, pt.service_group as template_service_group');
      $builder->join('usc_content_types as ct', 'ct.id = p.content_type_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_users as u', 'u.id = p.author_id', 'left');
      $builder->join('usc_service_groups as sg', 'sg.id = p.service_group_id', 'left');
      $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
      $builder->where('p.id', $id);
      $builder->where('p.deleted_at', null);
      return $builder->get()->getRowArray();
  }

  /**
   * Get page metadata by alias including tags
   * @param string $alias
   * @return array|null
   */
  // public function getPageMetadataByAlias($alias)
  // {
  //     $builder = $this->db->table('usc_posts as p');
  //     $builder->select('p.id, p.title, p.alias, p.excerpt, p.metadesc, p.metakey, p.content,
  //                      GROUP_CONCAT(DISTINCT t.name SEPARATOR ", ") as tags,
  //                      GROUP_CONCAT(DISTINCT t.alias SEPARATOR ", ") as tag_aliases');
  //     $builder->join('usc_post_tags as pt', 'pt.post_id = p.id', 'left');
  //     $builder->join('usc_tags as t', 't.id = pt.tag_id', 'left');
  //     $builder->where('p.alias', $alias);
  //     $builder->orWhere('p.slug', $alias);
  //     $builder->where('p.deleted_at', null);
  //     $builder->where('p.status', 'published');
  //     $builder->groupBy('p.id');
  //     $result = $builder->get()->getRowArray();

  //     if ($result) {
  //         // Parse custom_fields for tagline if it exists
  //         $customFields = json_decode($result['custom_fields'] ?? '{}', true);
  //         $result['tagline'] = $customFields['tagline'] ?? $result['excerpt'] ?? '';

  //         // Clean up tags
  //         $result['tags_array'] = !empty($result['tags']) ? explode(', ', $result['tags']) : [];
  //         $result['tag_aliases_array'] = !empty($result['tag_aliases']) ? explode(', ', $result['tag_aliases']) : [];
  //     }

  //     return $result;
  // }

  public function getPostWithMetadata($id)
  {
      $builder = $this->db->table('usc_posts as p');
      $builder->select('p.*, 
          GROUP_CONCAT(DISTINCT t.name) as tags,
          GROUP_CONCAT(DISTINCT t.alias) as tag_aliases,
          pc.name as category_name,
          sg.name as service_group_name,
          pt.name as template_name'
      );
      $builder->join('usc_post_tags as ptg', 'ptg.post_id = p.id', 'left');
      $builder->join('usc_tags as t', 't.id = ptg.tag_id', 'left');
      $builder->join('usc_post_category as pc', 'pc.id = p.category_id', 'left');
      $builder->join('usc_service_groups as sg', 'sg.id = p.service_group_id', 'left');
      $builder->join('usc_page_templates as pt', 'pt.id = p.template_id', 'left');
      $builder->where('p.id', $id);
      $builder->orWhere('p.alias', $id);
      $builder->orWhere('p.slug', $id);
      $builder->where('p.deleted_at', null);
      $builder->groupBy('p.id');
      
      $result = $builder->get()->getRowArray();
      
      if ($result) {
          // Process tags into arrays
          $result['tags_array'] = !empty($result['tags']) ? explode(',', $result['tags']) : [];
          $result['tag_aliases_array'] = !empty($result['tag_aliases']) ? explode(',', $result['tag_aliases']) : [];
          
          // Generate dynamic metadata if missing
          $result = $this->generateDynamicMetadata($result);
      }
      
      return $result;
  }

  /**
   * Generate dynamic metadata for posts missing SEO data
   */
  private function generateDynamicMetadata($post)
  {
      // Generate meta description if missing
      if (empty($post['metadesc'])) {
          $post['metadesc'] = $this->generateMetaDescription($post);
      }
      
      // Generate meta keywords if missing
      if (empty($post['metakey'])) {
          $post['metakey'] = $this->generateMetaKeywords($post);
      }
      
      // Generate excerpt if missing
      if (empty($post['excerpt'])) {
          $post['excerpt'] = $this->generateExcerpt($post);
      }
      
      return $post;
  }

  /**
   * Generate meta description from content
   */
  private function generateMetaDescription($post)
  {
      $baseDescription = "ABC Capital Bank Uganda - {$post['title']}";
      
      if (!empty($post['content'])) {
          // Strip HTML tags and get first 160 characters
          $cleanContent = strip_tags($post['content']);
          $excerpt = substr($cleanContent, 0, 157);
          if (strlen($cleanContent) > 157) {
              $excerpt .= '...';
          }
          return $excerpt;
      }
      
      if (!empty($post['excerpt'])) {
          return $post['excerpt'];
      }
      
      return $baseDescription . '. Experience premium banking services with competitive rates and excellent customer support.';
  }

  /**
   * Generate meta keywords from title, tags, and content
   */
  private function generateMetaKeywords($post)
  {
      $keywords = [];
      
      // Add title words
      $titleWords = explode(' ', $post['title']);
      $keywords = array_merge($keywords, array_slice($titleWords, 0, 5));
      
      // Add tags
      if (!empty($post['tags_array'])) {
          $keywords = array_merge($keywords, $post['tags_array']);
      }
      
      // Add service group and context
      if (!empty($post['service_group'])) {
          $keywords[] = $post['service_group'] . ' account';
      }
      
      if (!empty($post['nav_context'])) {
          $keywords[] = $post['nav_context'] . ' banking';
      }
      
      // Add standard banking keywords
      $standardKeywords = [
          'ABC Capital Bank', 'Uganda banking', 'financial solutions',
          'banking services', 'digital banking', 'Uganda'
      ];
      $keywords = array_merge($keywords, $standardKeywords);
      
      // Remove duplicates and limit to 15 keywords
      $keywords = array_unique($keywords);
      $keywords = array_slice($keywords, 0, 15);
      
      return implode(', ', $keywords);
    }

    /**
     * Generate excerpt from content
     */
    private function generateExcerpt($post)
    {
        if (!empty($post['content'])) {
            $cleanContent = strip_tags($post['content']);
            $excerpt = substr($cleanContent, 0, 150);
            if (strlen($cleanContent) > 150) {
                $excerpt .= '...';
            }
            return $excerpt;
        }
        
        return "Discover our {$post['title']} services at ABC Capital Bank Uganda.";
    }

    /**
     * Get page metadata by alias with tags
     */
    public function getPageMetadataByAlias($alias)
    {
        return $this->getPostWithMetadata($alias);
    }

    /**
     * Get posts by tag
     */
    public function getPostsByTag($tagAlias, $limit = null)
    {
        $builder = $this->db->table('usc_posts as p');
        $builder->select('p.*, t.name as tag_name, t.alias as tag_alias');
        $builder->join('usc_post_tags as pt', 'pt.post_id = p.id');
        $builder->join('usc_tags as t', 't.id = pt.tag_id');
        $builder->where('t.alias', $tagAlias);
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->orderBy('p.id', 'DESC');
        
        if ($limit) {
            $builder->limit($limit);
        }
        
        return $builder->get()->getResultArray();
    }

    /**
     * Update post metadata
     */
    public function updatePostMetadata($postId, $metadata)
    {
        $data = [
            'metadesc' => $metadata['metadesc'] ?? null,            'metakey' => $metadata['metakey'] ?? null,
            'excerpt' => $metadata['excerpt'] ?? null,
            'updated_at' => date('Y-m-d H:i:s'),
            'updated_by' => $metadata['updated_by'] ?? null
        ];
        
        return $this->update($postId, $data);
    }

    /**
     * Get popular tags with post counts
     */
    public function getPopularTags($limit = 20)
    {
        $builder = $this->db->table('usc_tags as t');
        $builder->select('t.*, COUNT(pt.id) as post_count');
        $builder->join('usc_post_tags as pt', 'pt.tag_id = t.id');
        $builder->join('usc_posts as p', 'p.id = pt.post_id');
        $builder->where('t.status', '1');
        $builder->where('p.deleted_at', null);
        $builder->where('p.status', 'published');
        $builder->groupBy('t.id');
        $builder->orderBy('post_count', 'DESC');
        $builder->limit($limit);
        
        return $builder->get()->getResultArray();
    }

    /**
     * Add tags to post
     */
    public function addTagsToPost($postId, $tagNames)
    {
        $tagModel = new TagsModel();
        $postTagsModel = new PostTagsModel();
       
        foreach ($tagNames as $tagName) {
            $tagName = trim($tagName);
            if (empty($tagName)) continue;
            
            // Find or create tag
            $tag = $tagModel->where('name', $tagName)->first();
            if (!$tag) {
                $alias = $this->generateAlias($tagName);
                $tagId = $tagModel->insert([
                    'name' => $tagName,
                    'alias' => $alias,
                    'status' => '1'
                ]);
            } else {
                $tagId = $tag['id'];
            }
            
            // Check if tag already associated with post
            $existing = $postTagsModel->where('post_id', $postId)
                                     ->where('tag_id', $tagId)
                                     ->first();
            if (!$existing) {
                $postTagsModel->insert([
                    'post_id' => $postId,
                    'tag_id' => $tagId
                ]);
            }
        }
        
        return true;
    }

    /**
     * Generate URL alias from title
     */
    private function generateAlias($title)
    {
        $alias = strtolower($title);
        $alias = preg_replace('/[^a-z0-9\s-]/', '', $alias);
        $alias = preg_replace('/\s+/', '-', $alias);
        $alias = trim($alias, '-');
        
        return $alias;
    }

}