<?php

namespace Modules\Posts\Controllers;

use App\Controllers\Auth\AuthController;
use Modules\Posts\Models\ServiceGroupsModel;
use Modules\Posts\Models\NavContextsModel;

class ServiceGroups extends AuthController
{
    public function initController($request, $response, $logger)
    {
        parent::initController($request, $response, $logger);
        $this->serviceGroupsModel = new ServiceGroupsModel();
        $this->navContextsModel = new NavContextsModel();
    }

    public function index()
    {
        $this->data['title'] = 'Service Groups';
        $this->data['bcrumb'] = 'Service Groups';
        $this->data['groups'] = $this->serviceGroupsModel->getAll();
        return view('Modules\Posts\Views\service_groups\category', $this->data);
    }

    public function add()
    {
        if ($this->request->getMethod() == 'POST') {
            $name = $this->request->getVar('name');
            if (empty($name)) {
                session()->setTempdata('error', 'Name required', 3);
                return redirect()->back();
            }
            $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($name)));
            $data = [
                'name' => $name,
                'slug' => $slug,
                'description' => $this->request->getVar('description'),
                'is_active' => 1,
                'created_at' => date('Y-m-d H:i:s')
            ];
            if ($this->serviceGroupsModel->insert($data)) {
                $id = $this->serviceGroupsModel->getInsertID();
                $selected = $this->request->getVar('nav_contexts') ?? [];
                $this->serviceGroupsModel->setNavContexts($id, is_array($selected) ? $selected : []);
                session()->setTempdata('success', 'Service Group added', 3);
                return redirect()->to('administrator/service-groups');
            }
        }
        return redirect()->to('administrator/service-groups');
    }

    public function edit($id = null)
    {
        $this->data['title'] = 'Edit Service Group';
        $this->data['bcrumb'] = 'Service Groups';
        $this->data['groups'] = $this->serviceGroupsModel->getAll();
        $this->data['nav_contexts'] = $this->navContextsModel->getAllRaw();
        $this->data['singleGroup'] = $this->serviceGroupsModel->where('id', $id)->first($id);

        // load assigned nav context ids for this service group so the edit form can pre-check them
        $assigned = [];
        if (!empty($id)) {
            $rows = $this->serviceGroupsModel->db->table('usc_service_group_nav_contexts')
                ->select('nav_context_id')
                ->where('service_group_id', (int)$id)
                ->get()
                ->getResultArray();
            $assigned = array_map(function($r){ return (int)$r['nav_context_id']; }, $rows ?: []);
        }
        $this->data['assigned'] = $assigned;
        return view('Modules\Posts\Views\service_groups\category', $this->data);
    }

    public function update()
    {
        if ($this->request->getMethod() == 'POST') {
            $id = $this->request->getVar('id');
            $name = $this->request->getVar('name');
            if (empty($name)) {
                session()->setTempdata('error', 'Name required', 3);
                return redirect()->back();
            }
            $slug = preg_replace('/[^a-z0-9]+/i', '-', trim(strtolower($name)));
            $data = [
                'name' => $name,
                'slug' => $slug,
                'description' => $this->request->getVar('description'),
                'is_active' => $this->request->getVar('is_active') ? 1 : 0,
                'updated_at' => date('Y-m-d H:i:s')
            ];
            if ($this->serviceGroupsModel->update($id, $data)) {
                $selected = $this->request->getVar('nav_contexts') ?? [];
                $this->serviceGroupsModel->setNavContexts($id, is_array($selected) ? $selected : []);
                session()->setTempdata('success', 'Service Group updated', 3);
                return redirect()->to('administrator/service-groups');
            }
        }
        return redirect()->to('administrator/service-groups');
    }

    public function delete($id)
    {
        $this->serviceGroupsModel->where('id', $id)->delete($id);
        session()->setTempdata('success', 'Deleted', 3);
        return redirect()->to('administrator/service-groups');
    }

    /**
     * Test endpoint to check if AJAX is working
     */
    public function test()
    {
        return $this->response->setJSON(['status' => 'success', 'message' => 'AJAX endpoint is working']);
    }

    /**
     * Return JSON list of service groups for a given nav context slug (AJAX)
     */
    public function groupsByNavContext($slug = null)
    {
        // Set JSON response headers first
        $this->response->setHeader('Content-Type', 'application/json');
        
        try {
            log_message('debug', '[ServiceGroups::groupsByNavContext] called with slug: ' . $slug);

            if (empty($slug)) {
                log_message('debug', '[ServiceGroups::groupsByNavContext] empty slug, returning empty array');
                return $this->response->setJSON([]);
            }

            // Debug: Check if nav context exists
            $navContext = $this->navContextsModel->getBySlug($slug);
            log_message('debug', '[ServiceGroups::groupsByNavContext] nav context found: ' . ($navContext ? 'yes' : 'no'));
            
            if (!$navContext) {
                log_message('debug', '[ServiceGroups::groupsByNavContext] nav context not found for slug: ' . $slug);
                return $this->response->setJSON([]);
            }

            $groups = $this->serviceGroupsModel->getByNavContext($slug);
            $count = is_array($groups) ? count($groups) : (is_object($groups) ? count((array)$groups) : 0);
            log_message('debug', '[ServiceGroups::groupsByNavContext] found groups count: ' . $count);
            
            // Debug: Log the actual groups data
            log_message('debug', '[ServiceGroups::groupsByNavContext] groups data: ' . json_encode($groups));

            // normalize to simple array
            $out = [];
            if (is_array($groups) && !empty($groups)) {
                foreach ($groups as $g) {
                    if (is_array($g) && isset($g['id']) && isset($g['name'])) {
                        $out[] = ['id' => (int)$g['id'], 'name' => $g['name']];
                    }
                }
            }

            log_message('debug', '[ServiceGroups::groupsByNavContext] returning: ' . json_encode($out));
            return $this->response->setJSON($out);
            
        } catch (\Exception $e) {
            // Log exception and return empty array to avoid HTML being sent back to AJAX callers
            log_message('error', '[ServiceGroups::groupsByNavContext] Exception: ' . $e->getMessage());
            log_message('error', '[ServiceGroups::groupsByNavContext] Stack trace: ' . $e->getTraceAsString());
            return $this->response->setStatusCode(500)
                                  ->setJSON(['error' => true, 'message' => 'Server error: ' . $e->getMessage()]);
        }
    }
}
