<?php

namespace Modules\Posts\Controllers;

use App\Controllers\Auth\AuthController;
use Modules\Posts\Models\BlockDefinitionsModel;
use Modules\Posts\Models\PostBlocksModel;
use Modules\Posts\Models\PostBlockItemsModel;
use Modules\Posts\Models\PostsModel;
use Modules\Posts\Models\ServiceGroupsModel;
use Modules\Posts\Models\ItemTypeDefinitionsModel;
use Modules\Posts\Models\BlockItemTypeRelationshipsModel;


class BlocksController extends AuthController
{
      public function initController($request, $response, $logger)
  {
    parent::initController($request, $response, $logger);
    $this->db = \Config\Database::connect();
    $this->blockdefinitionsModel = new BlockDefinitionsModel();
    $this->postBlocksModel = new PostBlocksModel();
    $this->postBlockItemsModel = new PostBlockItemsModel();
    $this->postsModel = new PostsModel();
    $this->servicegroupsModel = new ServiceGroupsModel();
    $this->itemtypedefinitionsModel = new ItemTypeDefinitionsModel();
    $this->blockitemtyperelationshipsModel = new BlockItemTypeRelationshipsModel();
   
  }

    /**
     * Display blocks management page
     */
    public function index()
    {
        $this->data['title'] = "Blocks Management";
        $this->data['bcrumb'] = "Blocks";
        
        // Handle active tab from query parameter
        $activeTab = $this->request->getGet('tab') ?? 'definitions';
        $this->data['activeTab'] = $activeTab;

        // Get all block definitions
        $this->data['blockDefinitions'] = $this->blockdefinitionsModel->findAll();
        
        // Get all post blocks with post info and items count
        $builder = $this->db->table('usc_post_blocks as pb');
        $builder->select('pb.*, p.title as post_title, COUNT(pbi.id) as items_count');
        $builder->join('usc_posts as p', 'p.id = pb.post_id', 'left');
        $builder->join('usc_post_block_items as pbi', 'pbi.block_id = pb.id', 'left');
        $builder->where('pb.deleted_at', null);
        $builder->groupBy('pb.id');
        $builder->orderBy('pb.post_id', 'DESC');
        $this->data['postBlocks'] = $builder->get()->getResult();
        
        // Get all block items
        $this->data['blockItems'] = $this->postBlockItemsModel
                                         ->where('deleted_at', null)
                                         ->orderBy('block_id', 'DESC')
                                         ->findAll();
        
        // Get all pages and services for filter
        $this->data['allPosts'] = $this->postsModel
                                      ->where('is_block', 'Yes')
                                      ->orWhere('content_type_id', 1) // Include pages
                                      ->orWhere('content_type_id', 3) // Include services
                                      ->findAll();

        // Get service groups for block definition categories
        $this->data['serviceGroups'] = $this->servicegroupsModel
                                           ->where('deleted_at', null)
                                           ->findAll();

        // Get all available item types
        $this->data['availableItemTypes'] = $this->itemtypedefinitionsModel->getActiveItemTypes();
        
        // Get all item type definitions for the table
        $this->data['itemTypeDefinitions'] = $this->itemtypedefinitionsModel->findAll();

        return view('Modules\Posts\Views\blocks\index', $this->data);
    }

    /**
     * Get active block definitions
     */
    public function getActiveDefinitions()
    {
        $definitions = $this->blockdefinitionsModel->getActiveBlocks();
        return $this->respond(['success' => true, 'data' => $definitions]);
    }

    /**
     * Get single block definition
     */
    public function getBlockDefinition($id)
    {
        $definition = $this->blockdefinitionsModel->find($id);
        if (!$definition) {
            return $this->respond(['success' => false, 'message' => 'Definition not found'], 404);
        }
        // Get associated item types
        $definition['item_types'] = $this->blockitemtyperelationshipsModel
                                        ->where('block_definition_id', $id)
                                        ->findAll();

        return $this->respond(['success' => true, 'data' => $definition]);
    }

    /**
     * Add new block definition
     */
    public function addBlockDefinition()
    {
        $rules = [
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[block_definitions.slug]'],
        ];

        if (!$this->validate($rules)) {
            if ($this->request->isAJAX()) {
                return $this->respond([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $this->validator->getErrors()
                ], 400);
            } else {
                session()->setTempdata('error', 'Validation failed', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        $data = $this->request->getPost();
        $itemTypes = $data['item_types'] ?? [];
        unset($data['item_types']);
        $data['created_by'] = $this->userdata->user_id;

        try {
            if ($this->blockdefinitionsModel->save($data)) {
                $blockDefinitionId = $this->blockdefinitionsModel->getInsertID();
                foreach ($itemTypes as $itemTypeId) {
                    $this->blockitemtyperelationshipsModel->insert([
                        'block_definition_id' => $blockDefinitionId,
                        'item_type_definition_id' => $itemTypeId,
                        'is_default' => 0,
                        'max_items' => 0,
                        'ordering' => 0
                    ]);
                }

                if ($this->request->isAJAX()) {
                    return $this->respond(['success' => true, 'message' => 'Block definition added successfully']);
                } else {
                    session()->setTempdata('success', 'Block definition added successfully', 3);
                    return redirect()->to('administrator/posts/blocks');
                }
            }

            if ($this->request->isAJAX()) {
                return $this->respond(['success' => false, 'message' => 'Failed to add block definition'], 500);
            } else {
                session()->setTempdata('error', 'Failed to add block definition', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        } catch (\Exception $e) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
            } else {
                session()->setTempdata('error', 'Error adding block definition', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }
    }

    /**
     * Update block definition
     */
    public function updateBlockDefinition()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required'],
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[block_definitions.slug,id,{id}]'],
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        $blockDefinitionId = $data['id'];
        $itemTypes = $data['item_types'] ?? [];
        unset($data['item_types']);
        
        if ($this->blockdefinitionsModel->save($data)) {
            // Delete existing relationships
            $this->blockitemtyperelationshipsModel->where('block_definition_id', $blockDefinitionId)->delete();

            // Insert new relationships
            foreach ($itemTypes as $itemTypeId) {
                $this->blockitemtyperelationshipsModel->insert([
                    'block_definition_id' => $blockDefinitionId,
                    'item_type_definition_id' => $itemTypeId,
                    'is_default' => 0, // Default value, adjust as needed
                    'max_items' => 0, // Default value, adjust as needed
                    'ordering' => 0 // Default value, adjust as needed
                ]);
            }
            return $this->respond(['success' => true, 'message' => 'Block definition updated successfully']);
        }

        return $this->respond(['success' => false, 'message' => 'Failed to update block definition'], 500);
    }

    /**
     * Delete block definition
     */
    public function deleteBlockDefinition($id)
    {
        if ($this->userdata->group_id != 1) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => false, 'message' => 'Access denied'], 403);
            } else {
                session()->setFlashdata('warning', 'Access denied');
                return redirect()->to(previous_url());
            }
        }

        // Check if any blocks are using this definition
        $definition = $this->blockdefinitionsModel->find($id);
        if ($definition) {
            $blocksUsingDefinition = $this->postBlocksModel
                                        ->where('block_type', $definition['slug'])
                                        ->countAllResults();
            
            if ($blocksUsingDefinition > 0) {
                if ($this->request->isAJAX()) {
                    return $this->respond([
                        'success' => false, 
                        'message' => 'Cannot delete definition. There are ' . $blocksUsingDefinition . ' blocks using this definition.'
                    ], 400);
                } else {
                    session()->setTempdata('error', 'Cannot delete definition. There are blocks using this definition.', 3);
                    return redirect()->to('administrator/posts/blocks');
                }
            }
        }

        if ($this->blockdefinitionsModel->delete($id)) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Block definition deleted']);
            } else {
                session()->setTempdata('success', 'Block definition deleted', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to delete block definition'], 500);
        } else {
            session()->setTempdata('error', 'Failed to delete block definition', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Add new block
     */
    public function addBlock()
    {
        $rules = [
            'post_id' => ['label' => 'Post', 'rules' => 'required|is_not_unique[usc_posts.id]'],
            'definition_id' => ['label' => 'Definition', 'rules' => 'required|is_not_unique[block_definitions.id]']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        // Get block definition
        $definition = $this->blockdefinitionsModel->find($data['definition_id']);
        if (!$definition) {
            return $this->respond(['success' => false, 'message' => 'Block definition not found'], 404);
        }
        
        // Check max instances if set
        if ($definition['max_instances'] && $definition['max_instances'] > 0) {
            $currentCount = $this->postBlocksModel
                                ->where('post_id', $data['post_id'])
                                ->where('block_type', $definition['slug'])
                                ->countAllResults();
            
            if ($currentCount >= $definition['max_instances']) {
                return $this->respond([
                    'success' => false, 
                    'message' => 'Maximum instances (' . $definition['max_instances'] . ') reached for this block type'
                ], 400);
            }
        }
        
        // Prepare block data
        $blockData = [
            'post_id' => $data['post_id'],
            'block_type' => $definition['slug'],
            'block_cat' => $data['block_cat'] ?? 'content',
            'heading' => $data['heading'] ?? null,
            'ordering' => $data['ordering'] ?? 0,
            'status' => $data['status'] ?? 'active'
        ];

        try {
            if ($this->postBlocksModel->save($blockData)) {
                return $this->respond([
                    'success' => true, 
                    'message' => 'Block added successfully',
                    'blockId' => $this->postBlocksModel->getInsertID()
                ]);
            }

            return $this->respond([
                'success' => false,
                'message' => 'Failed to add block: ' . implode(', ', $this->postBlocksModel->errors())
            ], 500);
        } catch (\Exception $e) {
            return $this->respond([
                'success' => false,
                'message' => 'Error adding block: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get single block
     */
    public function getBlock($id)
    {
        $block = $this->postBlocksModel->find($id);
        if (!$block) {
            return $this->respond(['success' => false, 'message' => 'Block not found'], 404);
        }
        return $this->respond(['success' => true, 'data' => $block]);
    }

    /**
     * View block details
     */
    public function viewBlock($id)
    {
        // Get block with extended info
        $builder = $this->db->table('usc_post_blocks as pb');
        $builder->select('pb.*, p.title as post_title, bd.name as definition_name, bd.description as definition_description');
        $builder->join('usc_posts as p', 'p.id = pb.post_id', 'left');
        $builder->join('block_definitions as bd', 'bd.slug = pb.block_type', 'left');
        $builder->where('pb.id', $id);
        $block = $builder->get()->getRow();

        if (!$block) {
            return $this->respond(['success' => false, 'message' => 'Block not found'], 404);
        }

        // Get block items
        $items = $this->postBlockItemsModel
            ->where('block_id', $id)
            ->where('deleted_at', null)
            ->orderBy('ordering', 'ASC')
            ->findAll();

        return $this->respond([
            'success' => true,
            'data' => [
                'block' => $block,
                'items' => $items
            ]
        ]);
    }

    
    /**
     * Update block
     */
    public function updateBlock()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_blocks.id]']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        // First check if block exists
        $existingBlock = $this->postBlocksModel->find($data['id']);
        if (!$existingBlock) {
            return $this->respond([
                'success' => false,
                'message' => 'Block not found'
            ], 404);
        }

        try {
            if ($this->postBlocksModel->save($data)) {
                return $this->respond([
                    'success' => true, 
                    'message' => 'Block updated successfully'
                ]);
            }

            return $this->respond([
                'success' => false,
                'message' => 'Failed to update block: ' . implode(', ', $this->postBlocksModel->errors())
            ], 500);
        } catch (\Exception $e) {
            return $this->respond([
                'success' => false,
                'message' => 'Error updating block: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update block order
     */
    public function updateBlockOrder()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_blocks.id]'],
            'ordering' => ['label' => 'Order', 'rules' => 'required|numeric']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        try {
            $this->db->transBegin();
            
            if ($this->postBlocksModel->updateWithOrdering($data['id'], $data)) {
                $this->db->transCommit();
                return $this->respond(['success' => true, 'message' => 'Block order updated successfully']);
            }

            $this->db->transRollback();
            return $this->respond(['success' => false, 'message' => 'Failed to update block order']);
        } catch (\Exception $e) {
            $this->db->transRollback();
            return $this->respond(['success' => false, 'message' => 'Error updating block order']);
        }
    }

    /**
     * Delete block
     */
    public function deleteBlock($id)
    {
        // Delete associated items first
        $this->postBlockItemsModel->where('block_id', $id)->delete();
        
        if ($this->postBlocksModel->delete($id)) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Block deleted']);
            } else {
                session()->setTempdata('success', 'Block deleted', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to delete block'], 500);
        } else {
            session()->setTempdata('error', 'Failed to delete block', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Get block items for a specific block
     */
    public function getBlockItems($blockId)
    {
        $items = $this->postBlockItemsModel
                     ->where('block_id', $blockId)
                     ->where('deleted_at', null)
                     ->orderBy('ordering', 'ASC')
                     ->findAll();
        
        if (!$items) {
            return $this->respond(['success' => false, 'message' => 'No items found for this block'], 404);
        }
        
        // Generate HTML for the items list
        $html = '<div class="table-responsive"><table class="table table-sm">';
        $html .= '<thead><tr><th>Title</th><th>Type</th><th>Order</th><th>Status</th><th>Actions</th></tr></thead>';
        $html .= '<tbody>';
        
        foreach ($items as $item) {
            $html .= '<tr>';
            $html .= '<td>' . esc($item['title'] ?? 'N/A') . '</td>';
            $html .= '<td><code>' . $item['item_type'] . '</code></td>';
            $html .= '<td><span class="badge bg-primary">' . $item['ordering'] . '</span></td>';
            $html .= '<td>' . ($item['status'] == 'active' ? '<span class="badge bg-success">Active</span>' : '<span class="badge bg-secondary">Inactive</span>') . '</td>';
            $html .= '<td>';
            $html .= '<button class="btn btn-sm btn-outline-primary edit-item" data-id="' . $item['id'] . '"><i class="bi bi-pencil"></i></button> ';
            $html .= '<button class="btn btn-sm btn-outline-danger delete-item" data-id="' . $item['id'] . '"><i class="bi bi-trash"></i></button>';
            $html .= '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table></div>';
        
        return $this->respond(['success' => true, 'html' => $html]);
    }

    /**
     * Get single block item
     */
    public function getBlockItem($id)
    {
        $item = $this->postBlockItemsModel->find($id);
        if (!$item) {
            return $this->respond(['success' => false, 'message' => 'Item not found'], 404);
        }
        return $this->respond(['success' => true, 'data' => $item]);
    }

    /**
     * Add new block item
     */
    public function addBlockItem()
    {
        $rules = [
            'block_id' => ['label' => 'Block', 'rules' => 'required|is_not_unique[usc_post_blocks.id]'],
            'title' => ['label' => 'Title', 'rules' => 'required']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        // Get block to check if it allows items
        $block = $this->postBlocksModel->find($data['block_id']);
        if (!$block) {
            return $this->respond(['success' => false, 'message' => 'Block not found'], 404);
        }
        
        $definition = $this->blockdefinitionsModel->where('slug', $block['block_type'])->first();
        
        if (!$definition || !$definition['is_repeatable']) {
            return $this->respond(['success' => false, 'message' => 'This block type does not support items'], 400);
        }
        
        // Set default values
        $data['item_type'] = $data['item_type'] ?? 'text';
        $data['ordering'] = $data['ordering'] ?? 0;
        $data['status'] = $data['status'] ?? 'active';
        
        if ($this->postBlockItemsModel->save($data)) {
            return $this->respond(['success' => true, 'message' => 'Item added successfully']);
        }

        return $this->respond(['success' => false, 'message' => 'Failed to add item'], 500);
    }

    /**
     * Update block item
     */
    public function updateBlockItem()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_block_items.id]'],
            'block_id' => ['label' => 'Block', 'rules' => 'required|is_not_unique[usc_post_blocks.id]'],
            'title' => ['label' => 'Title', 'rules' => 'required']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        if ($this->postBlockItemsModel->save($data)) {
            return $this->respond(['success' => true, 'message' => 'Item updated successfully']);
        }

        return $this->respond(['success' => false, 'message' => 'Failed to update item'], 500);
    }

    /**
     * Update block item order
     */
    public function updateItemOrder()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required|is_not_unique[usc_post_block_items.id]'],
            'ordering' => ['label' => 'Order', 'rules' => 'required|numeric']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        if ($this->postBlockItemsModel->save($data)) {
            return $this->respond(['success' => true, 'message' => 'Item order updated successfully']);
        }

        return $this->respond(['success' => false, 'message' => 'Failed to update item order'], 500);
    }

    /**
     * Delete block item
     */
    public function deleteBlockItem($id)
    {
        if ($this->postBlockItemsModel->delete($id)) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Item deleted']);
            } else {
                session()->setTempdata('success', 'Block item deleted', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to delete item'], 500);
        } else {
            session()->setTempdata('error', 'Failed to delete block item', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Get single item type definition
     */
    public function getItemType($id)
    {
        $itemType = $this->itemtypedefinitionsModel->find($id);
        if (!$itemType) {
            return $this->respond(['success' => false, 'message' => 'Item type not found'], 404);
        }
        return $this->respond(['success' => true, 'data' => $itemType]);
    }

    /**
     * View item type definition details
     */
    public function viewItemType($id)
    {
        $itemType = $this->itemtypedefinitionsModel->find($id);
        if (!$itemType) {
            return $this->response->setJSON(['success' => false, 'message' => 'Item type not found'])->setStatusCode(404);
        }

        // Get usage statistics
        $usageCount = $this->postBlockItemsModel
                          ->where('item_type', $itemType['slug'])
                          ->countAllResults();

        // Get associated block definitions
        $associatedBlocks = $this->blockitemtyperelationshipsModel
                                ->select('bd.name, bd.slug')
                                ->join('block_definitions as bd', 'bd.id = block_item_type_relationships.block_definition_id')
                                ->where('item_type_definition_id', $id)
                                ->findAll();

        $data = [
            'itemType' => $itemType,
            'usageCount' => $usageCount,
            'associatedBlocks' => $associatedBlocks
        ];

        return $this->response->setJSON(['success' => true, 'data' => $data]);
    }

    /**
     * Add new item type
     */
    public function addItemType()
    {
        $rules = [
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[usc_item_type_definitions.slug]']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        $data['created_by'] = $this->userdata->user_id;
        
        if ($this->itemtypedefinitionsModel->save($data)) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Item type added successfully']);
            } else {
                session()->setTempdata('success', 'Item type added successfully', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to add item type'], 500);
        } else {
            session()->setTempdata('error', 'Failed to add item type', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Update item type
     */
    public function updateItemType()
    {
        $rules = [
            'id' => ['label' => 'ID', 'rules' => 'required'],
            'name' => ['label' => 'Name', 'rules' => 'required'],
            'slug' => ['label' => 'Slug', 'rules' => 'required|is_unique[usc_item_type_definitions.slug,id,{id}]']
        ];

        if (!$this->validate($rules)) {
            return $this->respond([
                'success' => false, 
                'message' => 'Validation failed',
                'errors' => $this->validator->getErrors()
            ], 400);
        }

        $data = $this->request->getPost();
        
        if ($this->itemtypedefinitionsModel->save($data)) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Item type updated successfully']);
            } else {
                session()->setTempdata('success', 'Item type updated successfully', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to update item type'], 500);
        } else {
            session()->setTempdata('error', 'Failed to update item type', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }

    /**
     * Delete item type
     */
    public function deleteItemType($id)
    {
        // First get the item type to get its slug
        $itemType = $this->itemtypedefinitionsModel->find($id);
        if (!$itemType) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => false, 'message' => 'Item type not found'], 404);
            } else {
                session()->setTempdata('error', 'Item type not found', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        // Check if any blocks are using this item type's slug
        $itemsUsingType = $this->postBlockItemsModel
                              ->where('item_type', $itemType['slug'])
                              ->countAllResults();
        
        if ($itemsUsingType > 0) {
            if ($this->request->isAJAX()) {
                return $this->respond([
                    'success' => false, 
                    'message' => 'Cannot delete item type. There are ' . $itemsUsingType . ' items using this type.'
                ], 400);
            } else {
                session()->setTempdata('error', 'Cannot delete item type. There are items using this type.', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->itemtypedefinitionsModel->delete($id)) {
            if ($this->request->isAJAX()) {
                return $this->respond(['success' => true, 'message' => 'Item type deleted successfully']);
            } else {
                session()->setTempdata('success', 'Item type deleted successfully', 3);
                return redirect()->to('administrator/posts/blocks');
            }
        }

        if ($this->request->isAJAX()) {
            return $this->respond(['success' => false, 'message' => 'Failed to delete item type'], 500);
        } else {
            session()->setTempdata('error', 'Failed to delete item type', 3);
            return redirect()->to('administrator/posts/blocks');
        }
    }
}