<?php

namespace Modules\Jobs\Controllers;

use App\Controllers\Auth\AuthController;
use Modules\Jobs\Models\JobsModel;
use Modules\Jobs\Models\JobCategoryModel;

class Jobs extends AuthController
{
    public $jobsModel;
    public $jobCategoryModel;

    public function initController($request, $response, $logger)
    {
        parent::initController($request, $response, $logger);
        $this->jobsModel = new JobsModel();
        $this->jobCategoryModel = new JobCategoryModel();
    }

    public function index()
    {
        $this->data['title'] = "Jobs Dashboard";
        $this->data['bcrumb'] = "All Jobs";
        $this->data['jobs'] = $this->jobsModel->getAll();

        return view('Modules\Jobs\Views\all_jobs', $this->data);
    }

    public function addJob()
    {
        $this->data['title'] = "Add Job";
        $this->data['bcrumb'] = "Add Job";
        $this->data['categories'] = $this->jobCategoryModel->getForDropdown();

        if ($this->request->getMethod() == 'POST') {
            $rules = [
                'title' => ['label' => 'Job Title', 'rules' => 'required'],
                'deadline_date' => ['label' => 'Deadline Date', 'rules' => 'required'],
            ];
            
            $photoRules = [
                'photo' => [
                    'label' => 'Image File',
                    'rules' => 'is_image[photo]'
                        . '|mime_in[photo,image/jpg,image/jpeg,image/gif,image/png,image/webp]'
                        . '|max_size[photo,2024]'
                ],
            ];
            
            $fileRules = [
                'file' => [
                    'label' => 'Document File',
                    'rules' => 'max_size[file,4024]'
                ],
            ];

            if ($this->validate($rules)) {
                $slug = $this->jobsModel->generateSlug($this->request->getVar('title'));

                $postData = [
                    'title' => $this->request->getVar("title"),
                    'alias' => $slug,
                    'description' => $this->request->getVar("description"),
                    'deadline_date' => $this->request->getVar("deadline_date"),
                    'positions' => $this->request->getVar("positions"),
                    'job_type' => $this->request->getVar("job_type"),
                    'status' => $this->request->getVar("status"),
                    "urls" => $this->request->getVar("urls"),
                    'phone' => $this->request->getVar("phone"),
                    'email' => $this->request->getVar("email"),
                    'address' => $this->request->getVar("address"),
                    'category_id' => $this->request->getVar("category_id"),
                    'created_by' => $this->userdata->user_id,
                ];

                // Handle document upload
                $docFile = $this->request->getFile('file');
                if ($docFile && $docFile->isValid() && !$docFile->hasMoved()) {
                    if ($this->validate($fileRules)) {
                        $docFileName = $docFile->getRandomName();
                        $docFile->move(ROOTPATH . 'media/downloads/documents', $docFileName);
                        $postData['file'] = $docFileName;
                    } else {
                        $this->data['validation'] = $this->validator;
                        return view('Modules\Jobs\Views\add_job', $this->data);
                    }
                }

                // Handle photo upload
                $photoFile = $this->request->getFile('photo');
                if ($photoFile && $photoFile->isValid() && !$photoFile->hasMoved()) {
                    if ($this->validate($photoRules)) {
                        $photoFileName = $photoFile->getRandomName();
                        $photoFile->move(ROOTPATH . 'media/downloads/images', $photoFileName);
                        $postData['photo'] = $photoFileName;
                    } else {
                        $this->data['validation'] = $this->validator;
                        return view('Modules\Jobs\Views\add_job', $this->data);
                    }
                }

                $job = $this->jobsModel->save($postData);

                if ($job) {
                    $this->sendAdminNotifications('New Job: ' . $this->request->getVar("title") . ' CREATED', 'content', $this->userdata->user_id);
                    session()->setTempdata('success', 'Job posted successfully', 3);
                    return redirect()->to('administrator/jobs/dashboard');
                } else {
                    session()->setTempdata('error', 'Job NOT posted', 3);
                    return redirect()->to('administrator/jobs/add');
                }

            } else {
                $this->data['validation'] = $this->validator;
            }
        }

        return view('Modules\Jobs\Views\add_job', $this->data);
    }

    public function editJob($id = null)
    {
        $this->data['title'] = "Edit Job Post";
        $this->data['bcrumb'] = "Edit Job";
        $this->data['singleJob'] = $this->jobsModel->getSingleJob($id);
        $this->data['categories'] = $this->jobCategoryModel->getForDropdown();

        if (!$this->data['singleJob']) {
            session()->setTempdata('error', 'Job not found', 3);
            return redirect()->to('administrator/jobs/dashboard');
        }

        return view('Modules\Jobs\Views\edit_job', $this->data);
    }

    public function updateJob()
    {
        $id = $this->request->getVar('id');
        $this->data['title'] = "Edit Job Post";
        $this->data['bcrumb'] = "Edit Job Post";
        $this->data['singleJob'] = $this->jobsModel->getSingleJob($id);
        $this->data['categories'] = $this->jobCategoryModel->getForDropdown();

        if (!$this->data['singleJob']) {
            session()->setTempdata('error', 'Job not found', 3);
            return redirect()->to('administrator/jobs/dashboard');
        }

        if ($this->request->getMethod() == 'POST') {
            $rules = [
                'title' => ['label' => 'Job Title', 'rules' => 'required'],
                'deadline_date' => ['label' => 'Deadline Date', 'rules' => 'required'],
            ];

            $photoRules = [
                'photo' => [
                    'label' => 'Image File',
                    'rules' => 'is_image[photo]'
                        . '|mime_in[photo,image/jpg,image/jpeg,image/gif,image/png,image/webp]'
                        . '|max_size[photo,2024]'
                ],
            ];

            $fileRules = [
                'file' => [
                    'label' => 'Document File',
                    'rules' => 'max_size[file,4024]'
                ],
            ];

            if ($this->validate($rules)) {
                $slug = $this->jobsModel->generateSlug($this->request->getVar('title'));

                $postData = [
                    'title' => $this->request->getVar('title'),
                    'alias' => $slug,
                    'description' => $this->request->getVar('description'),
                    'deadline_date' => $this->request->getVar('deadline_date'),
                    'positions' => $this->request->getVar('positions'),
                    'job_type' => $this->request->getVar('job_type'),
                    'status' => $this->request->getVar('status'),
                    'urls' => $this->request->getVar('urls'),
                    'phone' => $this->request->getVar('phone'),
                    'email' => $this->request->getVar('email'),
                    'address' => $this->request->getVar('address'),
                    'category_id' => $this->request->getVar('category_id'),
                    'updated_by' => $this->userdata->user_id,
                    'updated_at' => date('Y-m-d H:i:s'),
                ];

                // Handle document upload
                $docFile = $this->request->getFile('file');
                if ($docFile && $docFile->isValid() && !$docFile->hasMoved()) {
                    if ($this->validate($fileRules)) {
                        $docFileName = $docFile->getRandomName();
                        $docFile->move(ROOTPATH . 'media/downloads/documents', $docFileName);
                        $postData['file'] = $docFileName;
                    } else {
                        $this->data['validation'] = $this->validator;
                        return view('Modules\Jobs\Views\edit_job', $this->data);
                    }
                }

                // Handle photo upload
                $photoFile = $this->request->getFile('photo');
                if ($photoFile && $photoFile->isValid() && !$photoFile->hasMoved()) {
                    if ($this->validate($photoRules)) {
                        $photoFileName = $photoFile->getRandomName();
                        $photoFile->move(ROOTPATH . 'media/downloads/images', $photoFileName);
                        $postData['photo'] = $photoFileName;
                    } else {
                        $this->data['validation'] = $this->validator;
                        return view('Modules\Jobs\Views\edit_job', $this->data);
                    }
                }

                if ($this->jobsModel->update($id, $postData)) {
                    $this->sendAdminNotifications('Job Updated: ' . $this->request->getVar('title'), 'content', $this->userdata->user_id);
                    session()->setTempdata('success', 'Job updated successfully', 3);
                    return redirect()->to('administrator/jobs/dashboard');
                } else {
                    session()->setTempdata('error', 'Job NOT updated', 3);
                    return redirect()->to('administrator/jobs/edit/' . $id);
                }

            } else {
                $this->data['validation'] = $this->validator;
            }
        }

        return view('Modules\Jobs\Views\edit_job', $this->data);
    }

    public function delJob($id = null)
    {
        $job = $this->jobsModel->find($id);

        if (!$job) {
            session()->setTempdata('error', 'Job not found', 3);
            return redirect()->to('administrator/jobs/dashboard');
        }

        if ($this->jobsModel->delete($id)) {
            $this->sendAdminNotifications('Job Deleted: ' . $job['title'], 'content', $this->userdata->user_id);
            session()->setTempdata('success', 'Job deleted successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to delete job', 3);
        }

        return redirect()->to('administrator/jobs/dashboard');
    }

    // Categories management
    public function categories()
    {
        $this->data['title'] = "Job Categories";
        $this->data['bcrumb'] = "Job Categories";
        $this->data['categories'] = $this->jobCategoryModel->getAll();

        return view('Modules\Jobs\Views\all_categories', $this->data);
    }

    public function addCategory()
    {
        $this->data['title'] = "Add Job Category";
        $this->data['bcrumb'] = "Add Job Category";

        if ($this->request->getMethod() == 'POST') {
            $rules = [
                'name' => ['label' => 'Category Name', 'rules' => 'required'],
            ];

            if ($this->validate($rules)) {
                $slug = $this->jobCategoryModel->generateSlug($this->request->getVar('name'));

                $postData = [
                    'name' => $this->request->getVar('name'),
                    'alias' => $slug,
                    'parent_id' => $this->request->getVar('parent_id') ?: null,
                ];

                if ($this->jobCategoryModel->save($postData)) {
                    session()->setTempdata('success', 'Category added successfully', 3);
                    return redirect()->to('administrator/jobs/categories');
                } else {
                    session()->setTempdata('error', 'Category NOT added', 3);
                }
            } else {
                $this->data['validation'] = $this->validator;
            }
        }

        $this->data['categories'] = $this->jobCategoryModel->getAll();
        return view('Modules\Jobs\Views\add_category', $this->data);
    }

    public function editCategory($id = null)
    {
        $this->data['title'] = "Edit Job Category";
        $this->data['bcrumb'] = "Edit Job Category";
        $this->data['category'] = $this->jobCategoryModel->getCategory($id);
        $this->data['categories'] = $this->jobCategoryModel->getAll();

        if (!$this->data['category']) {
            session()->setTempdata('error', 'Category not found', 3);
            return redirect()->to('administrator/jobs/categories');
        }

        return view('Modules\Jobs\Views\edit_category', $this->data);
    }

    public function updateCategory()
    {
        $id = $this->request->getVar('id');
        $this->data['title'] = "Edit Job Category";
        $this->data['bcrumb'] = "Edit Job Category";
        $this->data['category'] = $this->jobCategoryModel->getCategory($id);
        $this->data['categories'] = $this->jobCategoryModel->getAll();

        if (!$this->data['category']) {
            session()->setTempdata('error', 'Category not found', 3);
            return redirect()->to('administrator/jobs/categories');
        }

        if ($this->request->getMethod() == 'POST') {
            $rules = [
                'name' => ['label' => 'Category Name', 'rules' => 'required'],
            ];

            if ($this->validate($rules)) {
                $slug = $this->jobCategoryModel->generateSlug($this->request->getVar('name'));

                $postData = [
                    'name' => $this->request->getVar('name'),
                    'alias' => $slug,
                    'parent_id' => $this->request->getVar('parent_id') ?: null,
                ];

                if ($this->jobCategoryModel->update($id, $postData)) {
                    session()->setTempdata('success', 'Category updated successfully', 3);
                    return redirect()->to('administrator/jobs/categories');
                } else {
                    session()->setTempdata('error', 'Category NOT updated', 3);
                }
            } else {
                $this->data['validation'] = $this->validator;
            }
        }

        return view('Modules\Jobs\Views\edit_category', $this->data);
    }

    public function deleteCategory($id = null)
    {
        $category = $this->jobCategoryModel->getCategory($id);

        if (!$category) {
            session()->setTempdata('error', 'Category not found', 3);
            return redirect()->to('administrator/jobs/categories');
        }

        if ($this->jobCategoryModel->delete($id)) {
            session()->setTempdata('success', 'Category deleted successfully', 3);
        } else {
            session()->setTempdata('error', 'Failed to delete category', 3);
        }

        return redirect()->to('administrator/jobs/categories');
    }
}