<?php

namespace Modules\ForexExchange\Controllers;

use App\Controllers\BaseController;
use Modules\ForexExchange\Models\ForexExchangeModel;

class ForexExchange extends BaseController
{
    public $forexexchangeModel;
    public $userdata;

    public function initController(\CodeIgniter\HTTP\RequestInterface $request, \CodeIgniter\HTTP\ResponseInterface $response, \Psr\Log\LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);

        // Check if user is logged in
        if (!session()->get('loggedIn')) {
            return redirect()->to('back-pass/login');
        }

        // Initialize forex exchange model
        $this->forexexchangeModel = new ForexExchangeModel();
        
        // Set user data
        $logged_user = session()->get('logged_user');
        if ($logged_user) {
            $this->userdata = $this->authloginModel->getUserDetails($logged_user);
            $this->data['userdata'] = $this->userdata;
        } else {
            return redirect()->to('back-pass/login');
        }
    }

    public function index()
    {		
        // display header information
        $this->data['title'] = "Forex Exchange Rates";
        $this->data['bcrumb'] = "Forex Rates";

        // Get all countries with currencies for the selector
        $this->data['currencies'] = $this->countryModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        // Get forex rates with country details (both active and inactive)
        $this->data['forex_rates'] = $this->forexexchangeModel->getAll();
        
        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    // AJAX endpoint for currency search
    public function searchCurrencies()
    {
        try {
            $search = $this->request->getGet('search');
            $id = $this->request->getGet('id');
            $builder = $this->countryModel->builder();

            // If specific ID is requested, return that currency only
            if (!empty($id) && is_numeric($id)) {
                $builder->where('id', $id);
            } elseif (!empty($search)) {
                $builder->groupStart()
                       ->like('name', $search)
                       ->orLike('currency_code', $search)
                       ->orLike('currency_symbol', $search)
                       ->orLike('sortname', $search)
                       ->orLike('currency_name', $search)
                       ->groupEnd();
            }

            $builder->where('status', 1)
                   ->where('currency_code IS NOT NULL')
                   ->where('currency_code !=', '')
                   ->orderBy('name', 'ASC');

            // Only limit if not searching by specific ID
            if (empty($id)) {
                $builder->limit(50); // Limit results for better performance
            }

            $countries = $builder->get()->getResult();
            return $this->response->setJSON(['success' => true, 'data' => $countries]);
        } catch (\Exception $e) {
            log_message('error', 'Currency search error: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error fetching currencies'
            ])->setStatusCode(500);
        }
    }

    public function getCurrency($id)
    {
        try {
            if (!is_numeric($id)) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Invalid currency ID'
                ])->setStatusCode(400);
            }

            $country = $this->countryModel->where('id', $id)
                                        ->where('status', 1)
                                        ->where('currency_code IS NOT NULL')
                                        ->where('currency_code !=', '')
                                        ->first();

            if (!$country) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Currency not found'
                ])->setStatusCode(404);
            }

            return $this->response->setJSON([
                'success' => true,
                'data' => $country
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Get currency error: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error fetching currency details'
            ])->setStatusCode(500);
        }
    }

    public function addForex()
    {
        $this->data['currencies'] = $this->countryModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        if($this->request->getMethod() == 'POST'){
            $rules = [
                'currency_id' => ['label'=>'Currency', 'rules'=>'required'],
                'buying' => ['label'=>'Buying', 'rules'=>'required'],
                'selling' => ['label'=>'Selling', 'rules'=>'required'],
            ];

            if($this->validate($rules))
            {
                $forexData = [
                    'currency_id'   => $this->request->getVar("currency_id"),
                    'buying'        => $this->request->getVar("buying"),
                    'selling'       => $this->request->getVar("selling"),
                    'tt_buying'     => $this->request->getVar("tt_buying"),
                    'tt_selling'    => $this->request->getVar("tt_selling"),
                    'status'        => $this->request->getVar("status") ? 1 : 0,
                    'created_by'    => $this->userdata ? $this->userdata->user_id : session()->get('logged_user'),
                ];

                if($this->forexexchangeModel->save($forexData))
                {
                    $admins = $this->usersModel->getAdminUsers();
                    $currencyId = $this->request->getVar("currency_id");
                    $currency = $this->countryModel->where('id', $currencyId)->first();

                    $message = $currency['name'].' ('.$currency['currency_code'].') Forex Rate ADDED';
                    foreach($admins as $m)
                    {
                        $userz = [
                            'message' => $message,
                            'user_id' => $m->id,
                            'notification_type' => 'content',
                            'created_by' => $this->userdata->user_id,
                        ];
                        $this->notificationsModel->save($userz);
                    }
                    session()->setTempdata('success', 'Forex Rate added successfully', 3);
                    return redirect()->to('administrator/forex/rates');
                } else {
                    session()->setTempdata('error', 'Forex Rate NOT added', 3);
                    return redirect()->to('administrator/forex/rates');
                }
            } else {
                $this->data['validation'] = $this->validator;
            }
        }
        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    public function editForex($id = null)
    {
        $this->data['title'] = "Update Forex Exchange Rates";
        $this->data['bcrumb'] = "Update Forex Rates";

        $this->data['currencies'] = $this->countryModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        // Get forex rates with country details
        $this->data['forex_rates'] = $this->forexexchangeModel->getAll();

        // Get single rate with country details
        $this->data['singlerate'] = $this->forexexchangeModel->getRate($id);

        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    public function updateForex()
    {
        $this->data['title'] = "Update Forex Exchange Rates";
        $this->data['bcrumb'] = "Update Forex Rates";

        $this->data['currencies'] = $this->countryModel
            ->where('status', 1)
            ->where('currency_code IS NOT NULL')
            ->where('currency_code !=', '')
            ->findAll();

        if($this->request->getMethod() == 'POST'){
            $rules = [
                'currency_id' => ['label'=>'Currency', 'rules'=>'required'],
                'buying' => ['label'=>'Buying', 'rules'=>'required'],
                'selling' => ['label'=>'Selling', 'rules'=>'required'],
            ];

            if($this->validate($rules))
            {
                $id = $this->request->getVar("id");
                $forexData = [
                    'currency_id'   => $this->request->getVar("currency_id"),
                    'buying'        => $this->request->getVar("buying"),
                    'selling'       => $this->request->getVar("selling"),
                    'tt_buying'     => $this->request->getVar("tt_buying"),
                    'tt_selling'    => $this->request->getVar("tt_selling"),
                    'status'        => $this->request->getVar("status") ? 1 : 0,
                    'updated_by'    => $this->userdata ? $this->userdata->user_id : session()->get('logged_user'),
                    'updated_at'    => date('Y-m-d h:i:s'),
                ];

                if($this->forexexchangeModel->update($id, $forexData))
                {
                    $admins = $this->usersModel->getAdminUsers();
                    $currencyId = $this->request->getVar("currency_id");
                    $currency = $this->countryModel->where('id', $currencyId)->first();

                    $message = $currency['name'].' ('.$currency['currency_code'].') Forex Rate UPDATED';
                    foreach($admins as $m)
                    {
                        $userz = [
                            'message' => $message,
                            'user_id' => $m->id,
                            'notification_type' => 'content',
                            'created_by' => $this->userdata->user_id,
                        ];
                        $this->notificationsModel->save($userz);
                    }

                    session()->setTempdata('success', 'Forex Rate updated successfully', 3);
                    return redirect()->to('administrator/forex/rates');
                } else {
                    session()->setTempdata('error', 'Forex Rate NOT updated', 3);
                    return redirect()->to('administrator/forex/rates');
                }
            } else {
                $this->data['validation'] = $this->validator;
            }
        }
        return view('Modules\ForexExchange\Views\forex_rates', $this->data);
    }

    public function delForex($id = null)
    {
        $frate = $this->forexexchangeModel->where('id', $id)->delete($id);
        if($frate)
        {
            session()->setTempdata('success', 'Forex Rate deleted successfully', 3);
            return redirect()->to('administrator/forex/rates');
        }
    }
}