<?php

namespace Modules\Banners\Models;

use CodeIgniter\Model;

class BannersModel extends Model
{
    protected $table = 'usc_banners';
    protected $primaryKey = 'id';

    protected $allowedFields = [
        'name',
        'title_emphasized',
        'photo',
        'description',
        'url',
        'status',
        'created_at',
        'updated_at',
        'deleted_at',
        'service_group',
        'nav_context',
        'banner_type',
        'template_group',
        'service_group_id'
    ];

    protected $returnType     = 'array';
    protected $useTimestamps  = true;
    protected $useSoftDeletes = true;

    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    // Get all active banners with service group information
    public function getAll()
    {
        $builder = $this->db->table($this->table . ' as b');
        $builder->select('b.*, sg.name as service_group_name, nc.name as nav_context_name')
               ->join('usc_service_groups as sg', 'b.service_group_id = sg.id', 'left')
               ->join('usc_nav_contexts as nc', 'b.nav_context = nc.slug', 'left')
               ->where('b.deleted_at', null)
               ->where('b.status', '1')
               ->orderBy('b.id', 'asc');

        $banners = $builder->get()->getResultArray();
        return self::formatBannersData($banners);
    }

    // // Get banners by type (hero, promotional, featured, slider)
    // public function getByType($type)
    // {
    //     $builder = $this->db->table($this->table . ' as b');
    //     $builder->select('b.*, sg.name as service_group_name, nc.name as nav_context_name')
    //            ->join('usc_service_groups as sg', 'b.service_group_id = sg.id', 'left')
    //            ->join('usc_nav_contexts as nc', 'b.nav_context = nc.slug', 'left')
    //            ->where('b.deleted_at', null)
    //            ->where('b.banner_type', $type)
    //            ->orderBy('b.id', 'asc');

    //     $banners = $builder->get()->getResultArray();
    //     return self::formatBannersData($banners);
    // }

    // Get single banner by ID with service group information
    public function singleBanner($id)
    {
        $builder = $this->db->table($this->table . ' as b');
        $builder->select('b.*, sg.name as service_group_name, nc.name as nav_context_name')
               ->join('usc_service_groups as sg', 'b.service_group_id = sg.id', 'left')
               ->join('usc_nav_contexts as nc', 'b.nav_context = nc.slug', 'left')
               ->where('b.id', $id)
               ->where('b.deleted_at', null);

        $banner = $builder->get()->getRowArray();
        return self::formatBannerData($banner);
    }

    // // Get banners for homepage slider
    // public function getHomeSlider()
    // {
    //     $banners = $this->where('deleted_at', null)
    //                ->where('status', '1')
    //                ->where('banner_type', 'slider')
    //                ->where('nav_context', 'personal')
    //                ->orderBy('id', 'asc')
    //                ->findAll();

    //     return self::formatBannersData($banners);
    // }

    // // Get hero banners
    // public function getHeroBanners()
    // {
    //     $banners = $this->where('deleted_at', null)
    //                ->where('status', '1')
    //                ->where('banner_type', 'hero')
    //                ->orderBy('id', 'asc')
    //                ->findAll();

    //     return self::formatBannersData($banners);
    // }

    // // Get promotional banners
    // public function getPromotionalBanners()
    // {
    //     $banners = $this->where('deleted_at', null)
    //                ->where('status', '1')
    //                ->where('banner_type', 'promotional')
    //                ->orderBy('id', 'asc')
    //                ->findAll();

    //     return self::formatBannersData($banners);
    // }

    // // Get featured banners
    // public function getFeaturedBanners()
    // {
    //     $banners = $this->where('deleted_at', null)
    //                ->where('status', '1')
    //                ->where('banner_type', 'featured')
    //                ->orderBy('id', 'asc')
    //                ->findAll();

    //     return self::formatBannersData($banners);
    // }

    // // Get banners by context (personal/business)
    // public function getByContext($context = 'personal')
    // {
    //     $banners = $this->where('deleted_at', null)
    //                ->where('status', '1')
    //                ->where('nav_context', $context)
    //                ->orderBy('id', 'asc')
    //                ->findAll();

    //     return self::formatBannersData($banners);
    // }

    // // Get banners by navigation context and service group with dynamic mapping
    // public function getByContextAndServiceGroup($context, $serviceGroup)
    // {
    //     $builder = $this->db->table($this->table . ' as b');
    //     $builder->select('b.*, sg.name as service_group_name, nc.name as nav_context_name')
    //             ->join('usc_service_groups as sg', 'b.service_group_id = sg.id', 'left')
    //             ->join('usc_nav_contexts as nc', 'b.nav_context = nc.slug', 'left')
    //             ->where('b.deleted_at', null)
    //             ->where('b.status', '1')
    //             ->where('nc.slug', $context)
    //             ->where('sg.slug', $serviceGroup)
    //             ->orderBy('b.id', 'asc');
        
    //     $banners = $builder->get()->getResultArray();
    //     return self::formatBannersData($banners);
    // }

    // // Get banners with emphasized titles
    // public function getBannersWithEmphasis($context = 'personal')
    // {
    //     return $this->select('*, CASE WHEN title_emphasized IS NOT NULL AND title_emphasized != "" THEN title_emphasized ELSE name END as display_title, CASE WHEN title_emphasized IS NOT NULL AND title_emphasized != "" THEN REPLACE(name, title_emphasized, "") ELSE "" END as remaining_title')
    //                ->where('deleted_at', null)
    //                ->where('status', '1')
    //                ->where('nav_context', $context)
    //                ->orderBy('id', 'asc')
    //                ->findAll();
    // }

    // // Get single banner with emphasized title
    // public function getBannerWithEmphasis($id)
    // {
    //     return $this->select('*, CASE WHEN title_emphasized IS NOT NULL AND title_emphasized != "" THEN title_emphasized ELSE name END as display_title, CASE WHEN title_emphasized IS NOT NULL AND title_emphasized != "" THEN REPLACE(name, title_emphasized, "") ELSE "" END as remaining_title')
    //                ->where('id', $id)
    //                ->where('deleted_at', null)
    //                ->first();
    // }

    // // Helper method to format title with emphasis
    // public function formatTitleWithEmphasis($banner)
    // {
    //     // Handle both objects and arrays
    //     $titleEmphasized = is_object($banner) ? ($banner->title_emphasized ?? '') : ($banner['title_emphasized'] ?? '');
    //     $name = is_object($banner) ? ($banner->name ?? '') : ($banner['name'] ?? '');
        
    //     if (empty($titleEmphasized) || $titleEmphasized == '') {
    //         return esc($name);
    //     }

    //     $remaining = str_replace($titleEmphasized, '', $name);
    //     return esc($remaining) . '<span>' . esc($titleEmphasized) . '</span>';
    // }



 public function getPageHeroBannerByUrl($url)
    {
        return $this->where('url', $url)
                   ->where('banner_type', 'hero')

                   ->first();
    }






    // Get overview image banner by context
    public function getOverviewImage($context = 'personal')
    {
        return $this->where('deleted_at', null)
                   ->where('status', '1')
                   ->where('nav_context', $context)
                   ->where('banner_type', 'promotional')
                   ->where('template_group', 'overview_image')
                   ->first();
    }

    // Get service counts by context
    public function getServiceCounts($context = 'personal')
    {
        $counts = [
            'current' => 0,
            'saving' => 0,
            'loans' => 0,
            'investing' => 0,
            'insurance' => 0
        ];

        // Map business service groups to standard names
        $serviceGroupMap = [
            'sme-corporate' => 'current',
            'loans-business' => 'loans',
            'investing-business' => 'investing',
            'insurance-business' => 'insurance'
        ];

        $results = $this->select('service_group, COUNT(*) as count')
                       ->where('deleted_at', null)
                       ->where('status', '1')
                       ->where('nav_context', $context)
                       ->whereIn('service_group', ['current', 'saving', 'loans', 'investing', 'insurance', 'sme-corporate', 'loans-business', 'investing-business', 'insurance-business'])
                       ->groupBy('service_group')
                       ->findAll();

        foreach ($results as $result) {
            $group = $result['service_group'];
            
            // Map business groups to standard names
            if (isset($serviceGroupMap[$group])) {
                $group = $serviceGroupMap[$group];
            }
            
            if (isset($counts[$group])) {
                $counts[$group] += (int)$result['count'];
            }
        }

        return $counts;
    }

    /**
     * Get the full path for a banner image
     * All images are stored in media/files/ folder
     * 
     * @param string $filename The filename of the banner image
     * @return string The full URL to the banner image
     */
    public static function getBannerImagePath($filename)
    {
        if (empty($filename)) {
            return '';
        }

        // If the stored value already contains media/files/, use it as is
        if (strpos($filename, 'media/files/') === 0) {
            return base_url($filename);
        }
        
        // Otherwise, add the media/files/ prefix
        return base_url('media/files/' . $filename);
    }

    /**
     * Format banner data with proper image paths
     * 
     * @param array $banner Banner data array
     * @return array Banner data with formatted photo path
     */
    public static function formatBannerData($banner)
    {
        if (!empty($banner)) {
            // Handle both objects and arrays
            if (is_object($banner)) {
                if (isset($banner->photo)) {
                    $banner->photo = self::getBannerImagePath($banner->photo);
                }
            } elseif (is_array($banner) && isset($banner['photo'])) {
                $banner['photo'] = self::getBannerImagePath($banner['photo']);
            }
        }
        return $banner;
    }

    /**
     * Format multiple banners with proper image paths
     * 
     * @param array $banners Array of banner data
     * @return array Banners with formatted photo paths
     */
    public static function formatBannersData($banners)
    {
        if (empty($banners)) {
            return $banners;
        }
        
        return array_map(function($banner) {
            return self::formatBannerData($banner);
        }, $banners);
    }

    /**
     * Get banner for a specific post/service by post_id
     * 
     * @param int $postId The post ID
     * @return array|null Banner data or null if not found
     */
    public function getPostBanner($postId)
    {
        $banner = $this->where('post_id', $postId)
                       ->where('deleted_at', null)
                       ->where('status', '1')
                       ->first();
        
        if (empty($banner)) {
            return null;
        }
        
        return self::formatBannerData($banner);
    }

    /**
     * Get service groups by navigation context
     * 
     * @param string $navContextSlug The navigation context slug
     * @return array Array of service groups
     */
    public function getServiceGroupsByContext($navContextSlug)
    {
        $builder = $this->db->table('usc_service_groups sg');
        $builder->select('sg.id, sg.name, sg.slug')
                ->join('usc_service_group_nav_contexts sgnc', 'sg.id = sgnc.service_group_id')
                ->join('usc_nav_contexts nc', 'sgnc.nav_context_id = nc.id')
                ->where('nc.slug', $navContextSlug)
                ->where('sg.is_active', 1)
                ->orderBy('sg.name', 'asc');

        return $builder->get()->getResultArray();
    }

    /**
     * Get available page/post/service URLs with metadata for admin selects
     * Returns array of objects with keys: url, title, id, type, nav_context, service_group_id, service_group_name, template_id, template_name, template_group
     */
    public function getPageUrls()
    {
        $postsModel = new \Modules\Posts\Models\PostsModel();

        $items = [];

        $push = function($p, $type) use (&$items) {
            $menu_link = $p->menu_link ?? ($p->url ?? null);
            if (empty($menu_link)) return;

            $items[$menu_link] = [
                'url' => $menu_link,
                'title' => $p->title ?? $p->name ?? $menu_link,
                'id' => $p->id ?? null,
                'type' => $type,
                'nav_context' => $p->nav_context ?? null,
                'service_group_id' => $p->service_group_id ?? ($p->service_group ?? null),
                'service_group_name' => $p->service_group_name ?? null,
                'template_id' => $p->template_id ?? null,
                'template_name' => $p->template_name ?? null,
                'template_group' => $p->template_group ?? null,
            ];
        };

        $pages = $postsModel->getAllPagesList();
        foreach ($pages as $page) {
            $push($page, 'page');
        }

        $posts = $postsModel->getAllBlogPosts();
        foreach ($posts as $post) {
            $push($post, 'post');
        }

        $services = $postsModel->getAllServices();
        foreach ($services as $service) {
            $push($service, 'service');
        }

        $results = array_values($items);
        usort($results, function($a, $b) {
            return strcasecmp($a['title'] ?? $a['url'], $b['title'] ?? $b['url']);
        });

        return $results;
    }

    /**
     * Search banners by a given term.
     *
     * @param string $term The search term.
     * @return array An array of formatted banner data.
     */
    public function search($term)
    {
        $builder = $this->db->table($this->table . ' as b');
        $builder->select('b.*, sg.name as service_group_name, nc.name as nav_context_name')
               ->join('usc_service_groups as sg', 'b.service_group_id = sg.id', 'left')
               ->join('usc_nav_contexts as nc', 'b.nav_context = nc.slug', 'left')
               ->where('b.deleted_at', null)
               ->groupStart()
               ->like('b.name', $term)
               ->orLike('b.title_emphasized', $term)
               ->orLike('b.description', $term)
               ->orLike('b.url', $term)
               ->orLike('b.banner_type', $term)
               ->orLike('b.nav_context', $term)
               ->groupEnd()
               ->orderBy('b.id', 'desc');

        $banners = $builder->get()->getResultArray();
        return self::formatBannersData($banners);
    }
}