<?php

namespace Modules\Banners\Controllers;

use App\Controllers\BaseController;
use Modules\Banners\Models\BannersModel;

class Banners extends BaseController
{
    protected $data = [];

    public function index()
    {
        $this->data['title'] = 'Banner Management';
         $this->data['bcrumb'] = "Banners";

        $search = $this->request->getGet('search');

        if ($search) {
            $banners = $this->bannersModel->search($search);
        } else {
            $banners = $this->bannersModel->orderBy('id', 'desc')->findAll();
        }

        $this->data['banners'] = $banners; // Remove formatBannersData call to keep raw data
        // Provide available page/post/service urls with metadata for the view (no API required)
        $this->data['pageUrls'] = $this->bannersModel->getPageUrls();

        return view('Modules\\Banners\\Views\\index', $this->data);
    }

    public function list()
    {
        $type = $this->request->getGet('type');
        
        if (!$type) {
            return redirect()->to('administrator/banners')->with('error', 'Banner type is required');
        }

        $this->data['title'] = ucfirst($type) . ' Banners';
        $this->data['banners'] = $this->bannersModel->getByType($type);
        // Provide available page/post/service urls with metadata for the view (no API required)
        $this->data['pageUrls'] = $this->bannersModel->getPageUrls();
        return view('Modules\Banners\Views\list', $this->data);
    }

    public function new()
    {
        return redirect()->to('administrator/banners');
    }

    public function create()
    {
        // Handle file upload
        $photo = $this->request->getFile('photo');
        $newName = null;

        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            $newName = $photo->getRandomName();
            $photo->move(ROOTPATH . 'media/files', $newName);
        }

        // Resolve canonical URL: prefer menu_link > slug > alias when a page/post was selected
        $postedUrl = $this->request->getPost('url');
        $selectedId = $this->request->getPost('selected_link_id');
        $resolvedUrl = $postedUrl;
        $postsModel = new \Modules\Posts\Models\PostsModel();
        if (!empty($selectedId)) {
            $sel = $postsModel->getPostWithFullDetails($selectedId);
            if (!$sel) {
                $sel = $postsModel->getSinglePost($selectedId);
            }
            if ($sel) {
                $resolvedUrl = $sel->menu_link ?? ($sel->slug ?? ($sel->alias ?? $postedUrl));
            }
        } else {
            // try match by menu_link
            $found = $postsModel->where('menu_link', $postedUrl)->first();
            if ($found) {
                $resolvedUrl = $found['menu_link'] ?? $postedUrl;
            } else {
                // try find by slug/alias
                $single = $postsModel->getPostWithFullDetails($postedUrl);
                if ($single) {
                    $resolvedUrl = $single->menu_link ?? ($single->slug ?? ($single->alias ?? $postedUrl));
                }
            }
        }

        $data = [
            'name' => $this->request->getPost('name'),
            'title_emphasized' => $this->request->getPost('title_emphasized'),
            'description' => $this->request->getPost('description'),
            'url' => $resolvedUrl,
            'status' => $this->request->getPost('status'),
            'service_group' => $this->request->getPost('service_group'),
            'nav_context' => $this->request->getPost('nav_context'),
            'banner_type' => $this->request->getPost('banner_type'),
            'template_group' => $this->request->getPost('template_group'),
            'service_group_id' => $this->request->getPost('service_group_id'),
            'photo' => $newName
        ];

        if ($this->bannersModel->save($data)) {
            return redirect()->to('administrator/banners')->with('success', 'Banner created successfully');
        }

        return redirect()->back()->withInput()->with('error', 'Failed to create banner');
    }

    public function edit($id)
    {
        // Fetch all data for the index view
        $this->data['title'] = 'Banner Management';
        $banners = $this->bannersModel->orderBy('id', 'desc')->findAll();
        $this->data['banners'] = $banners; // Remove formatBannersData call to keep raw data
        $this->data['pageUrls'] = $this->bannersModel->getPageUrls();

        // Fetch the specific banner for editing
        $this->data['banner'] = $this->bannersModel->singleBanner($id);
        if (!$this->data['banner']) {
            return redirect()->back()->with('error', 'Banner not found');
        }
        
        $this->data['serviceGroups'] = $this->bannersModel->getServiceGroupsByContext($this->data['banner']['nav_context']);

        return view('Modules\\Banners\\Views\\index', $this->data);
    }

    public function getServiceGroups($context)
    {
        $groups = $this->bannersModel->getServiceGroupsByContext($context);
        return $this->response->setJSON([
            'status' => 'success',
            'data' => $groups
        ]);
    }

    public function update($id)
    {
        $banner = $this->bannersModel->find($id);
        if (!$banner) {
            return redirect()->back()->with('error', 'Banner not found');
        }

        $photo = $this->request->getFile('photo');
        $newName = $banner['photo']; // Keep existing photo by default

        if ($photo && $photo->isValid() && !$photo->hasMoved()) {
            // Delete old photo if exists
            if ($banner['photo'] && file_exists(ROOTPATH . 'media/files/' . $banner['photo'])) {
                unlink(ROOTPATH . 'media/files/' . $banner['photo']);
            }
            $newName = $photo->getRandomName();
            $photo->move(ROOTPATH . 'media/files', $newName);
        }

        // Resolve canonical URL on update (menu_link > slug > alias)
        $postedUrl = $this->request->getPost('url');
        $selectedId = $this->request->getPost('selected_link_id');
        $resolvedUrl = $postedUrl;
        $postsModel = new \Modules\Posts\Models\PostsModel();
        if (!empty($selectedId)) {
            $sel = $postsModel->getPostWithFullDetails($selectedId);
            if (!$sel) {
                $sel = $postsModel->getSinglePost($selectedId);
            }
            if ($sel) {
                $resolvedUrl = $sel->menu_link ?? ($sel->slug ?? ($sel->alias ?? $postedUrl));
            }
        } else {
            $found = $postsModel->where('menu_link', $postedUrl)->first();
            if ($found) {
                $resolvedUrl = $found['menu_link'] ?? $postedUrl;
            } else {
                $single = $postsModel->getPostWithFullDetails($postedUrl);
                if ($single) {
                    $resolvedUrl = $single->menu_link ?? ($single->slug ?? ($single->alias ?? $postedUrl));
                }
            }
        }

        $data = [
            'id' => $id,
            'name' => $this->request->getPost('name'),
            'title_emphasized' => $this->request->getPost('title_emphasized'),
            'description' => $this->request->getPost('description'),
            'url' => $resolvedUrl,
            'status' => $this->request->getPost('status'),
            'service_group' => $this->request->getPost('service_group'),
            'nav_context' => $this->request->getPost('nav_context'),
            'banner_type' => $this->request->getPost('banner_type'),
            'template_group' => $this->request->getPost('template_group'),
            'service_group_id' => $this->request->getPost('service_group_id'),
            'photo' => $newName
        ];

        if ($this->bannersModel->save($data)) {
            return redirect()->to('administrator/banners')->with('success', 'Banner updated successfully');
        }

        return redirect()->back()->withInput()->with('error', 'Failed to update banner');
    }

    public function delete($id)
    {
        $banner = $this->bannersModel->find($id);
        if (!$banner) {
            return redirect()->to('administrator/banners')->with('error', 'Banner not found');
        }

        // Delete photo file if exists
        if ($banner['photo'] && file_exists(ROOTPATH . 'media/files/' . $banner['photo'])) {
            unlink(ROOTPATH . 'media/files/' . $banner['photo']);
        }

        if ($this->bannersModel->delete($id)) {
            return redirect()->to('administrator/banners')->with('success', 'Banner deleted successfully');
        }

        return redirect()->to('administrator/banners')->with('error', 'Failed to delete banner');
    }

    // API methods for frontend
    public function getHeroBanners()
    {
        $banners = $this->bannersModel->getHeroBanners();
        return $this->response->setJSON($banners);
    }

    public function getPromotionalBanners()
    {
        $banners = $this->bannersModel->getPromotionalBanners();
        return $this->response->setJSON($banners);
    }

    public function getFeaturedBanners()
    {
        $banners = $this->bannersModel->getFeaturedBanners();
        return $this->response->setJSON($banners);
    }

    public function getByContext($context = 'personal')
    {
        // Validate context using NavContextsModel
        $navContextsModel = new \Modules\Posts\Models\NavContextsModel();
        $validContext = $navContextsModel->getBySlug($context);
        
        if (!$validContext) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Invalid navigation context'
            ])->setStatusCode(400);
        }

        $banners = $this->bannersModel->getByContext($context);
        return $this->response->setJSON(['status' => 'success', 'data' => $banners]);
    }

    public function getByServiceGroup($serviceGroup)
    {
        // Validate service group using ServiceGroupsModel
        $serviceGroupsModel = new \Modules\Posts\Models\ServiceGroupsModel();
        $validGroup = $serviceGroupsModel->getBySlug($serviceGroup);
        
        if (!$validGroup) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Invalid service group'
            ])->setStatusCode(400);
        }

        $banners = $this->bannersModel->getByServiceGroup($serviceGroup);
        return $this->response->setJSON(['status' => 'success', 'data' => $banners]);
    }

    public function getByContextAndServiceGroup($context, $serviceGroup)
    {
        // Validate both context and service group
        $navContextsModel = new \Modules\Posts\Models\NavContextsModel();
        $serviceGroupsModel = new \Modules\Posts\Models\ServiceGroupsModel();
        
        $validContext = $navContextsModel->getBySlug($context);
        $validGroup = $serviceGroupsModel->getBySlug($serviceGroup);
        
        if (!$validContext || !$validGroup) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Invalid navigation context or service group'
            ])->setStatusCode(400);
        }

        $banners = $this->bannersModel->getByContextAndServiceGroup($context, $serviceGroup);
        return $this->response->setJSON(['status' => 'success', 'data' => $banners]);
    }
}
