<?php

use App\Models\AuthLoginModel;
use Modules\Settings\Models\PermissionsModel;
use Modules\Settings\Models\UserGroupPermissionsModel;


if (!function_exists('has_permission')) {
    /**
     * Check if current user has a specific permission
     *
     * @param string $permission The permission name (e.g. 'Members.index')
     * @return bool
     */
    function has_permission(string $permission): bool
    {
        $permissionKey = trim($permission);
        
        // Get user ID from session
        $userId = session('user_id') ?: null;
        
        // If no user_id in session, try to get from logged_user session
        if (empty($userId) && session('logged_user')) {
            try {
                $auth = new AuthLoginModel();
                $details = $auth->getUserDetails(session('logged_user'));
                if ($details && isset($details->user_id)) {
                    $userId = $details->user_id;
                    // Cache group_id for this request
                    $groupId = $details->group_id;
                }
            } catch (\Throwable $e) {
                return false;
            }
        }

        if (empty($userId)) {
            return false;
        }

        // Get group_id if not already set
        if (empty($groupId)) {
            $db = \Config\Database::connect();
            $result = $db->query("
                SELECT group_id FROM usc_user_auth_group 
                WHERE user_id = ? 
                AND user_type = 'admin' 
                AND status = '1' 
                AND deleted_at IS NULL 
                LIMIT 1
            ", [$userId])->getRow();
            
            if (!$result) {
                return false;
            }
            $groupId = $result->group_id;
        }

        // Super Admin (group_id = 1) has all permissions
        if ($groupId == 1) {
            return true;
        }

        // Check if permission exists for this group
        $db = \Config\Database::connect();
        $hasPermission = $db->query("
            SELECT 1
            FROM usc_user_group_permissions ugp
            INNER JOIN usc_permissions p ON p.id = ugp.permission_id
            WHERE ugp.group_id = ?
            AND p.name = ?
            AND ugp.deleted_at IS NULL
            AND p.deleted_at IS NULL
            LIMIT 1
        ", [$groupId, $permissionKey])->getRow();

        return !empty($hasPermission);
    }

    // On Button functions with jquery
    function permission_attrs(
        string $permission,
        string $action,
        ...$params
    ): string {
        if (!has_permission($permission)) {
            return 'hidden title="Requires permission"';
        }

        $jsParams = array_map(function($param) {
            if (is_numeric($param)) {
                return $param; // Numbers without quotes
            }
            return "'" . addslashes($param) . "'"; // Strings with escaped quotes
        }, $params);

        return 'onclick="' . $action . '(' . implode(', ', $jsParams) . ')"';
    }

    // Link Buttons
    function permission_link(string $permission, string $url, string $text, array $attrs = []): string {
        if (!has_permission($permission)) {
            $attrs['class'] = ($attrs['class'] ?? '') . ' hidden';
            $attrs['title'] = 'Requires permission';
            $url = 'javascript:void(0)';
        }

        $attrString = '';
        foreach ($attrs as $key => $value) {
            $attrString .= " $key=\"" . htmlspecialchars($value) . "\"";
        }

        return "<a href=\"$url\"$attrString>$text</a>";
    }
}
