<?php

use Modules\Menu\Models\MenusModel;
use Modules\Menu\Models\MenuTypeModel;

/**
 * Get current page context from URL or page data
 */
if (!function_exists('get_page_context')) {
    function get_page_context()
    {
        // First, check if page_type is set in view data (from controller)
        $renderer = \Config\Services::renderer();
        $data = $renderer->getData();
        
        if (isset($data['page_type'])) {
            return $data['page_type'];
        }
        
        // Fallback to URL segment detection
        $uri = service('uri');
        $segment = $uri->getSegment(1);
        
        // Map URL segments to contexts
        if (empty($segment) || $segment === 'home') {
            return 'home';
        } elseif ($segment === 'personal') {
            return 'personal';
        } elseif ($segment === 'business') {
            return 'business';
        } elseif ($segment === 'forex') {
            return 'forex';
        } elseif ($segment === 'services') {
            // For service pages, check if we have nav_context in data
            if (isset($data['page']['nav_context'])) {
                return $data['page']['nav_context'];
            }
            return 'personal'; // Default for services
        } else {
            return 'home'; // Default to home for unknown pages
        }
    }
}

/**
 * Get menu type based on page context
 */
if (!function_exists('get_menu_type_by_context')) {
    function get_menu_type_by_context($context = null)
    {
        if ($context === null) {
            $context = get_page_context();
        }
        
        // Business has its own menu, everything else uses personal-menu
        return ($context === 'business') ? 'business-menu' : 'personal-menu';
    }
}


/**
 * Render Top Bar - Simple context switcher
 */
if (!function_exists('render_top_bar')) {
    function render_top_bar()
    {
        $context = get_page_context();
        
        ob_start(); ?>
<!-- Top black bar -->
<div class="top-bar">
    <div class="top-left">
        <a href="<?= base_url('personal'); ?>" class="<?= in_array($context, ['personal']) ? 'personal-highlight' : '' ?>">Personal</a>
        <a href="<?= base_url('business'); ?>" class="<?= $context === 'business' ? 'personal-highlight' : '' ?>">Business</a>
    </div>
    <div class="top-right">
        <a href="<?= base_url('forex'); ?>">Forex</a>
    </div>
</div>
<?php
        return ob_get_clean();
    }
}

/**
 * Render Main Navigation - Database-driven only
 */
if (!function_exists('render_main_navigation'))
{
    function render_main_navigation()
    {
        $menuModel = new MenusModel();
        $context = get_page_context();
        $menuType = get_menu_type_by_context($context);
        
        // Get menu tree from database
        $mainNavItems = $menuModel->getMenuTreeByType($menuType);

        ob_start();
        ?>
        <!-- Main white navbar -->
        <div class="main-nav">
            <a href="<?= base_url('/'); ?>">
                <img src="<?= base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'); ?>" alt="ABC Logo">
            </a>
            <div class="menu-links navbar navbar-expand-lg">
                <div class="container-fluid">
                    <?php if (!empty($mainNavItems)): ?>
                        <?php foreach ($mainNavItems as $item): ?>
                            <?php if (!empty($item['children'])): ?>
                                <!-- Dropdown Menu Item -->
                                <div class="menu-item dropdown">
                                    <span><?= esc($item['title']); ?> <small><?= esc($item['subtitle'] ?? ''); ?></small></span>
                                    <div class="dropdown-content">
                                        <?php foreach ($item['children'] as $child): ?>
                                            <a href="<?= base_url($child['link']); ?>" class="dropdown-item">
                                                <?php if (!empty($child['icon'])): ?>
                                                    <i class="<?= esc($child['icon']); ?>"></i>
                                                <?php endif; ?>
                                                <?= esc($child['title']); ?>
                                            </a>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            <?php else: ?>
                                <!-- Single Menu Item -->
                                <a href="<?= base_url($item['link']); ?>" class="menu-item">
                                    <span><?= esc($item['title']); ?> <small><?= esc($item['subtitle'] ?? ''); ?></small></span>
                                </a>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    
                    <!-- Search Icon
                    <div class="search-icon">
                        <i class="fas fa-search"></i>
                    </div> -->
                </div>
            </div>
            <!-- Mobile menu toggle -->
            <div class="mobile-toggle" id="mobileToggle">
                <span></span>
                <span></span>
                <span></span>
            </div>
        </div>

        <!-- Mobile menu container -->
        <div class="mobile-menu-container" id="mobileMenuContainer">
            <div class="mobile-menu">
                <div class="mobile-menu-header">
                    <img src="<?= base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'); ?>" alt="ABC Logo">
                    <a class="card-button <?= $context === 'personal' ? 'personal-highlight' : '' ?>" href="<?= base_url('personal'); ?>">Personal</a>
                    <a class="card-button <?= $context === 'business' ? 'personal-highlight' : '' ?>" href="<?= base_url('business'); ?>">Business</a>
                    <div class="close-menu" id="closeMenu">
                        <i class="fas fa-times"></i>
                    </div>
                </div>
                
                <!-- <div class="mobile-search">
                    <input type="text" placeholder="Search...">
                </div> -->
                
                <h2>Personal Accounts</h2>
                
                <?php if (!empty($mainNavItems)): ?>
                    <?php foreach ($mainNavItems as $item): ?>
                        <?php if (!empty($item['children'])): ?>
                            <!-- Mobile Dropdown -->
                            <div class="mobile-menu-item" data-target="<?= esc($item['mobile_target'] ?? 'mobile-dropdown-' . $item['id']); ?>">
                                <?= esc($item['title']); ?> <small><?= esc($item['subtitle'] ?? ''); ?></small>
                                <span class="dropdown-arrow"><i class="fas fa-chevron-down"></i></span>
                            </div>
                            <div class="mobile-dropdown-content" id="<?= esc($item['mobile_target'] ?? 'mobile-dropdown-' . $item['id']); ?>">
                                <?php foreach ($item['children'] as $child): ?>
                                    <a href="<?= base_url($child['link']); ?>" class="mobile-dropdown-item">
                                        <?php if (!empty($child['icon'])): ?>
                                            <i class="<?= esc($child['icon']); ?>"></i>
                                        <?php endif; ?>
                                        <?= esc($child['title']); ?>
                                    </a>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <!-- Single Mobile Menu Item -->
                            <a href="<?= base_url($item['link']); ?>" class="mobile-menu-item">
                                <span><?= esc($item['title']); ?> <small><?= esc($item['subtitle'] ?? ''); ?></small></span>
                            </a>
                        <?php endif; ?>
                    <?php endforeach; ?>
                <?php endif; ?>
                
                <div style="padding: 20px 0; border-top: 1px solid #eee; margin-top: 15px;">
                    <a href="<?= base_url('account'); ?>" style="display: block; padding: 12px; background: var(--kpy-warning); color: white; text-align: center; border-radius: 4px; text-decoration: none; font-weight: 500;">Create An Account</a>
                </div>
            </div>
        </div>

        <!-- Mobile Menu JavaScript -->
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const mobileToggle = document.getElementById('mobileToggle');
                const mobileMenuContainer = document.getElementById('mobileMenuContainer');
                const closeMenu = document.getElementById('closeMenu');

                if (mobileToggle) {
                    mobileToggle.addEventListener('click', function() {
                        mobileMenuContainer.style.display = 'block';
                        setTimeout(function() {
                            document.querySelector('.mobile-menu').style.right = '0';
                        }, 10);
                    });
                }

                if (closeMenu) {
                    closeMenu.addEventListener('click', function() {
                        document.querySelector('.mobile-menu').style.right = '-300px';
                        setTimeout(function() {
                            mobileMenuContainer.style.display = 'none';
                        }, 300);
                    });
                }

                document.querySelectorAll('.mobile-menu-item[data-target]').forEach(function(item) {
                    item.addEventListener('click', function() {
                        const targetId = this.getAttribute('data-target');
                        const dropdown = document.getElementById(targetId);
                        if (dropdown) {
                            dropdown.classList.toggle('show');
                            const arrow = this.querySelector('.dropdown-arrow');
                            if (arrow) {
                                arrow.classList.toggle('fa-chevron-down');
                                arrow.classList.toggle('fa-chevron-up');
                            }
                        }
                    });
                });

                const navbar = document.getElementById('main-nav');
                if (navbar) {
                    window.addEventListener('scroll', function() {
                        if (window.pageYOffset > 50) {
                            navbar.classList.add('scrolled');
                        } else {
                            navbar.classList.remove('scrolled');
                        }
                    });
                }
            });
        </script>
        <?php
        return ob_get_clean();
    }
}


/**
 * Render Footer - Database-driven only with consistent chevron icons
 */
if (!function_exists('render_footer')) {
    function render_footer()
    {
        $menuModel = new MenusModel();

        // Get footer menu types from database
        $footerQuickLinks = $menuModel->getMenuByType('footer-quick-links');
        $footerBankingServices = $menuModel->getMenuByType('footer-banking-services');
        $footerWhoWeAre = $menuModel->getMenuByType('footer-who-we-are');
        $socialMediaLinks = $menuModel->getMenuByType('social-media');

        ob_start(); ?>
        <!-- Footer -->
        <footer class="footer">
            <div class="footer-container">
                <!-- Contact Info -->
                <div class="footer-contact">
                    <div class="footer-logo">
                        <h2>ABC <span>Capital</span> Bank</h2>
                    </div>
                    <p><i class="fas fa-map-marker-alt"></i> 123 Financial Street, Banking District, BC 10001</p>
                    <p><i class="fas fa-phone-alt"></i> +1 (800) 123-4567</p>
                    <p><i class="fas fa-envelope"></i> info@abccapitalbank.com</p>
                    <div class="social-icon">
                        <?php if (!empty($socialMediaLinks)): ?>
                            <?php foreach ($socialMediaLinks as $social): ?>
                                <a href="<?= esc($social->link); ?>" target="_blank">
                                    <i class="<?= esc($social->icon); ?>"></i>
                                </a>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Quick Links -->
                <?php if (!empty($footerQuickLinks)): ?>
                <div class="footer-links">
                    <h3 class="footer-heading">Quick Links</h3>
                    <ul>
                        <?php foreach ($footerQuickLinks as $link): ?>
                            <li>
                                <a href="<?= base_url($link->link); ?>">
                                    <i class="fas fa-chevron-right"></i>
                                    <?= esc($link->title); ?>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>

                <!-- Banking Services -->
                <?php if (!empty($footerBankingServices)): ?>
                <div class="footer-links">
                    <h3 class="footer-heading">Banking Services</h3>
                    <ul>
                        <?php foreach ($footerBankingServices as $link): ?>
                            <li>
                                <a href="<?= base_url($link->link); ?>">
                                    <i class="fas fa-chevron-right"></i>
                                    <?= esc($link->title); ?>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>

                <!-- Who We Are -->
                <?php if (!empty($footerWhoWeAre)): ?>
                <div class="footer-links">
                    <h3 class="footer-heading">Who We Are</h3>
                    <ul>
                        <?php foreach ($footerWhoWeAre as $link): ?>
                            <li>
                                <a href="<?= base_url($link->link); ?>">
                                    <i class="fas fa-chevron-right"></i>
                                    <?= esc($link->title); ?>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
            </div>

            <div class="footer-bottom">
                <p>&copy; <?= date('Y'); ?> ABC Capital Bank. All Rights Reserved.</p>
                <div class="payment-methods">
                    <i class="fab fa-cc-visa"></i>
                    <i class="fab fa-cc-mastercard"></i>
                    <i class="fab fa-cc-amex"></i>
                    <i class="fab fa-cc-paypal"></i>
                    <i class="fab fa-cc-apple-pay"></i>
                </div>
            </div>
        </footer>

        <!-- Simple CSS for consistent icon styling -->
        <style>
            .footer-links ul li a i {
                margin-right: 8px;
                font-size: 0.8em;
                opacity: 0.8;
            }
            
            .footer-links ul li a:hover i {
                opacity: 1;
                transform: translateX(2px);
                transition: all 0.3s ease;
            }
        </style>
        <?php
        return ob_get_clean();
    }

}