<?php

/**
 * Get available block types
 */
function get_block_categories()
{
    return [
        'content' => 'Content',
        'header' => 'Header',
        'footer' => 'Footer',
        'sidebar' => 'Sidebar'
    ];
}

/**
 * Get block icon based on type
 */
function get_block_icon($type)
{
    $icons = [
        'text' => 'bi bi-text-paragraph',
        'heading' => 'bi bi-type-h1',
        'image' => 'bi bi-image',
        'gallery' => 'bi bi-images',
        'video' => 'bi bi-play-circle',
        'file' => 'bi bi-file-earmark',
        'list' => 'bi bi-list-ul',
        'table' => 'bi bi-table',
        'code' => 'bi bi-code-square',
        'quote' => 'bi bi-chat-quote',
        'button' => 'bi bi-box-arrow-up-right',
        'divider' => 'bi bi-hr',
        'html' => 'bi bi-code',
        'map' => 'bi bi-geo-alt',
        'form' => 'bi bi-input-cursor',
        'slider' => 'bi bi-sliders',
        'tabs' => 'bi bi-layout-three-columns',
        'accordion' => 'bi bi-arrows-collapse',
        'social' => 'bi bi-share',
        'embed' => 'bi bi-box-arrow-in-down-right'
    ];

    return $icons[$type] ?? 'bi bi-square';
}

/**
 * Format block for display
 */
function format_block_content($block)
{
    switch ($block['block_type']) {
        case 'text':
            return nl2br(esc($block['content']));
        
        case 'heading':
            return sprintf('<%1$s style="color: %2$s">%3$s</%1$s>', 
                esc($block['heading_type']), 
                esc($block['color']), 
                esc($block['heading'])
            );
        
        case 'image':
            $html = sprintf('<img src="%s" class="img-fluid" alt="%s">', 
                base_url('uploads/' . $block['photo']),
                esc($block['caption'])
            );
            if ($block['caption']) {
                $html .= sprintf('<figcaption class="figure-caption">%s</figcaption>', 
                    esc($block['caption'])
                );
            }
            return $html;
        
        case 'list':
            $items = explode("\n", $block['content']);
            $list_type = $block['list_type'] == 'ol' ? 'ol' : 'ul';
            $html = "<{$list_type}>";
            foreach ($items as $item) {
                $html .= sprintf('<li>%s</li>', esc(trim($item)));
            }
            $html .= "</{$list_type}>";
            return $html;
        
        case 'code':
            return sprintf('<pre><code>%s</code></pre>', 
                esc($block['code'])
            );
        
        default:
            return esc($block['content']);
    }
}

/**
 * Check if block type supports items
 */
function block_supports_items($type)
{
    return in_array($type, ['gallery', 'slider', 'tabs', 'accordion']);
}

/**
 * Get block validation rules based on type
 */
function get_block_validation_rules($type)
{
    $rules = [
        'text' => [
            'content' => 'required'
        ],
        'heading' => [
            'heading' => 'required',
            'heading_type' => 'required|in_list[h1,h2,h3,h4,h5,h6]'
        ],
        'image' => [
            'photo' => 'required'
        ],
        'list' => [
            'content' => 'required',
            'list_type' => 'required|in_list[ul,ol]'
        ],
        'code' => [
            'code' => 'required'
        ]
    ];

    return $rules[$type] ?? [];
}