<?php

namespace App\Controllers\Front;

use App\Controllers\BaseController;
use Modules\Sliders\Models\SlidersModel;
use Modules\Posts\Models\PostsModel;
use Modules\ForexExchange\Models\ForexExchangeModel;
use Modules\CustomerCare\Models\FeedbackSubjectModel;
use Modules\CustomerCare\Models\CustomerFeedbackModel;
use Modules\Team\Models\TeamModel;
use Modules\Team\Models\TeamCategoryModel;
use Modules\Jobs\Models\JobsModel; // Add this line

class FrontPages extends BaseController
{
    public $slidersModel;
    public $postsModel;
    public $forexExchangeModel;
    public $bannersModel;
    public $postBlocksModel;
    public $pagesModel;
    public $pagescategoryModel;
    public $postsCategoryModel;
    public $postTagsModel;
    public $feedbacksubjectModel;
    public $customerfeedbackModel;
    public $jobsModel; // Add this property
    public $data = [];
    protected $db;

    public function __construct()
    {
        // Initialize database connection
        $this->db = \Config\Database::connect();

        // Common data that might be needed across pages
        $this->data['bannersModel'] = $this->bannersModel;

        // Initialize customer care models
        $this->feedbacksubjectModel = new FeedbackSubjectModel();
        $this->customerfeedbackModel = new CustomerFeedbackModel();
        
        // Initialize Jobs Model
        $this->jobsModel = new JobsModel(); // Add this line
    }

    public function index()
    {
        // Get dynamic metadata from database with tags
        $metadata = $this->postsModel->getPageMetadataByAlias('home');

        if ($metadata) {
            $this->data = [
                'title' => $metadata['title'] ?? 'ABC Capital Bank - Uganda\'s Digital-Led Financial Solutions Provider',
                'tagline' => $metadata['excerpt'] ?? 'Your Partner in Digital Banking and Financial Innovation',
                'description' => $metadata['metadesc'] ?? 'ABC Capital Bank is a digital-led Tier 2 bank in Uganda, transitioning from its origins as Capital Finance Corporation in 1993. We offer seamless, customer-centric banking solutions for Retail & SME segments, focusing on digital payments, electronic loans, savings, and personalized services.',
                'keywords' => $metadata['metakey'] ?? 'ABC Capital Bank, ABC Bank Uganda, digital bank Uganda, SME banking Uganda, retail banking Uganda, Capital Finance Corporation, CFC Ltd, Bank of Uganda, financial solutions, digital payments, electronic loans, Uganda banking, financial technology, FinTech Uganda',
                'page_type' => 'home',
                'page_metadata' => $metadata,
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            // Fallback to hardcoded values if no database entry
            $this->data = [
                'title' => 'ABC Capital Bank - Uganda\'s Digital-Led Financial Solutions Provider',
                'tagline' => 'Your Partner in Digital Banking and Financial Innovation',
                'description' => 'ABC Capital Bank is a digital-led Tier 2 bank in Uganda, transitioning from its origins as Capital Finance Corporation in 1993. We offer seamless, customer-centric banking solutions for Retail & SME segments, focusing on digital payments, electronic loans, savings, and personalized services.',
                'keywords' => 'ABC Capital Bank, ABC Bank Uganda, digital bank Uganda, SME banking Uganda, retail banking Uganda, Capital Finance Corporation, CFC Ltd, Bank of Uganda, financial solutions, digital payments, electronic loans, Uganda banking, financial technology, FinTech Uganda',
                'page_type' => 'home',
                'tags' => ['banking', 'Uganda', 'financial solutions', 'digital banking']
            ];
        }

        // Get sliders from database
        $this->data['sliders'] = $this->slidersModel->getHomeSlides();

        // Load forex rates - limit to 5
        $allRates = $this->forexExchangeModel->getActiveRates();
        $this->data['forex_rates'] = array_slice($allRates, 0, 5);
        $this->data['forex_last_updated'] = !empty($this->data['forex_rates']) ?
            date('F j, Y', strtotime($this->data['forex_rates'][0]->updated_at)) :
            date('F j, Y');

        // Get service cards and featured services
        $this->data['service_cards'] = $this->postsModel->getServiceCards();
        $this->data['featured_services'] = $this->postsModel->getFeaturedServices();

        // Get popular tags for SEO
        $this->data['popular_tags'] = $this->postsModel->getPopularTags(10);

        return view('website/pages/home', $this->data);
    }

    // Personal Banking Page
    public function personalPage()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('personal-banking');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Personal Banking - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Banking Solutions for Your Personal Growth',
                'description' => $metadata['metadesc'] ?? 'Discover our comprehensive personal banking services including current accounts, savings, loans, and digital banking solutions.',
                'keywords' => $metadata['metakey'] ?? 'personal banking, current accounts, savings accounts, personal loans, digital banking, ABC Capital Bank Uganda',
                'page_type' => 'personal',
                'page_metadata' => $metadata,
                'tags' => $metadata['tags_array'] ?? [],
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif')
            ];
        } else {
            $data = [
                'title' => 'Personal Banking - ABC Capital Bank',
                'tagline' => 'Banking Solutions for Your Personal Growth',
                'description' => 'Discover our comprehensive personal banking services including current accounts, savings, loans, and digital banking solutions.',
                'keywords' => 'personal banking, current accounts, savings accounts, personal loans, digital banking, ABC Capital Bank Uganda',
                'page_type' => 'personal',
                'tags' => ['personal banking', 'current accounts', 'savings', 'loans'],
                'featured_image' => base_url('assets/front_assets/images/personal-banking.jpg')
            ];
        }

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('personal');
        $data['page_blocks'] = $this->postBlocksModel->getPageBlocks('personal-banking');
        $data['service_counts'] = $this->postBlocksModel->getServiceCountsByCategory('personal-banking');

        $data['current_accounts'] = $this->postBlocksModel->getServiceCardsByCategory('personal-banking', 'Current');
        $data['savings_accounts'] = $this->postBlocksModel->getServiceCardsByCategory('personal-banking', 'Savings');
        $data['loan_services'] = $this->postBlocksModel->getServiceCardsByCategory('personal-banking', 'Loans');
        $data['investing_services'] = $this->postBlocksModel->getServiceCardsByCategory('personal-banking', 'Investing');
        $data['insurance_services'] = $this->postBlocksModel->getServiceCardsByCategory('personal-banking', 'Insurance');

        $data['current_count'] = $data['service_counts']['Current'] ?? 0;
        $data['savings_count'] = $data['service_counts']['Savings'] ?? 0;
        $data['loans_count'] = $data['service_counts']['Loans'] ?? 0;
        $data['investing_count'] = $data['service_counts']['Investing'] ?? 0;
        $data['insurance_count'] = $data['service_counts']['Insurance'] ?? 0;

        $data['bannersModel'] = $this->bannersModel;

        return view('website/pages/personal', $data);
    }

    // Business Banking Page
    public function businessPage()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('business-banking');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'],
                'tagline' => $metadata['excerpt'],
                'description' => $metadata['metadesc'],
                'keywords' => $metadata['metakey'],
                'page_type' => 'business',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Business Banking - ABC Capital Bank',
                'tagline' => 'Banking Solutions for Your Business Growth',
                'description' => 'Comprehensive business banking solutions including SME/Corporate accounts, business loans, trade finance, and investment services.',
                'keywords' => 'business banking, SME banking, corporate accounts, business loans, trade finance, ABC Capital Bank Uganda',
                'page_type' => 'business',
                'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => ['business banking', 'SME', 'corporate accounts', 'loans']
            ];
        }

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('business');
        $data['page_blocks'] = $this->postBlocksModel->getPageBlocks('business-banking');
        $data['service_counts'] = $this->postBlocksModel->getServiceCountsByCategory('business-banking');

        $data['business_accounts'] = $this->postBlocksModel->getServiceCardsByCategory('business-banking', 'SME/Corporate');
        $data['business_loans'] = $this->postBlocksModel->getServiceCardsByCategory('business-banking', 'Loans');
        $data['trade_finance'] = $this->postBlocksModel->getServiceCardsByCategory('business-banking', 'Investing');
        $data['business_insurance'] = $this->postBlocksModel->getServiceCardsByCategory('business-banking', 'Insurance');

        $data['accounts_count'] = $data['service_counts']['SME/Corporate'] ?? 0;
        $data['loans_count'] = $data['service_counts']['Loans'] ?? 0;
        $data['trade_count'] = $data['service_counts']['Investing'] ?? 0;
        $data['insurance_count'] = $data['service_counts']['Insurance'] ?? 0;

        $data['bannersModel'] = $this->bannersModel;

        return view('website/pages/business', $data);
    }

    public function forexPage()
    {
        $metadata = $this->postsModel->getPageMetadataByAlias('forex');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'],
                'tagline' => $metadata['excerpt'],
                'description' => $metadata['metadesc'],
                'keywords' => $metadata['metakey'],
                'page_type' => 'forex',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/abc-logo.png'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Forex Services - ABC Capital Bank',
                'tagline' => 'Foreign Exchange & Currency Services',
                'description' => 'Professional forex services including foreign exchange, currency trading, and international money transfers.',
                'keywords' => 'forex, foreign exchange, currency trading, international transfers, ABC Capital Bank Uganda',
                'page_type' => 'forex',
                'featured_image' => base_url('assets/front_assets/images/abc-logo.png'),
                'tags' => ['forex', 'currency exchange', 'international transfers']
            ];
        }

        $data['forex_rates'] = $this->forexExchangeModel->getActiveRates();
        $data['forex_last_updated'] = !empty($data['forex_rates']) ?
            date('F j, Y', strtotime($data['forex_rates'][0]->updated_at)) :
            date('F j, Y');

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('forex');

        return view('website/pages/forex', $data);
    }

    // Ways to Bank Page
    public function waysToBankPage()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('ways-to-bank');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Ways of Banking - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Bank anytime, anywhere with ABC Capital Bank',
                'description' => $metadata['metadesc'] ?? 'Discover multiple ways to bank with ABC Capital Bank including branch banking, ATM services, online banking, and mobile banking solutions.',
                'keywords' => $metadata['metakey'] ?? 'ways to bank, branch banking, ATM banking, online banking, mobile banking, ABC Capital Bank Uganda',
                'page_type' => 'ways-to-bank',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/bank1.jpg'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Ways of Banking - ABC Capital Bank',
                'tagline' => 'Bank anytime, anywhere',
                'description' => 'Discover multiple ways to bank with ABC Capital Bank including branch banking, ATM services, online banking, and mobile banking solutions.',
                'keywords' => 'ways to bank, branch banking, ATM banking, online banking, mobile banking, ABC Capital Bank Uganda',
                'page_type' => 'ways-to-bank',
                'featured_image' => base_url('assets/front_assets/images/bank1.jpg'),
                'tags' => ['banking methods', 'branch locations', 'ATM', 'online banking']
            ];
        }

        $data['page'] = $metadata ?? [
            'title' => 'Ways to Bank',
            'excerpt' => 'Bank anytime, anywhere'
        ];

        $blocks = $this->postBlocksModel->getPageBlocks('ways-to-bank');
        $data['templateData'] = $this->convertWaysToBankBlocks($blocks, $data['page']);
        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('ways-to-bank');

        return view('website/pages/ways_to_bank_template', $data);
    }
    // Private Helper Methods for Block Conversion

    private function convertWaysToBankBlocks($blocks, $post)
    {
        $templateData = [
            'banner_config' => [
                'hero_title' => $post['title'] ?? 'Ways of Banking',
                'hero_subtitle' => $post['excerpt'] ?? 'Bank anytime, anywhere in person at a branch or ATM, from your laptop or mobile when you are on the move.',
                'hero_bg' => $post['photo'] ?? 'bank1.jpg',
                'title_emphasized' => 'Banking'
            ]
        ];
        
        foreach ($blocks as $block) {
            switch ($block['block_type']) {
                case 'branches':
                    $templateData['branches_title'] = $block['heading'] ?? 'Our Branch Network';
                    $templateData['branches_title_emphasized'] = $block['title_emphasized'] ?? 'Branch Network';
                    
                    $branches = [];
                    if (isset($block['items'])) {
                        foreach ($block['items'] as $item) {
                            if ($item['item_type'] === 'branch') {
                                $branches[] = [
                                    'title' => $item['title'] ?? '',
                                    'address' => $item['description'] ?? '',
                                    'phone' => $item['value'] ?? '',
                                    'hours' => $item['caption'] ?? '',
                                    'description' => $item['link'] ?? ''
                                ];
                            }
                        }
                    }
                    $templateData['branches'] = $branches;
                    break;
                    
                case 'banking_methods':
                    $templateData['banking_options_title'] = $block['heading'] ?? 'Ways to Bank With Us';
                    
                    $bankingMethods = [];
                    if (isset($block['items'])) {
                        foreach ($block['items'] as $item) {
                            if ($item['item_type'] === 'banking_method') {
                                $features = [];
                                if (!empty($item['description'])) {
                                    $features = array_map('trim', explode(',', $item['description']));
                                }
                                
                                $bankingMethods[] = [
                                    'title' => $item['title'] ?? '',
                                    'icon' => $item['icon'] ?? 'fas fa-star',
                                    'features' => $features,
                                    'description' => $item['value'] ?? ''
                                ];
                            }
                        }
                    }
                    $templateData['banking_methods'] = $bankingMethods;
                    break;

                
            }
        }
        
        return $templateData;
    }

    // Corporate Social Responsibility Page
    public function corporateSocialResponsibility()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('corporate-social-responsibility');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Corporate Social Responsibility - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'ABC Capital Bank is a socially responsible enterprise supporting sustainable development initiatives.',
                'description' => $metadata['metadesc'] ?? 'Learn about ABC Capital Bank commitment to corporate social responsibility and community development in Uganda.',
                'keywords' => $metadata['metakey'] ?? 'corporate social responsibility, CSR, community support, education, healthcare, environment, social development',
                'page_type' => 'csr',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Corporate Social Responsibility - ABC Capital Bank',
                'tagline' => 'ABC Capital Bank is a socially responsible enterprise supporting sustainable development initiatives.',
                'description' => 'Learn about ABC Capital Bank commitment to corporate social responsibility and community development in Uganda.',
                'keywords' => 'corporate social responsibility, CSR, community support, education, healthcare, environment, social development',
                'page_type' => 'csr',
                'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => ['CSR', 'sustainability', 'community development']
            ];
        }

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('corporate-social-responsibility');
        $blocks = $this->postBlocksModel->getPageBlocks('corporate-social-responsibility');
        $data['templateData'] = $this->convertCSRBlocks($blocks, $data['page_metadata'] ?? []);

        return view('website/pages/csr_template', $data);
    }

    private function convertCSRBlocks($blocks, $post)
{
    helper('ltcms');
    
    $templateData = [
        'banner_config' => [
            'hero_title' => $post['title'] ?? 'Corporate Social Responsibility',
            'hero_subtitle' => $post['excerpt'] ?? 'ABC Capital Bank is a socially responsible enterprise supporting sustainable development initiatives.',
            'hero_bg' => $post['photo'] ?? '',
            'title_emphasized' => 'Social Responsibility'
        ]
    ];

    foreach ($blocks as $block) {
        switch ($block['block_type']) {
            case 'csr-commitment':
                $templateData['commitment'] = [
                    'title' => $block['heading'] ?? 'Our Commitment',
                    'title_emphasized' => $block['title_emphasized'] ?? 'Commitment',
                    'content' => $block['content'] ?? ''
                ];
                break;
                
            case 'csr-focus-areas':
                $templateData['focus_areas'] = [
                    'title' => $block['heading'] ?? 'Focus Areas',
                    'title_emphasized' => $block['title_emphasized'] ?? 'Areas',
                    'items' => []
                ];
                
                // Get focus area items from database
                $focusItems = $this->postBlocksModel->getBlockItems($block['id']);
                foreach ($focusItems as $item) {
                    if ($item['item_type'] === 'csr_focus_area') {
                        $templateData['focus_areas']['items'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-star',
                            'description' => $item['description'] ?? ''
                        ];
                    }
                }
                break;
                
            case 'csr-highlights':
                $templateData['highlights'] = [
                    'title' => $block['heading'] ?? 'CSR Highlights',
                    'title_emphasized' => $block['title_emphasized'] ?? 'Highlights',
                    'items' => []
                ];
                
                // Get highlight items from database
                $highlightItems = $this->postBlocksModel->getBlockItems($block['id']);
                foreach ($highlightItems as $item) {
                    if ($item['item_type'] === 'csr_highlight') {
                        $templateData['highlights']['items'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-star',
                            'description' => $item['description'] ?? ''
                        ];
                    }
                }
                break;
        }
    }

    return $templateData;
}

    // Corporate Governance Page
    public function corporateGovernance()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('corporate-governance');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Corporate Governance - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Maintaining sound corporate governance structure with highest standards of business integrity.',
                'description' => $metadata['metadesc'] ?? 'Learn about ABC Capital Bank corporate governance structure, board responsibilities, and governance principles.',
                'keywords' => $metadata['metakey'] ?? 'corporate governance, board of directors, governance principles, transparency, accountability',
                'page_type' => 'governance',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Corporate Governance - ABC Capital Bank',
                'tagline' => 'Maintaining sound corporate governance structure with highest standards of business integrity.',
                'description' => 'Learn about ABC Capital Bank corporate governance structure, board responsibilities, and governance principles.',
                'keywords' => 'corporate governance, board of directors, governance principles, transparency, accountability',
                'page_type' => 'governance',
                'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => ['governance', 'board', 'transparency']
            ];
        }

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('corporate-governance');
        $blocks = $this->postBlocksModel->getPageBlocks('corporate-governance');
        $data['templateData'] = $this->convertGovernanceBlocks($blocks, $data['page_metadata'] ?? []);

        return view('website/pages/governance_template', $data);
    }

    private function convertGovernanceBlocks($blocks, $post)
{
    helper('ltcms');

    $templateData = [
        'banner_config' => [
            'hero_title' => $post['title'] ?? 'Corporate Governance',
            'hero_subtitle' => $post['excerpt'] ?? 'Maintaining sound corporate governance structure with highest standards of business integrity.',
            'hero_bg' => $post['photo'] ?? '',
            'title_emphasized' => 'Governance'
        ]
    ];

    foreach ($blocks as $block) {
        switch ($block['block_type']) {
            case 'governance-responsibilities':
                $templateData['responsibilities'] = [
                    'title' => $block['heading'] ?? 'Respective Responsibilities',
                    'title_emphasized' => $block['title_emphasized'] ?? 'Responsibilities',
                    'items' => []
                ];

                // Get responsibility items
                $responsibilityItems = $this->postBlocksModel->getBlockItems($block['id']);
                foreach ($responsibilityItems as $item) {
                    if ($item['item_type'] === 'responsibility_item') {
                        $templateData['responsibilities']['items'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-star',
                            'description' => $item['description'] ?? ''
                        ];
                    }
                }
                break;

            case 'board-directors-info':
                $templateData['board_directors'] = [
                    'title' => $block['heading'] ?? 'Board of Directors',
                    'title_emphasized' => $block['title_emphasized'] ?? 'Directors',
                    'content' => $block['content'] ?? '',
                    'highlights' => []
                ];

                // Get board highlight items
                $highlightItems = $this->postBlocksModel->getBlockItems($block['id']);
                foreach ($highlightItems as $item) {
                    if ($item['item_type'] === 'board_highlight') {
                        $templateData['board_directors']['highlights'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-star',
                            'description' => $item['description'] ?? ''
                        ];
                    }
                }
                break;

            case 'governance-principles':
                $templateData['principles'] = [
                    'title' => $block['heading'] ?? 'Governance Principles',
                    'title_emphasized' => $block['title_emphasized'] ?? 'Principles',
                    'items' => []
                ];

                // Get principle items
                $principleItems = $this->postBlocksModel->getBlockItems($block['id']);
                foreach ($principleItems as $item) {
                    if ($item['item_type'] === 'principle_item') {
                        $templateData['principles']['items'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-star',
                            'description' => $item['description'] ?? ''
                        ];
                    }
                }
                break;
        }
    }

    return $templateData;
}

    
    

// About Us Page
public function aboutUs()
{
   helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('about-us');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'About Us - ABC Capital Bank',
            'tagline' => $metadata['excerpt'] ?? 'Learn about ABC Capital Bank',
            'description' => $metadata['metadesc'] ?? 'Learn about ABC Capital Bank Uganda, our history, mission, and commitment to digital banking innovation.',
            'keywords' => $metadata['metakey'] ?? 'about ABC Capital Bank, bank history Uganda, digital banking, financial innovation',
            'page_type' => 'about-us',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'About Us - ABC Capital Bank',
            'tagline' => 'Learn about ABC Capital Bank',
            'description' => 'Learn about ABC Capital Bank Uganda, our history, mission, and commitment to digital banking innovation.',
            'keywords' => 'about ABC Capital Bank, bank history Uganda, digital banking, financial innovation',
            'page_type' => 'about-us',
            'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => ['about', 'history', 'mission']
        ];
    }

    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('about-us');

    $blocks = $this->postBlocksModel->getPageBlocks('about-us');
    $data['templateData'] = $this->convertAboutUsBlocks($blocks, $data['page_metadata'] ?? []);

    return view('website/pages/about_us_template', $data);
}
private function convertAboutUsBlocks($blocks, $post)
    {
        helper('ltcms');

        $templateData = [
            'banner_config' => [
                'hero_title' => $post['title'] ?? 'About Us',
                'hero_subtitle' => $post['excerpt'] ?? 'ABC is transitioning from a business-as-usual Tier one bank into a Digital-led Tier two bank.',
                'hero_bg' => $post['photo'] ?? 'car.webp',
                'title_emphasized' => 'Us'
            ]
        ];

        foreach ($blocks as $block) {
            switch ($block['block_type']) {
                case 'about_journey':
                    $templateData['journey'] = [
                        'title' => $block['heading'] ?? 'Our Journey',
                        'title_emphasized' => $block['title_emphasized'] ?? 'Journey',
                        'subtitle' => $block['subtitle'] ?? 'From humble beginnings as a microfinance institution to becoming Uganda\'s most innovative digital bank',
                        'items' => []
                    ];

                    // Get journey items
                    $journeyItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($journeyItems as $item) {
                        if ($item['item_type'] === 'journey_item') {
                            $templateData['journey']['items'][] = [
                                'title' => $item['title'] ?? '',
                                'description' => $item['description'] ?? ''
                            ];
                        }
                    }
                    break;

                case 'about_vision_mission':
                    $templateData['vision_mission'] = [
                        'title' => $block['heading'] ?? 'Vision & Mission',
                        'title_emphasized' => $block['title_emphasized'] ?? 'Mission',
                        'subtitle' => $block['subtitle'] ?? 'Driving digital transformation in Uganda\'s banking sector',
                        'items' => []
                    ];

                    // Get vision/mission items
                    $vmItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($vmItems as $item) {
                        if ($item['item_type'] === 'vision_mission_item') {
                            $templateData['vision_mission']['items'][] = [
                                'title' => $item['title'] ?? '',
                                'description' => $item['description'] ?? ''
                            ];
                        }
                    }
                    break;

                case 'about_values':
                    $templateData['values'] = [
                        'title' => $block['heading'] ?? 'Our Core Values',
                        'title_emphasized' => $block['title_emphasized'] ?? 'Values',
                        'subtitle' => $block['subtitle'] ?? 'The principles that guide everything we do',
                        'items' => []
                    ];

                    // Get values items
                    $valueItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($valueItems as $item) {
                        if ($item['item_type'] === 'value_item') {
                            $templateData['values']['items'][] = [
                                'title' => $item['title'] ?? '',
                                'icon' => $item['icon'] ?? '👥',
                                'description' => $item['description'] ?? ''
                            ];
                        }
                    }
                    break;

                case 'about_goals':
                    $templateData['goals'] = [
                        'title' => $block['heading'] ?? 'Our Goals',
                        'title_emphasized' => $block['title_emphasized'] ?? 'Goals',
                        'items' => []
                    ];

                    // Get goals items
                    $goalItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($goalItems as $item) {
                        if ($item['item_type'] === 'goal_item') {
                            $templateData['goals']['items'][] = [
                                'title' => $item['title'] ?? ''
                            ];
                        }
                    }
                    break;

                case 'about_benefits':
                    $templateData['benefits'] = [
                        'title' => $block['heading'] ?? 'Benefits of Our Transition',
                        'title_emphasized' => $block['title_emphasized'] ?? 'Transition',
                        'subtitle' => $block['subtitle'] ?? 'Why our digital transformation matters to you',
                        'items' => []
                    ];

                    // Get benefits items
                    $benefitItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($benefitItems as $item) {
                        if ($item['item_type'] === 'benefit_item') {
                            $templateData['benefits']['items'][] = [
                                'title' => $item['title'] ?? '',
                                'description' => $item['description'] ?? ''
                            ];
                        }
                    }
                    break;

                case 'about_contact':
                    $templateData['contact'] = [
                        'title' => $block['heading'] ?? 'Get In Touch',
                        'title_emphasized' => $block['title_emphasized'] ?? 'Touch',
                        'subtitle' => $block['subtitle'] ?? 'For more information on our transition to a digital Tier 2 bank in Uganda, contact our leadership team',
                        'numbers' => []
                    ];

                    // Get contact items
                    $contactItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($contactItems as $item) {
                        if ($item['item_type'] === 'contact_item') {
                            $templateData['contact']['numbers'][] = [
                                'value' => $item['value'] ?? ''
                            ];
                        }
                    }
                    break;
            }
        }

        return $templateData;
    }


    
    // Board of Directors - Executive Page
    public function boardOfDirectors()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('board-of-directors');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Board of Directors - Executive and Non Executive- ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Meet our executive and non executive board of directors and leadership team.',
                'description' => $metadata['metadesc'] ?? 'Meet the executive board of directors at ABC Capital Bank Uganda - experienced leaders driving our success.',
                'keywords' => $metadata['metakey'] ?? 'board of directors, executive board, bank leadership, management team, ABC Capital Bank',
                'page_type' => 'board-of_directors',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Board of Directors - ABC Capital Bank',
                'tagline' => 'Meet our executive board of directors and leadership team.',
                'description' => 'Meet the executive board of directors at ABC Capital Bank Uganda - experienced leaders driving our success.',
                'keywords' => 'board of directors, executive board, bank leadership, management team, ABC Capital Bank',
                'page_type' => 'board-executive',
                'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => ['executive board', 'leadership', 'management']
            ];
        }

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('board-of-directors');

        // Load team members for the executive board using Team module
        $teamModel = new TeamModel();
        $members = $teamModel->showBoardMembers();

        // Build templateData consistent with other templates
        $templateData = [
            'team_title' => $data['title'] ?? 'Board of Directors',
            'team_title_emphasized' => 'Directors',
            'team_description' => $data['tagline'] ?? '',
            'team_members' => []
        ];

        foreach ($members as $m) {
            $templateData['team_members'][] = [
                'image' => $m->photo ?? ($m->image ?? null),
                'name' => $m->title ?? $m->name ?? '',
                'position' => $m->position ?? '',
                'modal_id' => 'teamModal-' . ($m->id ?? uniqid())
            ];
        }

        $data['templateData'] = $templateData;

    // Reuse the generic team template for consistent banner + members layout
    // Use the dedicated board route (no filter) so the template can classify and show both executive and non-executive
    return $this->teamPageTemplate('board-of-directors');
    }

    //MAnagement Team
    public function managementTeam()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('management-team');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Board of Directors - Non Executive - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Meet our non-executive board of directors providing independent oversight.',
                'description' => $metadata['metadesc'] ?? 'Meet the non-executive board of directors at ABC Capital Bank Uganda providing independent governance.',
                'keywords' => $metadata['metakey'] ?? 'non-executive directors, board governance, independent directors, ABC Capital Bank',
                'page_type' => 'board-non-executive',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Board of Directors - Non Executive - ABC Capital Bank',
                'tagline' => 'Meet our non-executive board of directors providing independent oversight.',
                'description' => 'Meet the non-executive board of directors at ABC Capital Bank Uganda providing independent governance.',
                'keywords' => 'non-executive directors, board governance, independent directors, ABC Capital Bank',
                'page_type' => 'board-non-executive',
                'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => ['non-executive', 'governance', 'independent directors']
            ];
        }

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('management-team');

        // Load team members for non-executive board. We'll attempt to resolve by team category alias
        $teamModel = new TeamModel();
        $teamCategoryModel = new TeamCategoryModel();

        // Try to find a category matching common aliases for non-executive boards
        $categoryAliasCandidates = ['non-executive', 'non-executive-board', 'board-non-executive', 'nonexecutive'];
        $members = [];
        foreach ($categoryAliasCandidates as $alias) {
            $cat = $teamCategoryModel->where('alias', $alias)->first();
            if ($cat) {
                $members = $teamModel->getCategoryMembers($cat['id']);
                break;
            }
        }

        // Fallback: use all board members if category lookup failed
        if (empty($members)) {
            $members = $teamModel->showBoardMembers();
        }

        $templateData = [
            'team_title' => $data['title'] ?? 'Board of Directors',
            'team_title_emphasized' => 'Directors',
            'team_description' => $data['tagline'] ?? '',
            'team_members' => []
        ];

        foreach ($members as $m) {
            $templateData['team_members'][] = [
                'image' => $m->photo ?? ($m->image ?? null),
                'name' => $m->title ?? $m->name ?? '',
                'position' => $m->position ?? '',
                'modal_id' => 'teamModal-' . ($m->id ?? uniqid())
            ];
        }

        $data['templateData'] = $templateData;

    // Use the management-team route which will render the two lists (board vs management) using full staff list
    return $this->teamPageTemplate('management-team');
    }

    /**
     * Team page template resolver - handles URLs like /team/{alias}
     * Resolves a team category by alias and renders the shared team template.
     */
    public function teamPageTemplate($alias = null, $filter = null)
    {
        helper('ltcms');

        // Resolve category by alias
        $teamCategoryModel = new TeamCategoryModel();
        $teamModel = new TeamModel();

        $category = $teamCategoryModel->where('alias', $alias)->first();

        // Prepare basic data and compute a descriptive page title
        $baseName = $category['name'] ?? ucfirst(str_replace('-', ' ', $alias));
        $title = $baseName;
        if (!empty($filter) && in_array($filter, ['executive', 'non-executive'])) {
            // Make the page title reflect the subset being shown
            $title = $baseName . ' - ' . (ucfirst(str_replace('-', ' ', $filter)));
        }

        $data = [
            'title' => $title,
            'tagline' => $category['description'] ?? '',
            'page_type' => 'team',
        ];

        // Banner: try to load a hero banner keyed by the alias
        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl($alias);

        // Get members
        $members = [];

        // If the request is for the dedicated management route, load full staff so we can split
        if ($alias === 'management-team') {
            $members = $teamModel->showStaffMembers();
        } elseif ($alias === 'board-of-directors') {
            // Prefer an explicit board listing (if model supports it), otherwise fall back to staff
            $members = method_exists($teamModel, 'showBoardMembers') ? $teamModel->showBoardMembers() : $teamModel->showStaffMembers();
        } else {
            if ($category) {
                $members = $teamModel->getCategoryMembers($category['id']);
            } else {
                // If no category found, try to fetch by alias treating alias as numeric id or fallback to staff
                if (is_numeric($alias)) {
                    $members = $teamModel->getCategoryMembers((int)$alias);
                } else {
                    $members = $teamModel->showStaffMembers();
                }
            }
        }

        // Optional filtering: 'executive' or 'non-executive'
        if (!empty($filter) && !empty($members)) {
            $filtered = [];
            foreach ($members as $m) {
                $pos = trim((string)($m->position ?? ''));
                $isNonExec = stripos($pos, 'non-executive') !== false;
                $isExec = preg_match('/\b(executive|chief|ceo|chair|managing|md)\b/i', $pos) === 1;

                if ($filter === 'executive' && $isExec && !$isNonExec) {
                    $filtered[] = $m;
                } elseif ($filter === 'non-executive' && $isNonExec) {
                    $filtered[] = $m;
                }
            }

            // If filtering yielded results, use them. Otherwise fall back to unfiltered list.
            if (!empty($filtered)) {
                $members = $filtered;
            }
        }

        // Build templateData: use base name and emphasize subset when applicable
        // Build templateData with groupings for board pages so UI can show tabs
        $templateData = [
            'team_title' => $baseName,
            'team_title_emphasized' => !empty($filter) ? ucfirst(str_replace('-', ' ', $filter)) : 'Team',
            'team_description' => $data['tagline'] ?? '',
            'team_members_all' => [],
            'team_members_board' => [],
            'team_members_management' => [],
            'team_members' => [] // legacy single list
        ];

        // Helper to map member objects to arrays expected by the view
        $mapMember = function ($m) {
            return [
                'id' => $m->id ?? null,
                'image' => $m->photo ?? ($m->image ?? null),
                'name' => $m->title ?? $m->name ?? '',
                'position' => $m->position ?? '',
                'excerpt' => $m->excerpt ?? '',
                'content' => $m->content ?? '',
                'alias' => $m->alias ?? '',
                // common social/contact fields
                'facebook' => $m->facebook ?? null,
                'twitter' => $m->twitter ?? null,
                'linkedin' => $m->linkedin ?? null,
                'modal_id' => 'teamModal-' . ($m->id ?? uniqid())
            ];
        };

        // Build 'all' list first
        foreach ($members as $m) {
            $templateData['team_members_all'][] = $mapMember($m);
        }


        // Classify members into Board of Directors and Management (disjoint lists).
        // Board criteria: contains 'director', 'board', 'chair', 'trustee', 'governor' or explicit 'non-executive'.
        foreach ($members as $m) {
            $pos = trim((string)($m->position ?? ''));
            $posLower = strtolower($pos);
            $isBoard = false;

            // explicit non-executive markers
            if (stripos($posLower, 'non-executive') !== false || stripos($posLower, 'non executive') !== false) {
                $isBoard = true;
            }

            // board-related keywords (this will include executive directors as 'director')
            if (!$isBoard && preg_match('/\b(director|board|chair|trustee|governor)\b/i', $pos) === 1) {
                $isBoard = true;
            }

            if ($isBoard) {
                $templateData['team_members_board'][] = $mapMember($m);
            } else {
                $templateData['team_members_management'][] = $mapMember($m);
            }
        }

      

        $data['templateData'] = $templateData;

        return view('website/pages/team_template', $data);
    }













    // Saving Accounts Overview Page
    // Saving Accounts Overview Page - Simple version
public function savingAccountsOverview()
{
    helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('saving-accounts');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'Saving Accounts - ABC Capital Bank',
            'tagline' => $metadata['excerpt'] ?? 'Explore our range of saving accounts for your financial needs.',
            'description' => $metadata['metadesc'] ?? 'Discover ABC Capital Bank saving accounts in Uganda - premium savings, classic savings, foreign currency, and more.',
            'keywords' => $metadata['metakey'] ?? 'saving accounts, savings, deposit accounts, premium savings, classic savings, ABC Capital Bank',
            'page_type' => 'savings-overview',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'Saving Accounts - ABC Capital Bank',
            'tagline' => 'Explore our range of saving accounts for your financial needs.',
            'description' => 'Discover ABC Capital Bank saving accounts in Uganda - premium savings, classic savings, foreign currency, and more.',
            'keywords' => 'saving accounts, savings, deposit accounts, premium savings, classic savings, ABC Capital Bank',
            'page_type' => 'savings-overview',
            'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => ['savings', 'accounts', 'deposits']
        ];
    }

    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('saving-accounts');
    
    // Get all saving account products for the cards grid
    $data['saving_products'] = $this->postsModel
        ->where('service_group', 'saving')
        ->where('status', 'published')
        ->where('deleted_at', null)
        ->orderBy('menu_order', 'asc')
        ->findAll();

    return view('website/pages/saving_accounts_overview_template', $data);
}

  public function loanAccountsOverview()
{
    helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('loan-accounts');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'Loan Accounts - ABC Capital Bank',
            'tagline' => $metadata['excerpt'] ?? 'Explore our range of loan products for personal and business needs.',
            'description' => $metadata['metadesc'] ?? 'Discover ABC Capital Bank loan products in Uganda - personal loans, business loans, mortgages, and more.',
            'keywords' => $metadata['metakey'] ?? 'loans, personal loans, business loans, mortgages, loan accounts, ABC Capital Bank',
            'page_type' => 'loans-overview',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'Loan Accounts - ABC Capital Bank',
            'tagline' => 'Explore our range of loan products for personal and business needs.',
            'description' => 'Discover ABC Capital Bank loan products in Uganda - personal loans, business loans, mortgages, and more.',
            'keywords' => 'loans, personal loans, business loans, mortgages, loan accounts, ABC Capital Bank',
            'page_type' => 'loans-overview',
            'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => ['loans', 'financing', 'credit']
        ];
    }

    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('loan-accounts');
    
    // Get all loan products - both personal and business
    $all_loans = $this->postsModel
        ->whereIn('service_group', ['loans', 'loans-business']) // Get both personal and business loans
        ->where('status', 'published')
        ->where('deleted_at', null)
        ->orderBy('menu_order', 'asc')
        ->findAll();

    // Separate loans by context
    $data['personal_loans'] = array_values(array_filter($all_loans, function($loan) {
        return $loan['service_group'] === 'loans'; // Personal loans use 'loans' service group
    }));
    
    $data['business_loans'] = array_values(array_filter($all_loans, function($loan) {
        return $loan['service_group'] === 'loans-business'; // Business loans use 'loans-business' service group
    }));

    return view('website/pages/loan_accounts_overview_template', $data);
}

    

    // Contact Us Page
    // Contact Us Page
public function contactUs()
{
    helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('contact-us');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'Contact Us - ABC Capital Bank',
            'tagline' => $metadata['excerpt'] ?? 'Get in touch with ABC Capital Bank',
            'description' => $metadata['metadesc'] ?? 'Contact ABC Capital Bank Uganda for banking services, support, and inquiries. Find our branches, phone numbers, and email addresses.',
            'keywords' => $metadata['metakey'] ?? 'contact ABC Capital Bank, bank contact Uganda, customer service, branch locations, phone numbers, email',
            'page_type' => 'contact-us',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'Contact Us - ABC Capital Bank',
            'tagline' => 'Get in touch with ABC Capital Bank',
            'description' => 'Contact ABC Capital Bank Uganda for banking services, support, and inquiries. Find our branches, phone numbers, and email addresses.',
            'keywords' => 'contact ABC Capital Bank, bank contact Uganda, customer service, branch locations, phone numbers, email',
            'page_type' => 'contact-us',
            'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => ['contact', 'customer service', 'support']
        ];
    }

    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('contact-us');

    // Get dynamic subjects and account types for the contact form
    $feedbackSubjectModel = new FeedbackSubjectModel();
    $customerFeedbackModel = new CustomerFeedbackModel();

    $data['subjects'] = $feedbackSubjectModel->getAll();
    $data['accountTypes'] = $this->postsModel->getAllServices();

    // Get contact information blocks
    $blocks = $this->postBlocksModel->getPageBlocks('contact-us');
    $data['templateData'] = $this->convertContactBlocks($blocks, $data['page_metadata'] ?? []);

    return view('website/pages/contact_template', $data);
}

    public function submitContact()
    {
        helper(['form', 'url', 'security']);
        $response = service('response');
        $isAjax = $this->request->isAJAX();
        
        if ($this->request->getMethod() !== 'POST') {
            if ($isAjax) {
                return $response->setStatusCode(405)->setJSON([
                    'success' => false,
                    'message' => 'Method not allowed'
                ]);
            }
            return redirect()->back()->with('error', 'Method not allowed');
        }

        // Enhanced validation rules
        $rules = [
            'name' => [
                'label' => 'Full Name',
                'rules' => 'required|min_length[2]|max_length[100]|regex_match[/^[a-zA-Z\s]+$/]',
                'errors' => [
                    'regex_match' => 'The {field} may only contain letters and spaces'
                ]
            ],
            'email' => [
                'label' => 'Email Address',
                'rules' => 'required|valid_email|max_length[254]'
            ],
            'phone' => [
                'label' => 'Phone Number',
                'rules' => 'required|min_length[10]|max_length[20]|numeric',
                'errors' => [
                    'numeric' => 'The {field} must contain only numbers'
                ]
            ],
            'subject_id' => [
                'label' => 'Subject',
                'rules' => 'required|numeric|is_not_unique[feedback_subject.id]',
                'errors' => [
                    'is_not_unique' => 'Please select a valid subject'
                ]
            ],
            'message' => [
                'label' => 'Message',
                'rules' => 'required|min_length[10]|max_length[2000]|string',
                'errors' => [
                    'string' => 'The {field} must be a valid text'
                ]
            ],
            'account_type' => [
                'label' => 'Account Type',
                'rules' => 'permit_empty|numeric|is_not_unique[usc_posts.id]',
                'errors' => [
                    'is_not_unique' => 'Please select a valid account type'
                ]
            ],
            'honeypot' => [
                'label' => 'Honeypot',
                'rules' => 'max_length[0]',
                'errors' => [
                    'max_length' => 'Invalid form submission'
                ]
            ]
        ];

        if (!$this->validate($rules)) {
            if ($isAjax) {
                return $response->setStatusCode(422)->setJSON([
                    'success' => false,
                    'message' => 'Please check the form for errors',
                    'messages' => $this->validator->getErrors()
                ]);
            }
            return redirect()->back()
                ->with('error', 'Please check the form for errors')
                ->withInput()
                ->with('validation', $this->validator);
        }

        try {
            // Start transaction
            $this->db->transStart();

            // Sanitize and prepare form data
            $formData = [
                'name' => strip_tags($this->request->getPost('name')),
                'email' => filter_var($this->request->getPost('email'), FILTER_SANITIZE_EMAIL),
                'phone' => preg_replace('/[^0-9]/', '', $this->request->getPost('phone')),
                'subject_id' => (int)$this->request->getPost('subject_id'),
                'account_type' => $this->request->getPost('account_type') ? (int)$this->request->getPost('account_type') : null,
                'message' => strip_tags($this->request->getPost('message')),
            ];

            // Save to database
            $messageId = $this->customerfeedbackModel->insert($formData);

            if (!$messageId) {
                throw new \RuntimeException('Failed to save contact form data');
            }

            // Send confirmation email to user
            $emailSentToUser = $this->sendContactConfirmationEmail($formData, $messageId);
            
            // Send notification email to admins
            $emailSentToAdmin = $this->sendAdminNotificationEmail($formData, $messageId);

            // Log email sending status
            if (!$emailSentToUser || !$emailSentToAdmin) {
                log_message('error', 'Failed to send contact form emails. Message ID: ' . $messageId . 
                    ' User Email Status: ' . ($emailSentToUser ? 'Sent' : 'Failed') . 
                    ' Admin Email Status: ' . ($emailSentToAdmin ? 'Sent' : 'Failed'));
            }

            // Commit transaction
            $this->db->transComplete();

            if ($this->db->transStatus() === false) {
                throw new \RuntimeException('Database transaction failed');
            }

            $successMessage = 'Thank you for contacting us! We have received your message and will get back to you within 24-48 hours. Message ID: ' . $messageId;
            
            if ($isAjax) {
                return $response->setStatusCode(200)->setJSON([
                    'success' => true,
                    'messageId' => $messageId,
                    'message' => $successMessage,
                    'csrf_token' => csrf_hash() // Add new CSRF token
                ]);
            }
            
            return redirect()->back()->with('success', $successMessage);

        } catch (\Exception $e) {
            // Log the error with detailed information
            log_message('error', 'Contact form submission error: ' . $e->getMessage() . "\n" .
                'Stack trace: ' . $e->getTraceAsString() . "\n" .
                'Form data: ' . json_encode($formData ?? []));

            // Rollback transaction if active
            if ($this->db->transStatus() !== null) {
                $this->db->transRollback();
            }

            $errorMessage = 'Sorry, there was an error processing your request. Please try again or contact us directly.';
            
            if ($isAjax) {
                return $response->setStatusCode(500)->setJSON([
                    'success' => false,
                    'message' => $errorMessage
                ]);
            }
            return redirect()->back()
                ->with('error', $errorMessage)
                ->withInput();
        }
    }

    private function sendContactConfirmationEmail($formData, $messageId)
    {
        try {
            $email = \Config\Services::email();

            // Get subject title for email
            $subjectData = $this->feedbacksubjectModel->find($formData['subject_id']);
            $subjectTitle = $subjectData ? $subjectData['title'] : 'General Inquiry';

            $emailTemplate = view('email_templates/contact_submission_template', [
                'clientName' => esc($formData['name']),
                'messageId' => $messageId,
            ]);

            $email->setMailType('html');
            $email->setFrom(getenv('CONTACT_FORM_EMAIL') ?: 'ithelp@abccapitalbank.co.ug', 'ABC Capital Bank');
            $email->setTo($formData['email']);
            $email->setSubject('Contact Form Submission Confirmation - ABC Capital Bank');
            $email->setMessage($emailTemplate);

            $result = $email->send();

            if (!$result) {
                log_message('error', 'Failed to send contact confirmation email. Error: ' . $email->printDebugger(['headers']));
            }

            return $result;

        } catch (\Exception $e) {
            log_message('error', 'Error sending confirmation email: ' . $e->getMessage());
            return false;
        }
    }

    private function sendAdminNotificationEmail($formData, $messageId)
    {
        try {
            $email = \Config\Services::email();

            // Get subject title
            $subjectData = $this->feedbacksubjectModel->find($formData['subject_id']);
            $subjectTitle = $subjectData ? $subjectData['title'] : 'General Inquiry';

            // Get account type name if provided
            $accountTypeName = '';
            if (!empty($formData['account_type'])) {
                $accountType = $this->postsModel->find($formData['account_type']);
                $accountTypeName = $accountType ? $accountType['title'] : '';
            }

            $emailTemplate = view('email_templates/contact_admin_notification_template', [
                'clientName' => esc($formData['name']),
                'clientEmail' => esc($formData['email']),
                'phone' => esc($formData['phone']),
                'subject' => esc($subjectTitle),
                'accountType' => esc($accountTypeName),
                'message' => esc($formData['message']),
                'messageId' => $messageId,
            ]);

            $email->setMailType('html');
            $email->setFrom(getenv('CONTACT_FORM_EMAIL') ?: 'ithelp@abccapitalbank.co.ug', 'ABC Capital Bank System');
            
            // Get admin emails from environment variables or use defaults
            $adminEmails = explode(',', getenv('ADMIN_NOTIFICATION_EMAILS') ?: 'ithelp@abccapitalbank.co.ug');
            // Format admin emails for CC
            $ccEmails = array_map('trim', $adminEmails);
            $email->setCC($ccEmails);

            $email->setSubject('New Contact Form Message - Message ID: ' . $messageId);
            $email->setMessage($emailTemplate);

            $result = $email->send();

            if (!$result) {
                log_message('error', 'Failed to send admin notification email. Error: ' . $email->printDebugger(['headers']));
            }

            return $result;

        } catch (\Exception $e) {
            log_message('error', 'Error sending admin notification email: ' . $e->getMessage());
            return false;
        }
    }

private function convertContactBlocks($blocks, $post)
{
    $templateData = [
        'banner_config' => [
            'hero_title' => $post['title'] ?? 'Contact Us',
            'hero_subtitle' => $post['excerpt'] ?? 'Get in touch with ABC Capital Bank',
            'hero_bg' => $post['photo'] ?? '',
            'title_emphasized' => 'Us'
        ],
        'contact_methods' => [],
        'branches' => [],
        'contact_form' => [
            'title' => 'Send Us a Message',
            'subtitle' => 'Fill out the form below and we\'ll get back to you shortly'
        ]
    ];

    foreach ($blocks as $block) {
        switch ($block['block_type']) {
            case 'contact_methods':
                $templateData['contact_methods'] = [
                    'title' => $block['heading'] ?? 'Contact Methods',
                    'items' => []
                ];

                // Get contact method items
                $contactItems = $this->postBlocksModel->getBlockItems($block['id']);
                foreach ($contactItems as $item) {
                    if ($item['item_type'] === 'contact_method') {
                        $templateData['contact_methods']['items'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-star',
                            'details' => !empty($item['description']) ? explode("\n", $item['description']) : [],
                            'note' => $item['caption'] ?? '' 
                        ];
                    }
                }
                break;

            case 'branches':
                $templateData['branches'] = [
                    'title' => $block['heading'] ?? 'Our Branches',
                    'subtitle' => $block['subtitle'] ?? 'Visit us at any of our convenient locations',
                    'items' => []
                ];

                // Get branch items
                $branchItems = $this->postBlocksModel->getBlockItems($block['id']);
                foreach ($branchItems as $item) {
                    if ($item['item_type'] === 'branch') {
                        $templateData['branches']['items'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-building',
                            'details' => !empty($item['description']) ? explode("\n", $item['description']) : []
                        ];
                    }
                }
                break;

            case 'contact_form_info':
                $templateData['contact_form'] = [
                    'title' => $block['heading'] ?? 'Send Us a Message',
                    'subtitle' => $block['subtitle'] ?? 'Fill out the form below and we\'ll get back to you shortly'
                ];
                break;
        }
    }

    return $templateData;
}
// Loan Calculator Page
public function loanCalculator()
{
    helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('loan-calculator');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'Loan Calculator - ABC Capital Bank',
            'tagline' => $metadata['excerpt'] ?? 'Calculate Your Loan Repayments',
            'description' => $metadata['metadesc'] ?? 'Use ABC Capital Bank\'s loan calculator to estimate your monthly repayments, total interest, and overall cost for personal loans, business loans, and mortgages.',
            'keywords' => $metadata['metakey'] ?? 'loan calculator Uganda, loan repayment calculator, mortgage calculator, personal loan calculator, business loan calculator',
            'page_type' => 'loan-calculator',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'Loan Calculator - ABC Capital Bank',
            'tagline' => 'Calculate Your Loan Repayments',
            'description' => 'Use ABC Capital Bank\'s loan calculator to estimate your monthly repayments, total interest, and overall cost for personal loans, business loans, and mortgages.',
            'keywords' => 'loan calculator Uganda, loan repayment calculator, mortgage calculator, personal loan calculator, business loan calculator',
            'page_type' => 'loan-calculator',
            'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => ['loan calculator', 'financial calculator', 'repayment calculator']
        ];
    }

    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('loan-calculator');

    return view('website/pages/loan_calculator_template', $data);
}
    // Money Transfers Page
// Money Transfers Page
public function moneyTransfers()
{
    helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('money-transfers');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'Money Transfers - ABC Capital Bank',
            'tagline' => $metadata['excerpt'] ?? 'Secure, fast, and reliable ways to transfer money locally and internationally.',
            'description' => $metadata['metadesc'] ?? 'ABC Capital Bank money transfer services enable you to transfer money to your beneficiary\'s bank account anywhere across the world with best world\'s solutions.',
            'keywords' => $metadata['metakey'] ?? 'money transfers, international transfers, telegraphic transfers, western union, moneygram, RTGS, EFT, ABC Capital Bank',
            'page_type' => 'money-transfers',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'Money Transfers - ABC Capital Bank',
            'tagline' => 'Secure, fast, and reliable ways to transfer money locally and internationally.',
            'description' => 'ABC Capital Bank money transfer services enable you to transfer money to your beneficiary\'s bank account anywhere across the world with best world\'s solutions.',
            'keywords' => 'money transfers, international transfers, telegraphic transfers, western union, moneygram, RTGS, EFT, ABC Capital Bank',
            'page_type' => 'money-transfers',
            'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => ['money transfers', 'international', 'telegraphic transfers', 'western union']
        ];
    }

    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('money-transfers');

    return view('website/pages/money_transfers_template', $data);
}
// Open Account Page
// Add this method to your FrontPages controller
public function openAccount()
{
    helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('open-account');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'ABC CAPITAL BANK - Open an Account | Start Banking With Us | Begin Your Financial Journey With Us',
            'tagline' => $metadata['excerpt'] ?? 'Begin your financial journey with ABC Capital Bank',
            'description' => $metadata['metadesc'] ?? 'Open an account with ABC Capital Bank in Uganda. Learn about our account options, requirements, and how to apply online or visit our branches.',
            'keywords' => $metadata['metakey'] ?? 'open bank account Uganda, account requirements, banking services Uganda, ABC Capital Bank account',
            'page_type' => 'open-account',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'ABC CAPITAL BANK - Open an Account | Start Banking With Us | Begin Your Financial Journey With Us',
            'tagline' => 'Begin your financial journey with ABC Capital Bank',
            'description' => 'Open an account with ABC Capital Bank in Uganda. Learn about our account options, requirements, and how to apply online or visit our branches.',
            'keywords' => 'open bank account Uganda, account requirements, banking services Uganda, ABC Capital Bank account',
            'page_type' => 'open-account',
            'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
            'tags' => ['bank account', 'open account', 'banking services']
        ];
    }

    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('open-account');
    
    // Get blocks for open account page
    $blocks = $this->postBlocksModel->getPageBlocks('open-account');
    $data['templateData'] = $this->convertOpenAccountBlocks($blocks, $data['page_metadata'] ?? []);

    return view('website/pages/open_account_template', $data);
}

    // Add this private method to convert blocks for open account page
    private function convertOpenAccountBlocks($blocks, $post)
    {
        $templateData = [
            'banner_config' => [
                'hero_title' => $post['title'] ?? 'Open an Account',
                'hero_subtitle' => $post['excerpt'] ?? 'Begin your financial journey with ABC Capital Bank',
                'hero_bg' => $post['photo'] ?? '',
                'title_emphasized' => 'Account'
            ]
        ];

        foreach ($blocks as $block) {
            switch ($block['block_type']) {
                case 'account_requirements':
                    $templateData['requirements'] = [
                        'title' => $block['heading'] ?? 'Account Opening Requirements',
                        'items' => []
                    ];

                    // Get requirement items
                    $requirementItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($requirementItems as $item) {
                        $templateData['requirements']['items'][] = [
                            'account_type' => $item['item_type'] ?? 'general',
                            'title' => $item['title'] ?? '',
                            'description' => $item['description'] ?? ''
                        ];
                    }
                    break;

                case 'application_methods':
                    $templateData['application_methods'] = [
                        'title' => $block['heading'] ?? 'Ways to Apply',
                        'subtitle' => $block['subtitle'] ?? 'Choose the application method that works best for you',
                        'items' => []
                    ];

                    // Get application method items
                    $methodItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($methodItems as $item) {
                        $templateData['application_methods']['items'][] = [
                            'title' => $item['title'] ?? '',
                            'icon' => $item['icon'] ?? 'fas fa-star',
                            'description' => $item['description'] ?? ''
                        ];
                    }
                    break;

                case 'open_account_faq':
                    $templateData['faq'] = [
                        'title' => $block['heading'] ?? 'Account Opening FAQ',
                        'subtitle' => $block['subtitle'] ?? 'Common questions about opening an account with us',
                        'items' => []
                    ];

                    // Get FAQ items
                    $faqItems = $this->postBlocksModel->getBlockItems($block['id']);
                    foreach ($faqItems as $item) {
                        $templateData['faq']['items'][] = [
                            'question' => $item['title'] ?? '',
                            'answer' => $item['description'] ?? ''
                        ];
                    }
                    break;
            }
        }

        return $templateData;
    }



    // FAQs - Strategic Transitioning Page
public function faqsStrategicTransitioning()
{
    helper('ltcms');

    $metadata = $this->postsModel->getPageMetadataByAlias('faqs-strategic-transitioning');

    if ($metadata) {
        $data = [
            'title' => $metadata['title'] ?? 'FAQs - Strategic Transitioning - ABC Capital Bank',
            'tagline' => $metadata['excerpt'] ?? 'Strategic Transitioning FAQs',
            'description' => $metadata['metadesc'] ?? 'FAQs about ABC Capital Bank strategic transitioning from Tier 1 to Digital-led Tier 2 bank. Learn about the transition process, account safety, and digital banking benefits.',
            'keywords' => $metadata['metakey'] ?? 'FAQs strategic transitioning, ABC Capital Bank transition, Tier 1 to Tier 2 bank, digital banking Uganda, bank transition questions',
            'page_type' => 'faqs-strategic-transitioning',
            'page_metadata' => $metadata,
            'featured_image' => $metadata['photo'] ?? base_url('media/files/1711193726_b247131de3e4d5a072bc.jpg'),
            'tags' => $metadata['tags_array'] ?? []
        ];
    } else {
        $data = [
            'title' => 'FAQs - Strategic Transitioning - ABC Capital Bank',
            'tagline' => 'Strategic Transitioning FAQs',
            'description' => 'FAQs about ABC Capital Bank strategic transitioning from Tier 1 to Digital-led Tier 2 bank. Learn about the transition process, account safety, and digital banking benefits.',
            'keywords' => 'FAQs strategic transitioning, ABC Capital Bank transition, Tier 1 to Tier 2 bank, digital banking Uganda, bank transition questions',
            'page_type' => 'faqs-strategic-transitioning',
            'featured_image' => base_url('media/files/1711193726_b247131de3e4d5a072bc.jpg'),
            'tags' => ['faqs', 'strategic', 'transitioning', 'digital banking']
        ];
    }

    // Get the hero banner for FAQs page
    $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('faqs-strategic-transitioning');
    
    // If no specific banner found, use default
    if (!$data['hero_banner']) {
        $data['hero_banner'] = [
            'name' => 'FAQs - Strategic Transitioning',
            'title_emphasized' => 'Transitioning',
            'photo' => '1711193726_b247131de3e4d5a072bc.jpg',
            'description' => 'Explore commonly asked questions about strategic transitioning, including its definition, reasons, planning process, challenges, and key considerations.',
            'url' => 'faqs-strategic-transitioning'
        ];
    }

    // Get the page content
    $data['page_content'] = $metadata['content'] ?? '';

    // Get all subject answers for the FAQ section
    $subjectAnswersModel = new \Modules\CustomerCare\Models\SubjectAnswersModel();
    $data['faqs'] = $subjectAnswersModel->getAllWithSubjects();

    return view('website/pages/faqs_strategic_transitioning', $data);
}

    // Downloads Page
    public function downloadsPage()
    {
        helper('ltcms');

        $downloadsModel = new \Modules\Downloads\Models\DownloadsModel();
        $downloadsCategoryModel = new \Modules\Downloads\Models\DownloadsCategoryModel();

        $metadata = $this->postsModel->getPageMetadataByAlias('downloads');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Downloads - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Access our important documents and reports.',
                'description' => $metadata['metadesc'] ?? 'Download annual reports, financial statements, and other important documents from ABC Capital Bank Uganda.',
                'keywords' => $metadata['metakey'] ?? 'bank downloads, annual reports, financial statements, ABC Capital Bank documents',
                'page_type' => 'downloads',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Downloads - ABC Capital Bank',
                'tagline' => 'Access our important documents and reports.',
                'description' => 'Download annual reports, financial statements, and other important documents from ABC Capital Bank Uganda.',
                'keywords' => 'bank downloads, annual reports, financial statements, ABC Capital Bank documents',
                'page_type' => 'downloads',
                'featured_image' => base_url('assets/front_assets/images/logo-ABC-Capital-Bank-Uganda.gif'),
                'tags' => ['downloads', 'reports', 'documents']
            ];
        }

        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('downloads');

        if (!$data['hero_banner']) {
            $data['hero_banner'] = [
                'name' => 'Downloads',
                'title_emphasized' => 'Reports',
                'photo' => 'reports-page.jpg', // Assuming a default image for downloads
                'description' => 'Have a look at our Annual Reports and other important documents.',
                'url' => 'downloads'
            ];
        }

        // Get all categories
        $data['categories'] = $downloadsCategoryModel->getAll();

        // Get downloads grouped by category
        $downloadsByCategory = [];
        foreach ($data['categories'] as $category) {
            $categoryId = is_object($category) ? $category->id : $category['id'];
            $downloadsByCategory[$categoryId] = $downloadsModel->getByCategory($categoryId);
        }
        $data['downloads_by_category'] = $downloadsByCategory;

        // Also set all downloads for backward compatibility
        $data['downloads'] = $downloadsModel->getAll();

        return view('website/pages/downloads', $data);
    }



    // Annual Reports Page - Using DownloadsModel by category
    public function annualReports()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('annual-reports');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Annual Reports - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Financial Reports & Statements',
                'description' => $metadata['metadesc'] ?? 'Download ABC Capital Bank annual reports and financial statements. Access comprehensive financial performance data and corporate reports.',
                'keywords' => $metadata['metakey'] ?? 'annual reports Uganda, ABC Capital Bank financial statements, bank reports, financial performance',
                'page_type' => 'annual-reports',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/reports-banner.jpg'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Annual Reports - ABC Capital Bank',
                'tagline' => 'Financial Reports & Statements',
                'description' => 'Download ABC Capital Bank annual reports and financial statements. Access comprehensive financial performance data and corporate reports.',
                'keywords' => 'annual reports Uganda, ABC Capital Bank financial statements, bank reports, financial performance',
                'page_type' => 'annual-reports',
                'featured_image' => base_url('assets/front_assets/images/reports-banner.jpg'),
                'tags' => ['reports', 'annual', 'financial', 'statements']
            ];
        }

        // Get the hero banner for annual reports page
        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('annual-reports');
        
        // If no specific banner found, use default
        if (!$data['hero_banner']) {
            $data['hero_banner'] = [
                'name' => 'Annual Reports',
                'title_emphasized' => 'Reports',
                'photo' => 'reports-banner.jpg',
                'description' => 'Access our comprehensive annual reports and financial statements. Stay informed about our performance, growth, and financial health over the years.',
                'url' => 'annual-reports'
            ];
        }

        // Get the page content
        $data['page_content'] = $metadata['content'] ?? '';

        // Load annual reports using DownloadsModel by category (category_id = 2 for Annual Reports)
        $data['reports'] = $this->downloadsModel->getByCategory(2);

        return view('website/pages/annual_reports', $data);
    }

    // Bids and Tenders Page - Using DownloadsModel by category
    public function bidsTenders()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('bids-tenders');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Bids and Tenders - ABC Capital Bank',
                'tagline' => $metadata['excerpt'] ?? 'Procurement Opportunities',
                'description' => $metadata['metadesc'] ?? 'View ABC Capital Bank bids and tender opportunities. Access current procurement notices and bidding documents.',
                'keywords' => $metadata['metakey'] ?? 'bids Uganda, tenders ABC Capital Bank, procurement opportunities, bidding documents',
                'page_type' => 'bids-tenders',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/bids-tenders-banner.jpg'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Bids and Tenders - ABC Capital Bank',
                'tagline' => 'Procurement Opportunities',
                'description' => 'View ABC Capital Bank bids and tender opportunities. Access current procurement notices and bidding documents.',
                'keywords' => 'bids Uganda, tenders ABC Capital Bank, procurement opportunities, bidding documents',
                'page_type' => 'bids-tenders',
                'featured_image' => base_url('assets/front_assets/images/bids-tenders-banner.jpg'),
                'tags' => ['bids', 'tenders', 'procurement', 'opportunities']
            ];
        }

        // Get the hero banner for bids and tenders page
        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('bids-tenders');
        
        // If no specific banner found, use default
        if (!$data['hero_banner']) {
            $data['hero_banner'] = [
                'name' => 'Bids and Tenders',
                'title_emphasized' => 'Tenders',
                'photo' => 'bids-tenders-banner.jpg',
                'description' => 'Explore current bidding opportunities and tender notices. Partner with ABC Capital Bank in delivering quality services and projects through our procurement processes.',
                'url' => 'bids-tenders'
            ];
        }

        // Get the page content
        $data['page_content'] = $metadata['content'] ?? '';

        // Load bids and tenders using DownloadsModel by category (category_id = 4 for Bids & Tenders)
        $data['tenders'] = $this->downloadsModel->getByCategory(4);

        return view('website/pages/bids_tenders', $data);
    }
// Add this method to your Pages controller
// Updated Careers Page Method
    public function careersPage()
    {
        helper('ltcms');

        $metadata = $this->postsModel->getPageMetadataByAlias('careers');

        if ($metadata) {
            $data = [
                'title' => $metadata['title'] ?? 'Careers - ABC Capital Bank Uganda',
                'tagline' => $metadata['excerpt'] ?? 'Join Our Team',
                'description' => $metadata['metadesc'] ?? 'Explore banking career opportunities at ABC Capital Bank Uganda. Join our team of professionals in Kampala and across Uganda.',
                'keywords' => $metadata['metakey'] ?? 'banking careers Uganda, ABC Capital Bank jobs, finance careers Kampala, bank jobs Uganda, employment opportunities',
                'page_type' => 'careers',
                'page_metadata' => $metadata,
                'featured_image' => $metadata['photo'] ?? base_url('assets/front_assets/images/careers-banner.jpg'),
                'tags' => $metadata['tags_array'] ?? []
            ];
        } else {
            $data = [
                'title' => 'Careers - ABC Capital Bank Uganda',
                'tagline' => 'Join Our Team',
                'description' => 'Explore banking career opportunities at ABC Capital Bank Uganda. Join our team of professionals in Kampala and across Uganda.',
                'keywords' => 'banking careers Uganda, ABC Capital Bank jobs, finance careers Kampala, bank jobs Uganda, employment opportunities',
                'page_type' => 'careers',
                'featured_image' => base_url('assets/front_assets/images/careers-banner.jpg'),
                'tags' => ['careers', 'jobs', 'employment', 'banking', 'Uganda']
            ];
        }

        // Get the hero banner for careers page
        $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('careers');
        
        // If no specific banner found, use default
        if (!$data['hero_banner']) {
            $data['hero_banner'] = $this->bannersModel->getPageHeroBannerByUrl('jobs');
        }

        // If still no banner, create default
        if (!$data['hero_banner']) {
            $data['hero_banner'] = [
                'name' => 'Careers at ABC Capital Bank Uganda',
                'title_emphasized' => 'Careers',
                'photo' => 'careers-banner.jpg',
                'description' => 'Join our dynamic team and build a rewarding career in Uganda\'s banking industry. Explore exciting opportunities for growth and development at ABC Capital Bank.',
                'url' => 'careers'
            ];
        }

        // Get active jobs from database
        $data['jobs'] = $this->jobsModel->forHomePage();
        
        return view('website/pages/careers', $data);
    }

    
}
