<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\BlockDefinitionsModel;
use App\Models\BlockItemTypeRelationshipsModel;
use App\Models\ItemTypeDefinitionsModel;

class BlocksController extends BaseController
{
    protected $blockDefinitions;
    protected $blockItemTypeRelationships;
    protected $itemTypeDefinitions;

    public function __construct()
    {
        $this->blockDefinitions = new BlockDefinitionsModel();
        $this->blockItemTypeRelationships = new BlockItemTypeRelationshipsModel();
        $this->itemTypeDefinitions = new ItemTypeDefinitionsModel();
    }

    /**
     * Get available item types for a specific block
     * 
     * @param int $blockId
     * @return \CodeIgniter\HTTP\Response
     */
    public function getAvailableItemTypes($blockId)
    {
        try {
            // Get the block's definition
            $block = $this->blockDefinitions->find($blockId);
            if (!$block) {
                return $this->response->setStatusCode(404)->setJSON([
                    'success' => false,
                    'message' => 'Block not found'
                ]);
            }

            // Get item types based on block definition
            $itemTypes = $this->itemTypeDefinitions->getByBlockDefinition($block['definition_id']);

            // If no specific item types are defined for this block definition,
            // return all active item types
            if (empty($itemTypes)) {
                $itemTypes = $this->itemTypeDefinitions->where('is_active', 1)->findAll();
            }

            // Format response
            $formattedTypes = array_map(function($type) {
                return [
                    'id' => $type['id'],
                    'name' => $type['name'],
                    'slug' => $type['slug'],
                    'description' => $type['description'],
                    'icon' => $type['icon'],
                    'fields_config' => json_decode($type['fields_config'] ?? '{}'),
                    'settings_config' => json_decode($type['settings_config'] ?? '{}')
                ];
            }, $itemTypes);

            return $this->response->setJSON([
                'success' => true,
                'data' => $formattedTypes
            ]);

        } catch (\Exception $e) {
            log_message('error', '[BlockItemTypes] Error getting item types: ' . $e->getMessage());
            return $this->response->setStatusCode(500)->setJSON([
                'success' => false,
                'message' => 'Internal server error'
            ]);
        }
    }
}